<?php
/**
* SystemAdminServiceclass.php システム管理者用DBアクセスサービス
* PHP versions：4.4.2
* @パッケージ：-
* @作者：K.Shimomura
* @作成日/作成者：2006/09/01 (IIS)K.Shimomura
* @更新日/更新者：
* @バージョン：1.0.0
*
* Copyright (c) 2006 Iwate Information System Co.,Ltd. All Rights Reserved.       
* You may see the License at LICENSE_EN.txt(English) or LICENSE_JA.txt(Japanese). 
*/
require_once("class/db/dbaccess/AdminUserJoinMapTableAccess.class.php");
require_once("class/db/dbaccess/AdminUserTableAccess.class.php");
require_once("class/db/dbaccess/BaseMapTableAccess.class.php");
require_once("class/db/dbaccess/CreateTableAccess.class.php");
require_once("class/db/dbaccess/DropTableAccess.class.php");
require_once("class/db/dbaccess/MapTableAccess.class.php");
require_once("class/db/dbaccess/UserTableAccess.class.php");
require_once("class/db/service/KakikoService.class.php");

require_once("function/KakikoFile.php");

class SystemAdminService extends KakikoService {

    /**
     * コンストラクタ
     * @引数： (I) directryname  - string ディレクトリ名称
     */
    function SystemAdminService($directryname = "") {

		// 基底クラスのコンストラクタ呼び出し
    	$this->KakikoService($directryname);

		// DB接続情報の設定
		$this->setDBConnectInfo();
    }

    /**
     * 関数概要：管理者ユーザテーブルキー指定重複チェック
     * 処理詳細：管理者ユーザテーブルに対してキー指定で重複データチェックを
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) key       - string 管理者ユーザテーブルキー(ユーザID)
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkAdminUserDataByKey($key) {

		// DBオブジェクトの生成
		$this->createDBObject();

		$table_access = new AdminUserTableAccess();

		// 管理ユーザデータのキー指定チェック
		$rtn = $table_access->checkFindByKeyData($this->o_db, $key);

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $rtn;
	}

    /**
     * 関数概要：マップテーブルディレクトリ名称指定重複チェック
     * 処理詳細：マップテーブルに対してキー指定で重複データチェックを
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： (I) directry  - string ディレクトリ名称
     * @戻値： boolean true :データなし
     *      ：         false:データあり
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function checkMapDataByDirectryName($directry) {

		// DBオブジェクトの生成
		$this->createDBObject();

		$table_access = new MapTableAccess();

		// マップデータのキー指定チェック
		$rtn = $table_access->checkFindByDirectryNameData($this->o_db, $directry);

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $rtn;
	}

    /**
     * 関数概要：マップ管理者情報一覧データ取得
     * 処理詳細：管理者ユーザテーブルとマップテーブルをリンクしたデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： 
     * @戻値： array   管理者ユーザリンクマップデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getMapAdminInfoDataList() {

		// DBオブジェクトの生成
		$this->createDBObject();

		// 一覧情報の取得
		// SQL:	SELECT adminusertable.userid, 
		//				adminusertable.username, 
		//				maptable.contentsname,
		//				maptable.directryname,
		//				adminusertable.mailaddress,
		//				adminusertable.adddatetime,
		//				adminusertable.statusflg
		//		FROM   adminusertable
		//			LEFT JOIN maptable
		//				ON adminusertable.userid = maptable.userid
		//		WHERE  adminusertable.usertype = '1'
		//		ORDER BY adminusertable.userid

		$table_access = new AdminUserJoinMapTableAccess();

		// 管理ユーザデータのユーザ種別がマップ管理者であるデータ全件取得
		$rtn = $table_access->selectMapAdminUserListData($this->o_db, KAKIKOMAP_USERTYPE_MAP);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_001_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;
	}

    /**
     * 関数概要：コンテンツ状態情報一覧データ取得
     * 処理詳細：管理者ユーザテーブルとマップテーブルをリンクしたデータの取得を
     *         ：行うクエリ作成を呼び出す
     *
     * @引数： 
     * @戻値： array   管理者ユーザリンクマップデータクラス
     *      ：         DBエラーの場合はnull
     * @作成日/作成者：2006/09/01 (IIS)K.Shimomura
     * @更新日/更新者：
     * @更新内容：
     */
	function getContentsStatusInfoDataList() {

		// DBオブジェクトの生成
		$this->createDBObject();

		// 一覧情報の取得
		// SQL:	SELECT adminusertable.userid,
		//				adminusertable.username,
		//				maptable.publiccontents,
		//				maptable.contentsname,
		//				maptable.directryname,
		//				adminusertable.adddatetime
		//		FROM   adminusertable
		//			LEFT JOIN maptable
		//				ON adminusertable.userid = maptable.userid
		//		WHERE  adminusertable.usertype = '1'
		//		ORDER BY adminusertable.userid

		$table_access = new AdminUserJoinMapTableAccess();

		// 管理ユーザデータのユーザ種別がマップ管理者であるデータ全件取得
		$rtn = $table_access->selectContentsListData($this->o_db, KAKIKOMAP_USERTYPE_MAP);
		if ($rtn != true) {
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_002_01"), E_USER_ERROR);
			return null;
		}

		// データオブジェクトの取得
		$array_data = $table_access->getDataList();

		// 解放
		$table_access->free();

		// DB接続の切断
		$this->disconnectDB();

		return $array_data;
	}

	/**
	 *	関数名：マップ管理者登録
     * @引数：(I) $mapDataobj       - マップテーブル用データオブジェクト
     *        (I) $basemapDataobj   - ベースマップテーブル用データオブジェクト
     *        (I) $userDataobj      - ユーザテーブル用データオブジェクト
     * @戻値：$bool			- true  正常登録
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：マップ管理者登録を行います。
	 *            
     */
	function insertMapAdminUser($mapDataobj, $basemapDataobj, $userDataobj) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		$maptable_access = new MapTableAccess();

		// データ追加処理
		$rtn = $maptable_access->insertData($this->o_db, $mapDataobj);

		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_003_01"), E_USER_ERROR);

			return false;
		}

		$basemaptable_access = new BaseMapTableAccess();
		
		// データ追加処理
		$rtn = $basemaptable_access->insertData($this->o_db, $basemapDataobj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$basemaptable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_003_02"), E_USER_ERROR);

			return false;
		}

	    // マップ配下のテーブル作成

	    // ファイル名定義
		$filename = "../../../../contents_temp/KakikoDB.sql";
	
		$createtable_access = new CreateTableAccess($filename);
	
		$rtn = $createtable_access->CreateTableContents($this->o_db, $this->directryname);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$basemaptable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_003_03"), E_USER_ERROR);

			return false;
		}

		$usertable_access = new UserTableAccess($this->directryname);
		
		// マップ配下ユーザデータ追加処理
		$rtn = $usertable_access->insertData($this->o_db, $userDataobj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$maptable_access->free();
			$basemaptable_access->free();
			$usertable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_003_04"), E_USER_ERROR);

			return false;
		}

		// ファイルコピー処理
		if (!$this->createNewMap()) {
			$o_db->rollback();
			$maptable_access->free();
			$basemaptable_access->free();
			$usertable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_003_04"), E_USER_ERROR);

			return false;
		}

		// 解放
		$maptable_access->free();
		$basemaptable_access->free();
		$usertable_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return true;

	}

	/**
	 *	関数名：ユーザ&マップテーブルデータキー更新
     * @引数：(I) $targetUserDataobj    - 対象列(条件列/更新列)指定用ユーザデータオブジェクト
     *        (I) $valUserDataobj       - 設定値(条件値/更新値)指定用ユーザデータオブジェクト
     *        (I) $targetMapDataobj     - 対象列(条件列/更新列)指定用マップデータオブジェクト
     *        (I) $valMapDataobj        - 設定値(条件値/更新値)指定用マップデータオブジェクト
     * @戻値：$bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @作成日/作成者：
     * @使用方法：ユーザテーブルデータ及びマップテーブルデータをキー(ユーザID)で更新を行います。
	 *            
     */
	function updateUserAndMapDataByKey($targetUserDataobj, $valUserDataobj, $targetMapDataobj, $valMapDataobj) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// ユーザDBアクセスインスタンス生成
		$table_access = new UserTableAccess($this->directryname);

		// ユーザデータ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetUserDataobj, $valUserDataobj);

		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_004_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$table_access->free();

		// マップDBアクセスインスタンス生成
		$table_access = new MapTableAccess();

		// ユーザデータ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetMapDataobj, $valMapDataobj);

		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_004_02"), E_USER_ERROR);
			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();
		
		return true;

	}

	/**
	 *	関数名：マップテーブルデータキー更新
     * @引数：(I) $targetDataobj    - 対象列(条件列/更新列)指定用データオブジェクト
     *        (I) $valDataobj       - 設定値(条件値/更新値)指定用データオブジェクト
     * @戻値：$bool			- true  正常更新
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：マップテーブルデータをキー(ユーザID)で更新を行います。
	 *            
     */
	function updateMapDataByKey($targetDataobj, $valDataobj) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// マップDBアクセスインスタンス生成
		$table_access = new MapTableAccess();

		// マップデータ更新
		$rtn = $table_access->updateByKeyData($this->o_db, $targetDataobj, $valDataobj);
		if ($rtn != true) {
			$this->o_db->rollback();
			$table_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_005_01"), E_USER_ERROR);
			return false;
		}

		// 解放
		$table_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return true;

	}

	/**
	 *	関数名：マップ管理者削除
     * @引数：(I) $userid   - ユーザID
     * @戻値：$bool			- true  正常削除
     *                      - false 異常
     * @作成日/作成者：2006/09/01 (IIS) K.Shimomura
     * @更新日/更新者：
     * @使用方法：マップ管理者削除を行います。
	 *            
     */
	function deleteMapAdminUser($userid) {

		// DBオブジェクトの生成
		$this->createDBObject();

		// トランザクション開始
		$this->o_db->autoCommit(false);

		// 管理者ユーザテーブルデータ削除処理
		$adminusertable_access = new AdminUserTableAccess();

		$rtn = $adminusertable_access->deleteByKeyData($this->o_db, $userid);
		if ($rtn != true) {
			$this->o_db->rollback();
			$adminusertable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_006_01"), E_USER_ERROR);

			return false;
		}

		// マップ配下のテーブル削除
		// ファイル名定義
		$filename = "../../../../contents_temp/DeleteKakikoDB.sql";
		
		$droptable_access = new DropTableAccess($filename);
		$rtn = $droptable_access->DropTableContents($this->o_db, $this->directryname);
		if ($rtn != true) {
			$this->o_db->rollback();
			$adminusertable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_006_02"), E_USER_ERROR);

			return false;
		}

		// ファイル削除処理
		if (!$this->deleteMap()) {
			$this->o_db->rollback();
			$adminusertable_access->free();
			$this->disconnectDB();
			trigger_error($this->o_resource->getValue("S1_006_03"), E_USER_ERROR);

			return false;
		}

		// 解放
		$adminusertable_access->free();

		// トランザクションコミット
		$this->o_db->commit();

		// DB接続の切断
		$this->disconnectDB();

		return true;

	}


//#################################################################################################
//                                        以下、サービス内使用関数定義
//#################################################################################################

	/**
	* 関数概要：新規マップ追加時のファイル操作処理
	* 処理詳細：ベースよりファイル郡をコピーし新規マップを作成する
	*
	* @引数： 
	* @戻値： boolean TRUE :正常
	*      ：         FALSE:異常
	* @作成日/作成者：2006/09/01 (IIS)H.Keta
	* @更新日/更新者：
	* @更新内容：
	*/
	function createNewMap() {
	
	    // ファイルパス定義
	    $copy_base       = "../../../../contents_temp/front";
	    $to_src          = "../../contents";
	    $to_tmp          = "../../../../smarty/kakikomap/templates";
	    $to_conf_ini     = "../../../../conf/kakikomap/ini/auto";
	    $to_conf_require = "../../../../conf/kakikomap/require/common";
	    $to_conf_db      = "../../../../conf/kakikomap/require/db";
	    $to_conf_res     = "../../../../conf/kakikomap/resource/contents";
	    $to_conf_rss     = "../../../../conf/kakikomap/rss";
	    
	    // 置き換え用定義
	    $search = "@@DIRECTORY_NAME@@";
		$dbhost_str = "@@DB_HOST@@";
		$dbname_str = "@@DB_NAME@@";
			
	    // ディレクトリ名
	    $dir = trim($this->directryname);
	
	    // ----------------------------------
	    // マップ別ソースファイルコピー
	    // ----------------------------------
	    if (!kakikoCopy($copy_base."/src", $to_src.DIRECTORY_SEPARATOR.$dir)) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_src.DIRECTORY_SEPARATOR.$dir.DIRECTORY_SEPARATOR."DefineDirectryName.php", $search, $dir)) {
	        $this->rollbackCopy();
	        return false;
	    }
	    // ----------------------------------
	    // マップ別テンプレートファイルコピー
	    // ----------------------------------
	    if (!kakikoCopy($copy_base."/tmp", $to_tmp.DIRECTORY_SEPARATOR.$dir)) {
	        $this->rollbackCopy();
	        return false;
	    }
	    // ----------------------------------
	    // マップ別iniファイルコピー
	    // ----------------------------------
	    //setting.ini
	    if (!kakikoCopy($copy_base."/conf/ini/xxxxxxxx_setting.ini", $to_conf_ini.DIRECTORY_SEPARATOR.$dir."_setting.ini")) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_setting.ini", $search, $dir)) {
	        $this->rollbackCopy();
	        return false;
	    }

	    //mapadmininfo.ini
	    if (!kakikoCopy($copy_base."/conf/ini/xxxxxxxx_mapadmininfo.ini", $to_conf_ini.DIRECTORY_SEPARATOR.$dir."_mapadmininfo.ini")) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_mapadmininfo.ini", $search, $dir)) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_mapadmininfo.ini", $dbhost_str, $this->dbhost)) {	//DBHOST設定
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_mapadmininfo.ini", $dbname_str, $this->dbname)) {	//DBNAME設定
	        $this->rollbackCopy();
	        return false;
	    }

	    //userinfo.ini
	    if (!kakikoCopy($copy_base."/conf/ini/xxxxxxxx_userinfo.ini", $to_conf_ini.DIRECTORY_SEPARATOR.$dir."_userinfo.ini")) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_userinfo.ini", $search, $dir)) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_userinfo.ini", $dbhost_str, $this->dbhost)) {	//DBHOST設定
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_userinfo.ini", $dbname_str, $this->dbname)) {	//DBNAME設定
	        $this->rollbackCopy();
	        return false;
	    }
	    
	    // ----------------------------------
	    // マップ別requireファイルコピー
	    // ----------------------------------
	    if (!kakikoCopy($copy_base."/conf/require/xxxxxxxx_settingRead.php", $to_conf_require.DIRECTORY_SEPARATOR.$dir."_settingRead.php")) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_require.DIRECTORY_SEPARATOR.$dir."_settingRead.php", $search, $dir)) {
	        $this->rollbackCopy();
	        return false;
	    }
		
		
	    // ----------------------------------
	    // マップ別dbファイルコピー
	    // ----------------------------------	    
		//mapadmininfoRead.php
	    if (!kakikoCopy($copy_base."/conf/require/xxxxxxxx_mapadmininfoRead.php", $to_conf_db.DIRECTORY_SEPARATOR.$dir."_mapadmininfoRead.php")) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_db.DIRECTORY_SEPARATOR.$dir."_mapadmininfoRead.php", $search, $dir)) {
	        $this->rollbackCopy();
	        return false;
	    }
		
		//userinfoRead.php
	    if (!kakikoCopy($copy_base."/conf/require/xxxxxxxx_userinfoRead.php", $to_conf_db.DIRECTORY_SEPARATOR.$dir."_userinfoRead.php")) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_db.DIRECTORY_SEPARATOR.$dir."_userinfoRead.php", $search, $dir)) {
	        $this->rollbackCopy();
	        return false;
	    }

	    // ----------------------------------
	    // マップ別resourceファイルコピー
	    // ----------------------------------
	    if (!kakikoCopy($copy_base."/conf/res/xxxxxxxx.xml", $to_conf_res.DIRECTORY_SEPARATOR.$dir.".xml")) {
	        $this->rollbackCopy();
	        return false;
	    }
	    if (!$this->fileReplace($to_conf_res.DIRECTORY_SEPARATOR.$dir.".xml", $search, $dir)) {
	        $this->rollbackCopy();
	        return false;
	    }
	    // ----------------------------------
	    // マップ別rssファイルコピー
	    // ----------------------------------
        // コピー先のフォルダが存在しない場合は作成
        if (!file_exists($to_conf_rss.DIRECTORY_SEPARATOR.$dir)) {
            $return = mkdir($to_conf_rss.DIRECTORY_SEPARATOR.$dir);
        }

	    return true;
	
	}
	
	/**
	* 関数概要：ファイル内の文字列を指定文字で置換える
	* 処理詳細：ファイル内の文字列を指定文字で置換える
	*
	* @引数： (I) fpath        - string ファイルパス
	*      ： (I) search　     - string 置換え前文字
	*      ： (I) replace      - string 置換え後文字
	* @戻値： boolean TRUE :正常
	*      ：         FALSE:異常
	* @作成日/作成者：2006/09/01 (IIS)H.Keta
	* @更新日/更新者：
	* @更新内容：
	*/
	function fileReplace($fpath, $search, $replace) {
	
	    $return = true;
	
	    // ------------------------------------
	    // ファイルの内容をバッファへ読み出し
	    // ------------------------------------
	    // ファイルオープン
	    if (!$handle = fopen($fpath, "rb")) {
	        return false;
	    }
	    // ファイルの全内容を読み込み
	    $filebuf = fread($handle, filesize($fpath));
	    // ファイルクローズ
	    fclose($handle);
	
	    // ------------------------------------
	    // バッファ内の文字列を置き換え
	    // ------------------------------------
	    $filebuf = str_replace($search, $replace, $filebuf);
	
	    // ------------------------------------
	    // バッファの内容をファイルへ書き込み
	    // ------------------------------------
	    // ファイルオープン
	    if (!$handle = fopen($fpath, "w")) {
	        return false;
	    }
	    // バッファの内容を書き込み
	    if (fwrite($handle, $filebuf) == false) {
	        $return = false;
	    }
	    // ファイルクローズ
	    fclose($handle);
	    
	    return $return;
	
	}
	
	/**
	* 関数概要：ファイルコピーキャンセル処理
	* 処理詳細：ファイルコピーした内容をキャンセルする
	*
	* @引数： なし
	* @戻値： なし
	* @作成日/作成者：2006/09/01 (IIS)H.Keta
	* @更新日/更新者：
	* @更新内容：
	*/
	function rollbackCopy() {
	
	    // ファイルパス定義
	    $copy_base       = "../../../../contents_temp/front";
	    $to_src          = "../../contents";
	    $to_tmp          = "../../../../smarty/kakikomap/templates";
	    $to_conf_ini     = "../../../../conf/kakikomap/ini/auto";
	    $to_conf_require = "../../../../conf/kakikomap/require/common";
	    $to_conf_db      = "../../../../conf/kakikomap/require/db";
	    $to_conf_res     = "../../../../conf/kakikomap/resource/contents";
	    
	    // ディレクトリ名
	    $dir = trim($this->directryname);
	
	    if (file_exists($to_src.DIRECTORY_SEPARATOR.$dir)) {
	        kakikoDelete($to_src.DIRECTORY_SEPARATOR.$dir);
	    }
	    if (file_exists($to_tmp.DIRECTORY_SEPARATOR.$dir)) {
	        kakikoDelete($to_tmp.DIRECTORY_SEPARATOR.$dir);
	    }
	    if (file_exists($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_setting.ini")) {
	        kakikoDelete($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_setting.ini");
	    }
	    if (file_exists($to_conf_require.DIRECTORY_SEPARATOR.$dir."_settingRead.php")) {
	        kakikoDelete($to_conf_require.DIRECTORY_SEPARATOR.$dir."_settingRead.php");
	    }
	    if (file_exists($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_mapadmininfo.ini")) {
	        kakikoDelete($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_mapadmininfo.ini");
	    }
	    if (file_exists($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_userinfo.ini")) {
	        kakikoDelete($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_userinfo.ini");
	    }
	    
/*	    if (file_exists($to_conf_db.DIRECTORY_SEPARATOR.$dir."_kakikomapadmininfo.php")) {
	        kakikoDelete($to_conf_db.DIRECTORY_SEPARATOR.$dir."_kakikomapadmininfo.php");
	    }
	    if (file_exists($to_conf_db.DIRECTORY_SEPARATOR.$dir."_kakikouserinfo.php")) {
	        kakikoDelete($to_conf_db.DIRECTORY_SEPARATOR.$dir."_kakikouserinfo.php");
	    }
*/
	    if (file_exists($to_conf_db.DIRECTORY_SEPARATOR.$dir."_mapadmininfoRead.php")) {
	        kakikoDelete($to_conf_db.DIRECTORY_SEPARATOR.$dir."_mapadmininfoRead.php");
	    }
	    if (file_exists($to_conf_db.DIRECTORY_SEPARATOR.$dir."_userinfoRead.php")) {
	        kakikoDelete($to_conf_db.DIRECTORY_SEPARATOR.$dir."_userinfoRead.php");
	    }
	    
	    if (file_exists($to_conf_res.DIRECTORY_SEPARATOR.$dir.".xml")) {
	        kakikoDelete($to_conf_res.DIRECTORY_SEPARATOR.$dir.".xml");
	    }
	
	}

	/**
	* 関数概要：マップ削除時のファイル操作処理
	* 処理詳細：対象のマップ関連ファイル郡を削除する
	*
	* @引数： 
	* @戻値： boolean TRUE :正常
	*      ：         FALSE:異常
	* @作成日/作成者：2006/09/01 (IIS)H.Keta
	* @更新日/更新者：
	* @更新内容：
	*/
	function deleteMap() {
	
	    // ファイルパス定義
	    $to_src          = "../../contents";
	    $to_tmp          = "../../../../smarty/kakikomap/templates";
	    $to_conf_ini     = "../../../../conf/kakikomap/ini/auto";
	    $to_conf_require = "../../../../conf/kakikomap/require/common";
	    $to_conf_db      = "../../../../conf/kakikomap/require/db";
	    $to_conf_res     = "../../../../conf/kakikomap/resource/contents";
	    $to_conf_rss     = "../../../../conf/kakikomap/rss";
	    
	    // ディレクトリ名
	    $dir = trim($this->directryname);
	
	    // ----------------------------------
	    // マップ別ソースファイル削除
	    // ----------------------------------
	    if (file_exists($to_src.DIRECTORY_SEPARATOR.$dir)) {
	        kakikoDelete($to_src.DIRECTORY_SEPARATOR.$dir);
	    }
	    // ----------------------------------
	    // マップ別テンプレートファイル削除
	    // ----------------------------------
	    if (file_exists($to_tmp.DIRECTORY_SEPARATOR.$dir)) {
	        kakikoDelete($to_tmp.DIRECTORY_SEPARATOR.$dir);
	    }
	    // ----------------------------------
	    // マップ別iniファイル削除
	    // ----------------------------------
	    if (file_exists($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_setting.ini")) {
	        kakikoDelete($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_setting.ini");
	    }
	    if (file_exists($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_mapadmininfo.ini")) {
	        kakikoDelete($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_mapadmininfo.ini");
	    }
	    if (file_exists($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_userinfo.ini")) {
	        kakikoDelete($to_conf_ini.DIRECTORY_SEPARATOR.$dir."_userinfo.ini");
	    }
	    // ----------------------------------
	    // マップ別requireファイル削除
	    // ----------------------------------
	    if (file_exists($to_conf_require.DIRECTORY_SEPARATOR.$dir."_settingRead.php")) {
	        kakikoDelete($to_conf_require.DIRECTORY_SEPARATOR.$dir."_settingRead.php");
	    }
	    // ----------------------------------
	    // マップ別dbファイル削除
	    // ----------------------------------
/*	    if (file_exists($to_conf_db.DIRECTORY_SEPARATOR.$dir."_kakikomapadmininfo.php")) {
	        kakikoDelete($to_conf_db.DIRECTORY_SEPARATOR.$dir."_kakikomapadmininfo.php");
	    }
	    if (file_exists($to_conf_db.DIRECTORY_SEPARATOR.$dir."_kakikouserinfo.php")) {
	        kakikoDelete($to_conf_db.DIRECTORY_SEPARATOR.$dir."_kakikouserinfo.php");
	    }
*/
	    if (file_exists($to_conf_db.DIRECTORY_SEPARATOR.$dir."_mapadmininfoRead.php")) {
	        kakikoDelete($to_conf_db.DIRECTORY_SEPARATOR.$dir."_mapadmininfoRead.php");
	    }
	    if (file_exists($to_conf_db.DIRECTORY_SEPARATOR.$dir."_userinfoRead.php")) {
	        kakikoDelete($to_conf_db.DIRECTORY_SEPARATOR.$dir."_userinfoRead.php");
	    }
   	    // ----------------------------------
	    // マップ別resourceファイル削除
	    // ----------------------------------
	    if (file_exists($to_conf_res.DIRECTORY_SEPARATOR.$dir.".xml")) {
	        kakikoDelete($to_conf_res.DIRECTORY_SEPARATOR.$dir.".xml");
	    }
	    // ----------------------------------
	    // マップ別rssファイル削除
	    // ----------------------------------
	    if (file_exists($to_conf_rss.DIRECTORY_SEPARATOR.$dir)) {
	        kakikoDelete($to_conf_rss.DIRECTORY_SEPARATOR.$dir);
	    }

	    return true;
	
	}

}


?>
