/*
Kasumi: Dictionary management tool for Anthy
Copyright (C) 2004 Takashi Nakamoto

This file is part of Kasumi

Kasumi is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Please send bug-reports, opinions and patches, if any, to me by E-Mail.
My E-mail address is following: bluedwarf@openoffice.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gtk/gtk.h>
#include <string.h>
#include <stdio.h>
#include "intl.h"
#include "anthy_dic.h"
#include "wordlist.h"
#include "eucjp.h"
#include "io.h"

void analyze_anthy_dic(char *dic){
  int i,number_of_lines;
  char **lines;
  char *euc;

  first_anthy_word = NULL;

  /* Divide the given string(anthy private dictionary file) to lines dealing
     with the string as encoded by EUC-JP */
  number_of_lines = eucjp_count_str(dic, "\n");
  lines = (char **)calloc(number_of_lines,sizeof(char*));
  eucjp_split_str(dic,'\n',lines);

  /* Analyze the divided string deaing with the strings as encoded by EUC-JP,
     then generate an array of anthy_word*/
  i = 0;
  while(i<number_of_lines){
    if(is_comment_line(lines[i]) || is_empty_line(lines[i])){
      /* nothing to do */
    }else if(is_word_entry(lines[i])){
      AnthyWord newword;

      newword.yomi = get_utf8_yomi(lines[i]);
      newword.freq = get_frequency(lines[i]);
      newword.word = get_utf8_word(lines[i]);
      newword.part = NOTHING;

      while(!is_empty_line(lines[++i])){
        get_key_and_val(lines[i],&newword,i+1);
      }
      
      if(newword.part == NOTHING){
        g_printf("line %d: part of speech must be set.",i-1);
      }
      
      add_word(newword);      
      
      i--;
      free(newword.yomi);
      free(newword.word);
    }else{
      g_printf("Syntax error at line %d\n",i+1);
      exit(1);
    }
    i++;
  }
  
  for(i=0;i<number_of_lines;i++){
    free(lines[i]);
  }
  free(lines);
  
}

int count_figures(int number){
  int ret=1;

  if(number >= 0){
    number -= (number % 10);
    
    while(number != 0){
      number /= 10;
      number -= (number % 10);      
      ret++;
    }
  }else{
    /* not supported nagative integer */
  }

  return ret;
}

char *itoa(int number){
  int i = 0;
  char *ret;

  ret = (char *)calloc((count_figures(number)+1),sizeof(char));

  sprintf(ret,"%d",number);

  return ret;
}

void save_anthy_dic(){
  AnthyWord *p;
  int nbyte = 1,i=0;
  char c;
  char *data,*euc;

  p = first_anthy_word;
  while(p != NULL){
    nbyte += strlen(p->yomi) + count_figures(p->freq) + strlen(p->word) + 3;
    switch(p->part){
    case NOUN:
      nbyte += 81;
      break;
    case ADVERB:
      nbyte += 68;
      break;
    case PERSON:
      nbyte += 12;
      break;
    case PLACE:
      nbyte += 12;
      break;
    case ADJECTIVE:
      nbyte += 14;
      break;
    }
    
    p = p->next;
  }

  data = (char *)calloc(nbyte,sizeof(char));

  p = first_anthy_word;
  while(p != NULL){
    euc = g_convert(p->yomi,strlen(p->yomi),"EUC-JP","UTF-8",NULL,NULL,NULL);
    add_buffer(data,euc,strlen(euc),i);
    i += strlen(euc);
    free(euc);

    *(data+i) = SPACE;
    i++;

    euc = itoa(p->freq);
    add_buffer(data,euc,strlen(euc),i);
    i += strlen(euc);
    free(euc);

    *(data+i) = SPACE;
    i++;

    euc = g_convert(p->word,strlen(p->word),"EUC-JP","UTF-8",NULL,NULL,NULL);
    add_buffer(data,euc,strlen(euc),i);
    i += strlen(euc);
    free(euc);

    *(data+i) = '\n';
    i++;

    add_buffer(data,EUCJP_HINNSHI,strlen(EUCJP_HINNSHI),i);
    i += strlen(EUCJP_HINNSHI);

    add_buffer(data," = ",3,i);
    i += 3;

    switch(p->part){
    case NOUN:
      add_buffer(data,EUCJP_MEISHI,strlen(EUCJP_MEISHI),i);
      i += strlen(EUCJP_MEISHI);

      *(data+i) = '\n';
      i++;
      
      add_buffer(data,EUCJP_NASETSUZOKU,strlen(EUCJP_NASETSUZOKU),i);
      i += strlen(EUCJP_NASETSUZOKU);

      add_buffer(data," = ",3,i);
      i += 3;

      c = p->noun_na_connection == YES ? 'y' : 'n';
      *(data+i) = c;
      i++;

      *(data+i) = '\n';
      i++;

      add_buffer(data,EUCJP_SASETSUZOKU,strlen(EUCJP_SASETSUZOKU),i);
      i += strlen(EUCJP_SASETSUZOKU);

      add_buffer(data," = ",3,i);
      i += 3;

      c = p->noun_sa_connection == YES ? 'y' : 'n';
      *(data+i) = c;
      i++;

      *(data+i) = '\n';
      i++;
      
      add_buffer(data,EUCJP_SURUSETSUZOKU,strlen(EUCJP_SURUSETSUZOKU),i);
      i += strlen(EUCJP_SURUSETSUZOKU);

      add_buffer(data," = ",3,i);
      i += 3;

      c = p->noun_suru_connection == YES ? 'y' : 'n';
      *(data+i) = c;
      i++;

      *(data+i) = '\n';
      i++;
      
      add_buffer(data,EUCJP_GOKANNNOMIDEBUNNSETSU,
                 strlen(EUCJP_GOKANNNOMIDEBUNNSETSU),i);
      i += strlen(EUCJP_GOKANNNOMIDEBUNNSETSU);

      add_buffer(data," = ",3,i);
      i += 3;

      c = p->noun_gokan == YES ? 'y' : 'n';
      *(data+i) = c;
      i++;

      *(data+i) = '\n';
      i++;
      add_buffer(data,EUCJP_KAKUJOSHISETSUZOKU,strlen(EUCJP_KAKUJOSHISETSUZOKU),i);
      i += strlen(EUCJP_KAKUJOSHISETSUZOKU);

      add_buffer(data," = ",3,i);
      i += 3;

      c = p->noun_kakujoshi_connection == YES ? 'y' : 'n';
      *(data+i) = c;
      i++;

      *(data+i) = '\n';
      i++;
      
      break;
    case ADVERB:
      add_buffer(data,EUCJP_FUKUSHI,strlen(EUCJP_FUKUSHI),i);
      i += strlen(EUCJP_FUKUSHI);

      *(data+i) = '\n';
      i++;
      
      add_buffer(data,EUCJP_TOSETSUZOKU,strlen(EUCJP_TOSETSUZOKU),i);
      i += strlen(EUCJP_TOSETSUZOKU);

      add_buffer(data," = ",3,i);
      i += 3;

      c = p->adv_to_connection == YES ? 'y' : 'n';
      *(data+i) = c;
      i++;

      *(data+i) = '\n';
      i++;

      add_buffer(data,EUCJP_TARUSETSUZOKU,strlen(EUCJP_TARUSETSUZOKU),i);
      i += strlen(EUCJP_TARUSETSUZOKU);

      add_buffer(data," = ",3,i);
      i += 3;

      c = p->adv_taru_connection == YES ? 'y' : 'n';
      *(data+i) = c;
      i++;

      *(data+i) = '\n';
      i++;

      add_buffer(data,EUCJP_SURUSETSUZOKU,strlen(EUCJP_SURUSETSUZOKU),i);
      i += strlen(EUCJP_SURUSETSUZOKU);

      add_buffer(data," = ",3,i);
      i += 3;

      c = p->adv_suru_connection == YES ? 'y' : 'n';
      *(data+i) = c;
      i++;

      *(data+i) = '\n';
      i++;

      add_buffer(data,EUCJP_GOKANNNOMIDEBUNNSETSU
                 ,strlen(EUCJP_GOKANNNOMIDEBUNNSETSU),i);
      i += strlen(EUCJP_GOKANNNOMIDEBUNNSETSU);

      add_buffer(data," = ",3,i);
      i += 3;

      c = p->adv_gokan == YES ? 'y' : 'n';
      *(data+i) = c;
      i++;

      *(data+i) = '\n';
      i++;
      
      break;
    case PERSON:
      add_buffer(data,EUCJP_JINNMEI,strlen(EUCJP_JINNMEI),i);
      i += strlen(EUCJP_JINNMEI);
      break;
    case PLACE:
      add_buffer(data,EUCJP_CHIMEI,strlen(EUCJP_CHIMEI),i);
      i += strlen(EUCJP_CHIMEI);
      break;
    case ADJECTIVE:
      add_buffer(data,EUCJP_KEIYOUSHI,strlen(EUCJP_KEIYOUSHI),i);
      i += strlen(EUCJP_KEIYOUSHI);
      break;
    }

    *(data+i) = '\n';
    i++;
    *(data+i) = '\n';
    i++;
    
    p = p->next;
  }

  *(data+i) = '\0';

  write_data_to_dic(data);

  free(data);
}

AnthyWord *add_word(AnthyWord newword){
  GtkTreeIter iter;
  GtkTreePath *path;
  char *part_of_speech;
  AnthyWord *copyword;

  copyword = (AnthyWord *)calloc(1,sizeof(AnthyWord));
  *copyword = newword;
  copyword->word = (char *)calloc(strlen(newword.word),sizeof(char));
  strcpy(copyword->word,newword.word);
  copyword->yomi = (char *)calloc(strlen(newword.yomi),sizeof(char));
  strcpy(copyword->yomi,newword.yomi);
  
  gtk_list_store_append(wordlist,&iter);

  switch(newword.part){
  case NOUN:
    part_of_speech = UTF8_MEISHI;
    break;
  case ADVERB:
    part_of_speech = UTF8_FUKUSHI;
    break;
  case PERSON:
    part_of_speech = UTF8_JINNMEI;
    break;
  case PLACE:
    part_of_speech = UTF8_CHIMEI;
    break;
  case ADJECTIVE:
    part_of_speech = UTF8_KEIYOUSHI;
    break;
  }

  gtk_list_store_set(wordlist, &iter,
                     COL_WORD, copyword->word,
                     COL_YOMI, copyword->yomi,
                     COL_FREQ, (guint)copyword->freq,
                     COL_PART, part_of_speech,
                     -1);

  path = gtk_tree_model_get_path(GTK_TREE_MODEL(wordlist),&iter);
  copyword->row_reference=gtk_tree_row_reference_new(GTK_TREE_MODEL(wordlist),
                                                     path);
  
  copyword->next = NULL;

  if(first_anthy_word == NULL){
    first_anthy_word = copyword;
  }else{
    AnthyWord *lastword;
    lastword = last_anthy_word();
    lastword->next = copyword;
  }

  /* return new allocated AnthyWord */
  return copyword;
}

int remove_word(AnthyWord *word){
  AnthyWord *p,*prev;
  GtkTreePath *path;
  GtkTreeIter iter;
  
  p = first_anthy_word;

  if(p == word){
    first_anthy_word = p->next;
    free(p->word);
    free(p->yomi);
    path = gtk_tree_row_reference_get_path(p->row_reference);
    gtk_tree_model_get_iter(GTK_TREE_MODEL(wordlist),&iter,path);
    gtk_list_store_remove(GTK_LIST_STORE(wordlist),&iter);
    free(p);
    return 1;
  }

  while(p->next != word){
    p = p->next;
    if(p == NULL){
      /* specified word was not registered */
      return 0;
    }
  }

  prev = p;
  p = prev->next;
  prev->next = prev->next->next;

  free(p->word);
  free(p->yomi);
  path = gtk_tree_row_reference_get_path(p->row_reference);
  gtk_tree_model_get_iter(GTK_TREE_MODEL(wordlist),&iter,path);
  gtk_list_store_remove(GTK_LIST_STORE(wordlist),&iter);
  free(p);
  return 1;
}

AnthyWord *last_anthy_word(){
  AnthyWord *p;
  p = first_anthy_word;

  if(p == NULL) return NULL;

  while(p->next != NULL){
    p = p->next;
  }

  return p;
}

AnthyWord *get_anthy_word_from_iter(GtkTreeModel *model, GtkTreeIter *aiter){
  AnthyWord *p;
  GtkTreePath *path;
  GtkTreeIter iter;
  p = first_anthy_word;

  while(p){
    path = gtk_tree_row_reference_get_path(p->row_reference);
    gtk_tree_model_get_iter(model,&iter,path);
    if((iter.stamp == aiter->stamp) &&
       (iter.user_data == aiter->user_data) &&
       (iter.user_data2 == aiter->user_data2) &&
       (iter.user_data3 == aiter->user_data3)){
      return p;
    }
    p = p->next;
  }

  return NULL;
}
