/*
Kasumi: Dictionary management tool for Anthy
Copyright (C) 2004 Takashi Nakamoto

This file is part of Kasumi

Kasumi is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Please send bug-reports, opinions and patches, if any, to me by E-Mail.
My E-mail address is following: bluedwarf@openoffice.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gtk/gtk.h>
#include <string.h>
#include "intl.h"
#include "eucjp.h"
#include "anthy_dic.h"

int eucjp_count_str(const char *str, const char *str_search)
{
  int number = 1;
  char *p;

  if(!(p = strstr(str,str_search))){
    return 0;
  }

  while(p = strstr(p,str_search)){
    number++;
    p++;
  }

  return number;
}

void eucjp_split_str(char *str, const char delimiter, char **lines){
  int i=0, j=0;
  char *start,*end;

  start = str;
  while(end = strchr(start,(int)delimiter)){
    lines[i] = (char *)malloc(end-start+1);
    j=0;
    while(start < end){
      lines[i][j] = *start;
      j++;
      start++;
    }
    lines[i++][j] = '\0';

    start = end + 1;
  }

  end = strchr(start,(int)'\0');
  lines[i] = (char *)malloc(end-start+1);
  j=0;
  while(start < end){
    lines[i][j] = *start;
    j++;
    start++;
  }
  lines[i++][j] = '\0';
}

int is_comment_line(const char *line){
  int i = 0;

  while(*(line+i) == SPACE){
    i++;
  }
  
  if(*(line+i) == '#'){
    return TRUE;
  }
  
  return FALSE;
}

int is_empty_line(const char *line){
  int i = 0;
  
  while(*(line+i) == SPACE){
    i++;
  }

  if(*(line+i) == '\0'){
    return TRUE;
  }
  
  return FALSE;
}

int is_word_entry(const char *line){
  int i = 0;
  enum {YOMI,FREQ,WORD} status;

  status = YOMI;

  while(*(line+i) != '\0'){
    switch(status){
    case YOMI:
      if(*(line+i) == SPACE){
        status = FREQ;
      }
      break;
    case FREQ:
      if(*(line+i) == SPACE){
        status = WORD;
      }else if(*(line+i) < '0' || *(line+i) > '9'){
      /* if frequency entry does not consist only of digit */        
        return FALSE;
      }
      break;
    case WORD:
      break;
    }
    i++;
  }

  if(status == WORD){
    return TRUE;
  }

  return FALSE;
}

gchar *get_utf8_yomi(const char *word_entry){
  char *yomi;
  gchar *ret;
  int i = 0;
  
  while(*(word_entry+i) != SPACE){
    i++;
  }

  yomi = (char *)calloc(i+1,sizeof(char));
  
  i=0;
  while(*(word_entry+i) != SPACE){
    yomi[i] = *(word_entry+i);
    i++;
  }
  yomi[i] = '\0';
  
  ret = g_convert(yomi,strlen(yomi),"UTF-8","EUC-JP",NULL,NULL,NULL);
  free(yomi);

  return ret;
}

int get_frequency(const char *word_entry){
  int i=0,ret=0;

  while(*(word_entry+i) != SPACE) i++;
  i++;
  while(*(word_entry+i) != SPACE){
    ret = ret * 10 + (*(word_entry+i) - '0');
    i++;
  }
  
  return ret;
}

char *get_utf8_word(const char *word_entry){
  char *word,*p;
  gchar *ret;
  int i = 0,j;
  
  while(*(word_entry+i) != SPACE) i++;
  i++;
  while(*(word_entry+i) != SPACE) i++;
  i++;
  j=0;
  p = word_entry+i;
  while(*(word_entry+i) != '\0'){
    i++; j++;
  }

  word = (char *)calloc(j+1,sizeof(char));
  
  i=0;
  while(*(p+i) != '\0'){
    word[i] = *(p+i);
    i++;
  }
  word[i] = '\0';
  
  ret = g_convert(word,strlen(word),"UTF-8","EUC-JP",NULL,NULL,NULL);
  free(word);

  return ret;
}

void get_key_and_val(const char *line, AnthyWord *newword,int nline){
  int key_len=0,val_len=0,i=0;
  char *key,*val;
  
  while(*(line+i) != EQUAL){
    i++;
  }
  key_len = i + 1;
  key = (char *)calloc(key_len,sizeof(char));

  i++;
  while(*(line+i) != '\0'){
    i++;
  }
  val_len = i - key_len + 1;
  val = (char *)calloc(val_len,sizeof(char));

  for(i=0;i<key_len;i++){
    key[i] = *(line+i);
    
    if(*(line+i) == SPACE){
      break;
    }
  }
  key[i] = '\0';

  for(i=0;i<val_len;i++){
    if(*(line+i) == SPACE){
      continue;
    }
    
    val[i] = *(line+key_len+1+i);
  }
  val[i] = '\0';

  if(strcmp(key,EUCJP_HINNSHI) == 0){
    if(strcmp(val,EUCJP_MEISHI) == 0){
      newword->part = NOUN;
    }else if(strcmp(val,EUCJP_FUKUSHI) == 0){
      newword->part = ADVERB;
    }else if(strcmp(val,EUCJP_JINNMEI) == 0){
      newword->part = PERSON;
    }else if(strcmp(val,EUCJP_CHIMEI) == 0){
      newword->part = PLACE;
    }else if(strcmp(val,EUCJP_KEIYOUSHI) == 0){
      newword->part = ADJECTIVE;
    }else{
      g_printf("line %d: invalid part of speech.\n",nline);
      exit(1);
    }
  }else if(strcmp(key,EUCJP_NASETSUZOKU) == 0){
    if(newword->part == NOUN){
      newword->noun_na_connection = val_is_yes_or_no(val);
      if(newword->noun_na_connection == ERROR){
        g_printf("line %d: the value of %s must be 'y' or 'n'.\n",
                 nline,EUCJP_NASETSUZOKU);
        exit(1);
      }
    }else{
      g_printf("line %d: %s is an option only for %s.\n",
               nline,EUCJP_NASETSUZOKU,EUCJP_MEISHI);
      exit(1);      
    }
  }else if(strcmp(key,EUCJP_GOKANNNOMIDEBUNNSETSU) == 0){
    if(newword->part == NOUN){
      newword->noun_gokan = val_is_yes_or_no(val);
      if(newword->noun_gokan == ERROR){
        g_printf("line %d: the value of %s must be 'y' or 'n'.\n",
                 nline,EUCJP_GOKANNNOMIDEBUNNSETSU);
        exit(1);
      }
    }else if(newword->part == ADVERB){
      newword->adv_gokan = val_is_yes_or_no(val);
      if(newword->adv_gokan == ERROR){
        g_printf("line %d: the value of %s must be 'y' or 'n'.\n",
                 nline,EUCJP_GOKANNNOMIDEBUNNSETSU);
        exit(1);
      }
    }else{
      g_printf("line %d: %s is an option only for %s and %s.\n",
               nline,EUCJP_GOKANNNOMIDEBUNNSETSU,EUCJP_MEISHI,EUCJP_FUKUSHI);
      exit(1);      
    }
  }else if(strcmp(key,EUCJP_SASETSUZOKU) == 0){
    if(newword->part == NOUN){
      newword->noun_sa_connection = val_is_yes_or_no(val);
      if(newword->noun_sa_connection == ERROR){
        g_printf("line %d: the value of %s must be 'y' or 'n'.\n",
                 nline,EUCJP_SASETSUZOKU);
        exit(1);
      }
    }else{
      g_printf("line %d: %s is an option only for %s.\n",
               nline,EUCJP_SASETSUZOKU,EUCJP_MEISHI);
      exit(1);      
    }
  }else if(strcmp(key,EUCJP_SURUSETSUZOKU) == 0){
    if(newword->part == NOUN){
      newword->noun_suru_connection = val_is_yes_or_no(val);
      if(newword->noun_suru_connection == ERROR){
        g_printf("line %d: the value of %s must be 'y' or 'n'.\n",
                 nline,EUCJP_SURUSETSUZOKU);
        exit(1);
      }
    }else if(newword->part == ADVERB){
      newword->adv_suru_connection = val_is_yes_or_no(val);
      if(newword->adv_suru_connection == ERROR){
        g_printf("line %d: the value of %s must be 'y' or 'n'.\n",
                 nline,EUCJP_SURUSETSUZOKU);
        exit(1);
      }
    }else{
      g_printf("line %d: %s is an option only for %s and %s.\n",
               nline,EUCJP_SURUSETSUZOKU,EUCJP_MEISHI,EUCJP_FUKUSHI);
      exit(1);      
    }
  }else if(strcmp(key,EUCJP_KAKUJOSHISETSUZOKU) == 0){
    if(newword->part == NOUN){
      newword->noun_kakujoshi_connection = val_is_yes_or_no(val);
      if(newword->noun_kakujoshi_connection == ERROR){
        g_printf("line %d: the value of %s must be 'y' or 'n'.\n",
                 nline,EUCJP_KAKUJOSHISETSUZOKU);
        exit(1);
      }
    }else{
      g_printf("line %d: %s is an option only for %s.\n",
               nline,EUCJP_KAKUJOSHISETSUZOKU,EUCJP_MEISHI);
      exit(1);      
    }
  }else if(strcmp(key,EUCJP_TOSETSUZOKU) == 0){
    if(newword->part == ADVERB){
      newword->adv_to_connection = val_is_yes_or_no(val);
      if(newword->adv_to_connection == ERROR){
        g_printf("line %d: the value of %s must be 'y' or 'n'.\n",
                 nline,EUCJP_TOSETSUZOKU);
        exit(1);
      }
    }else{
      g_printf("line %d: %s is an option only for %s.\n",
               nline,EUCJP_TOSETSUZOKU,EUCJP_FUKUSHI);
      exit(1);      
    }
  }else if(strcmp(key,EUCJP_TARUSETSUZOKU) == 0){
    if(newword->part == ADVERB){
      newword->adv_taru_connection = val_is_yes_or_no(val);
      if(newword->adv_taru_connection == ERROR){
        g_printf("line %d: the value of %s must be 'y' or 'n'.\n",
                 nline,EUCJP_TARUSETSUZOKU);
        exit(1);
      }
    }else{
      g_printf("line %d: %s is an option only for %s.\n",
               nline,EUCJP_TARUSETSUZOKU,EUCJP_FUKUSHI);
      exit(1);
    }
  }
  
  free(key);
  free(val);
}

enum yes_or_no val_is_yes_or_no(const char *val){
  if(strcmp(val,"y") == 0){
    return YES;
  }else if(strcmp(val,"n") == 0){
    return NO;
  }else{
    return ERROR;
  }
}
