/*
 *  Copyright (C) 2006 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "rd-image-factory.h"
#include "rendako.h"
#include "rd-thumbnailer.h"
#include "rd-marshalers.h"

#include <glib/gi18n.h>

#define N_THUMBNAILERS 2

typedef struct _RdImageFactoryPrivate RdImageFactoryPrivate;
struct _RdImageFactoryPrivate
{
	GList  *thumbnailers;
	GQueue *uris;
};

enum {
	READY_SIGNAL,
	LAST_SIGNAL
};

static gint rd_image_factory_signals[LAST_SIGNAL] = {0};

#define RD_IMAGE_FACTORY_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), RD_TYPE_IMAGE_FACTORY, RdImageFactoryPrivate))

static void     rd_image_factory_class_init     (RdImageFactoryClass *klass);
static void     rd_image_factory_init           (RdImageFactory      *image_factory);
static void     rd_image_factory_dispose        (GObject         *object);
static void     rd_image_factory_finalize       (GObject         *object);

static void cb_ready_image (RdThumbnailer *thumbnailer, gpointer image_data, const gchar *uri, gpointer data);

static GObjectClass *parent_class = NULL;

G_DEFINE_TYPE (RdImageFactory, rd_image_factory, G_TYPE_OBJECT)

static void
rd_image_factory_class_init (RdImageFactoryClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *)   klass;
  
	gobject_class->dispose  = rd_image_factory_dispose;
	gobject_class->finalize = rd_image_factory_finalize;

	rd_image_factory_signals[READY_SIGNAL]
		= g_signal_new ("ready",
				G_TYPE_FROM_CLASS (klass),
				(GSignalFlags) (G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED),
				G_STRUCT_OFFSET (RdImageFactoryClass, ready),
				NULL, NULL,
				_rd_marshal_VOID__POINTER_STRING,
				G_TYPE_NONE, 2,
				G_TYPE_POINTER,
				G_TYPE_STRING);

	g_type_class_add_private (gobject_class, sizeof(RdImageFactoryPrivate));
}

static void
rd_image_factory_init (RdImageFactory *image_factory)
{
	gint i;
	RdImageFactoryPrivate *priv = RD_IMAGE_FACTORY_GET_PRIVATE (image_factory);

	priv->uris = NULL;

	for (i = 0; i < N_THUMBNAILERS; i++)
	{
		RdThumbnailer *thumbnailer = rd_thumbnailer_new ();
		g_signal_connect (thumbnailer, "ready", G_CALLBACK (cb_ready_image), image_factory);
		priv->thumbnailers = g_list_prepend (priv->thumbnailers, thumbnailer);
	}

	priv->uris = g_queue_new ();
}

static void
cb_ready_image (RdThumbnailer *thumbnailer, gpointer image_data, const gchar *uri, gpointer data)
{
	RdImageFactory *factory = RD_IMAGE_FACTORY (data);
	RdImageFactoryPrivate *priv = RD_IMAGE_FACTORY_GET_PRIVATE (data);
	gchar *queue_uri;

	g_signal_emit (factory, rd_image_factory_signals[READY_SIGNAL], 0, image_data, uri);

	if (g_queue_is_empty (priv->uris))
		return;

	/* request a new image for an uri in queue */
	queue_uri = g_queue_pop_head (priv->uris);
	rd_thumbnailer_request_uri (thumbnailer, queue_uri);
	g_free (queue_uri);
}

static void
cb_each_thumbnailer (RdThumbnailer *thumbnailer, gpointer user_data)
{
	g_signal_handlers_disconnect_by_func (thumbnailer,
					      G_CALLBACK (cb_ready_image),
					      NULL);
	g_object_unref (thumbnailer);
}

static void
rd_image_factory_dispose (GObject *object)
{
	RdImageFactoryPrivate *priv = RD_IMAGE_FACTORY_GET_PRIVATE (object);

	if (priv->thumbnailers)
	{
		g_list_foreach (priv->thumbnailers, (GFunc) cb_each_thumbnailer, NULL);
		priv->thumbnailers = NULL;	
	}

	if (priv->uris)
	{
		g_queue_free (priv->uris);
		priv->uris = NULL;
	}

	if (G_OBJECT_CLASS(parent_class)->dispose)	
		G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
rd_image_factory_finalize (GObject *object)
{
	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS (parent_class)->finalize (object);
}

RdImageFactory *
rd_image_factory_new (void)
{
	RdImageFactory *image_factory;

	image_factory = g_object_new (RD_TYPE_IMAGE_FACTORY, NULL);

	return image_factory;
}

void
rd_image_factory_request_uri (RdImageFactory *factory, const gchar *uri)
{
	RdImageFactoryPrivate *priv = RD_IMAGE_FACTORY_GET_PRIVATE (factory);
	GList *list;

	for (list = priv->thumbnailers; list; list = g_list_next (list))
	{
		RdThumbnailer *t = RD_THUMBNAILER (list->data);	

		if (!rd_thumbnailer_is_creating (t))
		{
			rd_thumbnailer_request_uri (t, uri);
			return ;
		}
	}

	/* All thumbnailers are on duty */
	g_queue_push_tail (priv->uris, g_strdup (uri));
}

