/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <libxml/parser.h>

#include "kz-home.h"
#include "gobject-utils.h"

static void kz_home_button_class_init (KzHomeButtonClass *class);
static void kz_home_button_init       (KzHomeButton *kz);
static void kz_home_button_destroy    (GtkObject *object);

static GtkWidget *kz_home_menu_new   (KzHomeButton *kzhome);
static KzHomeList *kz_home_parse_site(xmlDocPtr doc, xmlNodePtr cur, KzHomeButton *kzhome);
static void kz_home_file_load        (KzHomeButton *kzhome);
static void kz_home_file_save        (KzHomeButton *kzhome);
static void kz_home_menu_update      (KzHomeButton *kzhome);

/* submenu */
static void kz_home_menu_submenu_popup(KzHomeList *kzhomelist, GdkEventButton *event);
static void kz_home_menu_submenu_delete(GtkMenuItem *menuite, KzHomeList *kzhomelist);

/* callbacks */
static void kz_home_button_press_cb  (GtkWidget *widget, GdkEventButton *event,
				      KzHomeButton *kzhome);
static void kz_home_menu_add_press_cb(GtkWidget *widget, GdkEventButton *event,
				      KzHomeButton *kzhome);
static void kz_home_menu_press_cb    (GtkWidget *widget, GdkEventButton *event,
				      KzHomeList *kzhomelist);

static GtkButtonClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE(kz_home_button, "KzHomeButton", KzHomeButton,
		   kz_home_button_class_init, kz_home_button_init,
		   GTK_TYPE_BUTTON)

static void
kz_home_button_class_init (KzHomeButtonClass *class)
{
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent (class);

	object_class = (GtkObjectClass *) class;

	/* GtkObject signals */
	object_class->destroy = kz_home_button_destroy;
}

static void
kz_home_button_init (KzHomeButton *kzhome)
{
	/* widgets */
	kzhome->kz = NULL;
	kzhome->PopupMenu = NULL;
	
	/* lairs list  */
	kzhome->home_list = NULL;
}

/* create new home button */
GtkWidget *
kz_home_button_new (KzWindow *kz)
{
	KzHomeButton *kzhome = KZ_HOME_BUTTON(g_object_new(kz_home_button_get_type(), NULL));
	
	kzhome->kz = kz;
	kzhome->home_list = g_slist_alloc();
	
	/* create button */
	gtk_container_add(GTK_CONTAINER(kzhome),
			  gtk_image_new_from_stock(GTK_STOCK_HOME,GTK_ICON_SIZE_SMALL_TOOLBAR));
	gtk_button_set_relief(GTK_BUTTON(kzhome), GTK_RELIEF_NONE);
	g_signal_connect(G_OBJECT(kzhome), "button_press_event",
			 G_CALLBACK(kz_home_button_press_cb), kzhome);
		/* load lairs list file */
	kz_home_file_load(kzhome);
	
	/* create lairs menu */
	kzhome->PopupMenu = kz_home_menu_new(kzhome);
	gtk_widget_show_all(GTK_WIDGET(kzhome));
	
	return GTK_WIDGET(kzhome);
}

static void
kz_home_button_destroy (GtkObject *object)
{
	g_slist_free(KZ_HOME_BUTTON(object)->home_list);
	KZ_HOME_BUTTON(object)->home_list = NULL;
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static KzHomeList *
kz_home_parse_site(xmlDocPtr doc, xmlNodePtr cur, KzHomeButton *kzhome)
{
	KzHomeList *list = NULL;
	list = g_new0(KzHomeList, 1);
	
	list->kzhome = kzhome;
	list->title = xmlGetProp(cur, "title");
	list->uri   = xmlGetProp(cur, "uri");

	return list;
}

static void 
kz_home_file_load (KzHomeButton *kzhome)
{
        gchar *homefile;
	xmlDocPtr doc;
	xmlNodePtr cur;
	
        homefile = g_strdup_printf("%s/.kazehakase/home.xml", g_get_home_dir());
	doc = xmlParseFile(homefile);
	g_free(homefile);
	if (doc == NULL) return;

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL)
	{
		xmlFreeDoc(doc);
		return;
	}
	
	/* base name is <home> */
	if (xmlStrcmp(cur->name, (const xmlChar *) "home"))
	{
		xmlFreeDoc(doc);
		return;
	}
	
	cur = cur->xmlChildrenNode;
	while (cur != NULL) 
	{
		if ((!xmlStrcmp(cur->name, (const xmlChar*) "site")))
		{
			g_slist_append(kzhome->home_list, 
				       kz_home_parse_site(doc, cur, kzhome));
		}
		cur = cur->next;
	}
	xmlFreeDoc(doc);
}
/* save home.xml (always create newfile, silly!)*/
static void 
kz_home_file_save (KzHomeButton *kzhome)
{
        gchar *homefile;
	xmlDocPtr doc;
	xmlNodePtr home,site;

	KzHomeList *kzhomelist;
	GSList *list;
	guint list_num, n;
	
	/* create new doc pointer */
	doc = xmlNewDoc("1.0");	

	/* base name is <home> */
	home = xmlNewNode(NULL, "home");
	xmlNodeSetName(home, "home");
	xmlDocSetRootElement(doc, home);
	
	/* add sites */
	list = kzhome->home_list;
	list_num = g_slist_length(kzhome->home_list);
	for (n = 1; n < list_num; n++)
	{
		kzhomelist = (KzHomeList*) g_slist_nth_data(list, n);

		site = xmlNewTextChild(home, NULL, "site", NULL);
		xmlNewProp(site, "uri",   kzhomelist->uri);
		xmlNewProp(site, "title", kzhomelist->title);
	}
	
        homefile = g_strdup_printf("%s/.kazehakase/home.xml", g_get_home_dir());
	xmlSaveFormatFileEnc (homefile, doc, "UTF-8",1);
	g_free(homefile);
	
	xmlFreeDoc(doc);
}

/* create lairs menu */
static GtkWidget *
kz_home_menu_new(KzHomeButton *kzhome)
{
	GtkWidget *menu, *menu_item;
	GSList *list;
	KzHomeList *kzhomelist;
	guint list_num, n;
	
	menu = gtk_menu_new();	
	/* Add lairs */
	menu_item = gtk_menu_item_new_with_label("Add lairs");
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      menu_item);
	g_signal_connect(G_OBJECT(menu_item), "button_press_event",
			 G_CALLBACK(kz_home_menu_add_press_cb),
			 kzhome);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu),
			      gtk_separator_menu_item_new());
	
	/* create lairs menu */
	list = kzhome->home_list;
	list_num = g_slist_length(kzhome->home_list);
	for (n = 1; n < list_num; n++)
	{
		kzhomelist = (KzHomeList*) g_slist_nth_data(list, n);

		menu_item = gtk_menu_item_new_with_label(kzhomelist->title);
		gtk_menu_shell_append(GTK_MENU_SHELL(menu),
				      menu_item);
		g_signal_connect(G_OBJECT(menu_item), "button_press_event",
				 G_CALLBACK(kz_home_menu_press_cb),
				 kzhomelist);
	}
	gtk_widget_show_all(menu);
	return menu;
}

static void
kz_home_menu_submenu_popup(KzHomeList *kzhomelist, GdkEventButton *event)
{
	GtkWidget *submenu, *menu_item;
	
	submenu = gtk_menu_new();

	menu_item = gtk_menu_item_new_with_label("Remove from list");
	gtk_menu_shell_append(GTK_MENU_SHELL(submenu),
			      menu_item);
	g_signal_connect(G_OBJECT(menu_item), "activate",
			 G_CALLBACK(kz_home_menu_submenu_delete),
			 kzhomelist);
	gtk_widget_show_all(submenu);
	gtk_menu_popup(GTK_MENU(submenu), NULL, NULL, NULL,
		       NULL, event->button, event->time);
}

static void
kz_home_menu_submenu_delete(GtkMenuItem *menuitem, KzHomeList *kzhomelist)
{
	g_slist_remove(kzhomelist->kzhome->home_list, kzhomelist);
	
	kz_home_menu_update(kzhomelist->kzhome);
	
	g_free(kzhomelist);
}

static void
kz_home_menu_update(KzHomeButton *kzhome)
{
	/* save home.xml */
	kz_home_file_save(kzhome);
	
	/* update lairs menu */
	if (kzhome->PopupMenu != NULL)
	{	
		gtk_widget_destroy(kzhome->PopupMenu);
		g_free(kzhome->PopupMenu);
	}
	kzhome->PopupMenu = kz_home_menu_new(kzhome);
}

static void
kz_home_button_press_cb(GtkWidget *widget, GdkEventButton *event,
			KzHomeButton *kzhome)
{
	KzWindow *kz;
	kz = kzhome->kz;
	
	switch ( event->button ) {
	 case 1: /* open in current tab */
		if (g_slist_length(kzhome->home_list) > 1)
			kz_window_load_url(kz, ((KzHomeList *)g_slist_nth_data(kzhome->home_list, 1))->uri);
		break;
	 case 2: /* open in new tab */
		if (g_slist_length(kzhome->home_list) > 1)
			kz_window_open_new_tab(kz, ((KzHomeList *)g_slist_nth_data(kzhome->home_list, 1))->uri);
		break;
	 case 3: /* popup secret menu */
		gtk_menu_popup(GTK_MENU(kzhome->PopupMenu), NULL, NULL, NULL,
			       NULL, event->button, event->time);
		break;
	}
}

static void
kz_home_menu_add_press_cb(GtkWidget *widget, GdkEventButton *event,
			  KzHomeButton *kzhome)
{
	GSList *list;
	
	KzWindow *kz;
	KzHomeList *kzhomelist = NULL;

	kz = kzhome->kz;
	
	kzhomelist = g_new0(KzHomeList, 1);
	
	kzhomelist->kzhome = kzhome;

	kzhomelist->title = kz_window_get_title(kz);
	kzhomelist->uri   = kz_window_get_uri(kz);

	/* append new site */
	list = g_slist_last(kzhome->home_list);
	g_slist_append(list, kzhomelist);
	
	kz_home_menu_update(kzhome);
}

static void
kz_home_menu_press_cb(GtkWidget *widget, GdkEventButton *event,
			     KzHomeList *kzhomelist)
{
	KzWindow *kz;
	kz = kzhomelist->kzhome->kz;
	
	if (kzhomelist->uri)
	{
		switch ( event->button ) {
		 case 1:
			kz_window_load_url(kz, kzhomelist->uri);
			break;
		 case 2:
			kz_window_open_new_tab(kz, kzhomelist->uri);
			break;
		 case 3: /* popup menu(remove etc.) */
			kz_home_menu_submenu_popup(kzhomelist, event);
			break;
		}
	}
}
