/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <gtk/gtk.h>
#include <fcntl.h>
#include <sys/utsname.h>
#include <nsBuildID.h>

#include "config.h"
#include "prefs.h"

static guint prefs_parser(GScanner *scanner);
static void prefs_load(void);
static gchar *prefs_set_user_agent ();

/* these codes were picked from dillo-0.7.0pre */
/* symbol array */
static const struct
{
	gchar *name;
	guint  token;
} symbols[] = 
{
	{ "width", KZRC_TOKEN_WIDTH },
	{ "height", KZRC_TOKEN_HEIGHT },
	{ "proxy", KZRC_TOKEN_PROXY },
	{ "mailer", KZRC_TOKEN_MAILER },
	{ "user_agent", KZRC_TOKEN_USERAGENT },
};

static const guint n_symbols = sizeof (symbols) / sizeof (symbols[0]);

/*
 * Read tokens from kzrc and set values in the prefs structure.
 * (Although this function can be called several times, and not leak,
 *  preferences aren't yet enabled for on-the-fly changes).
 */
static guint 
prefs_parser(GScanner *scanner)
{
	guint symbol;

	/* expect a valid symbol */
	g_scanner_get_next_token(scanner);
	symbol = scanner->token;
	if (scanner->token == G_TOKEN_EQUAL_SIGN)
	{
		g_scanner_get_next_token (scanner);
		return G_TOKEN_NONE;
	} else if (symbol < KZRC_TOKEN_FIRST || symbol > KZRC_TOKEN_LAST )
		return G_TOKEN_SYMBOL;

	/* expect '=' */
	g_scanner_get_next_token(scanner);
	if (scanner->token != G_TOKEN_EQUAL_SIGN)
		return G_TOKEN_EQUAL_SIGN;

	/* expect a string */
	g_scanner_get_next_token(scanner);
	if (scanner->token != G_TOKEN_STRING)
		return G_TOKEN_STRING;

	/* assign value and exit successfully */
	switch (symbol) 
	{
	 case KZRC_TOKEN_WIDTH:
		prefs.width = strtol(scanner->value.v_string,NULL,10);
		break;
	 case KZRC_TOKEN_HEIGHT:
		prefs.height = strtol(scanner->value.v_string,NULL,10);
		break;
	 case KZRC_TOKEN_PROXY:
		if (prefs.proxy)
			g_free(prefs.proxy);
		prefs.proxy = g_strdup(scanner->value.v_string);
		break;
	 case KZRC_TOKEN_MAILER:
		if (prefs.mailer)
			g_free(prefs.mailer);
		prefs.mailer = g_strdup(scanner->value.v_string);
		break;
	 case KZRC_TOKEN_USERAGENT:
		if (prefs.user_agent)
			g_free(prefs.user_agent);
		prefs.user_agent = g_strdup(scanner->value.v_string);
		break;
	 default:
		break;   /* Not reached */
	}
	return G_TOKEN_NONE;
}

static void 
prefs_load(void)
{
	GScanner *scanner;
	gint fd;
	guint i, expected_token;
	gchar *file;

	/* Here we load and set options from dillorc */
	file = g_strdup_printf("%s/%s", g_get_home_dir(), ".kazehakase/kzrc");
	fd = open(file, O_RDONLY);
	g_free(file);
	if (fd < 0 && (fd = open("/etc/kzrc", O_RDONLY)) < 0)
		return;

	fcntl(fd, F_SETFL, FD_CLOEXEC | fcntl(fd, F_GETFL));

	scanner = g_scanner_new(NULL);

	/* Adjust lexing behaviour to suit our needs */
	/* Specifies the chars which can be used in identifiers */
	scanner->config->cset_identifier_nth = (
						G_CSET_a_2_z
						"~-_:&%#/.0123456789"
						G_CSET_A_2_Z
						G_CSET_LATINS   /*??? I don't know if we need these two */
						G_CSET_LATINC   /*??? */
						);
	/* Specifies the chars which can start identifiers */
	scanner->config->cset_identifier_first = (
						  G_CSET_a_2_z
						  G_CSET_A_2_Z
						  "_0123456789"
						  );
	/* Don't return G_TOKEN_SYMBOL, but the symbol's value */
	scanner->config->symbol_2_token = TRUE;
	/* Don't return G_TOKEN_IDENTIFIER, but convert it to string */
	scanner->config->identifier_2_string = TRUE;

	/* load symbols into the scanner */
	g_scanner_freeze_symbol_table(scanner);
	for (i = 0; i < n_symbols; i++)
		g_scanner_add_symbol(scanner, symbols[i].name,
				     GINT_TO_POINTER (symbols[i].token));
	g_scanner_thaw_symbol_table(scanner);
	
	/* feed in the text */
	g_scanner_input_file(scanner, fd);

	/* give the error handler an idea on how the input is named */
	scanner->input_name = "kzrc";

	/*
	 * Scanning loop, we parse the input until it's end is reached,
	 * the scanner encountered a lexing error, or our sub routine came
	 * across invalid syntax
	 */
	do {
		expected_token = prefs_parser(scanner);

		/* Give an error message upon syntax errors */
		if (expected_token == G_TOKEN_SYMBOL)
			g_scanner_unexp_token (scanner, expected_token, NULL, "symbol", NULL,
					       NULL, FALSE);
		else if (expected_token == G_TOKEN_STRING)
			g_scanner_unexp_token (scanner, expected_token, NULL, "string", NULL,
					       NULL, FALSE);
		else if (expected_token == G_TOKEN_EQUAL_SIGN)
			g_scanner_unexp_token (scanner, expected_token, NULL, "=", NULL,
					       NULL, FALSE);
		g_scanner_peek_next_token (scanner);
	} while ( /* expected_token == G_TOKEN_NONE && */
		  scanner->next_token != G_TOKEN_EOF &&
		  scanner->next_token != G_TOKEN_ERROR);
	
	/* give an error message upon syntax errors */
	if (expected_token != G_TOKEN_NONE)
		g_scanner_unexp_token(scanner, expected_token, NULL, "symbol", NULL,
				      NULL, TRUE);

	/* finish parsing */
	g_scanner_destroy(scanner);
}

static gchar *
prefs_set_user_agent ()
{
	static gchar *user_agent = NULL;
	struct utsname name;
	gchar *system;

	if (uname (&name) >= 0)
	{
		system = g_strdup_printf ("%s %s",
					  name.sysname, 
					  name.machine);
	}
	else
	{
		system = g_strdup ("Unknown");
	}
		
	user_agent = g_strdup_printf("Mozilla/5.0 %s/" VERSION " (X11; %s; U;) Gecko/%d",
				     PACKAGE,
				     system,
				     NS_BUILD_ID/100);
	g_free (system);

	return user_agent;
}

void
prefs_init(void)
{
	prefs.proxy  = NULL;
	prefs.mailer = NULL;
	prefs.user_agent = prefs_set_user_agent();
	prefs.width  = 640;
	prefs.height = 450;
	prefs_load();
}

void 
prefs_free(void)
{
	if (prefs.proxy)
		g_free(prefs.proxy);
	if (prefs.mailer)
		g_free(prefs.mailer);
	if (prefs.user_agent)
		g_free(prefs.user_agent);
}
