/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-tablabel.h"

#include "gobject-utils.h"

/* XPM */
#include "pixmaps/small-close.xpm"

static void kz_tab_label_class_init (KzTabLabelClass *class);
static void kz_tab_label_init       (KzTabLabel *kztab);
static void kz_tab_label_destroy    (GtkObject *object);
static void kz_tab_label_close_cb   (GtkWidget *button, KzTabLabel *kztab);
static void kz_tab_label_color_init (void);

/* callbacks for mozilla */
static void title_changed_cb (GtkMozEmbed *embed, KzTabLabel *kztab);
static void net_start_cb     (GtkMozEmbed *embed, KzTabLabel *kztab);
static void net_stop_cb      (GtkMozEmbed *embed, KzTabLabel *kztab);
static void destroy_cb       (GtkObject *object, KzTabLabel *kztab);

static GtkHBoxClass *parent_class = NULL;

static GtkRcStyle *kz_tab_label_style_normal  = NULL;
static GtkRcStyle *kz_tab_label_style_loading = NULL;
 
KZ_OBJECT_GET_TYPE(kz_tab_label, "KzTabLabel", KzTabLabel,
		   kz_tab_label_class_init, kz_tab_label_init,
		   GTK_TYPE_HBOX)

static void
kz_tab_label_class_init (KzTabLabelClass *class)
{
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent (class);

	object_class = (GtkObjectClass *) class;

	/* GtkObject signals */
	object_class->destroy = kz_tab_label_destroy;

	kz_tab_label_color_init();
}

static void
kz_tab_label_init (KzTabLabel *kztab)
{
	kztab->kzembed      = NULL;
	kztab->label        = NULL;
	kztab->close_button = NULL;
}

static void
kz_tab_label_destroy (GtkObject *object)
{
	KzTabLabel *kztab;

	g_return_if_fail (KZ_IS_TAB_LABEL(object));

	kztab = KZ_TAB_LABEL(object);

	if (kztab->kzembed) {
		g_signal_handlers_disconnect_by_func(G_OBJECT(kztab->kzembed),
						     net_start_cb, kztab);
		g_signal_handlers_disconnect_by_func(G_OBJECT(kztab->kzembed),
						     net_stop_cb, kztab);
		g_signal_handlers_disconnect_by_func(G_OBJECT(kztab->kzembed),
						     title_changed_cb, kztab);
		g_signal_handlers_disconnect_by_func(G_OBJECT(kztab->kzembed),
						     destroy_cb, kztab);
	}
	kztab->kzembed = NULL;

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		GTK_OBJECT_CLASS (parent_class)->destroy(object);
}

GtkWidget *
kz_tab_label_new (KzWindow *kz, KzMozEmbed *kzembed)
{
	KzTabLabel *kztab;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkWidget *button, *image;

	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), NULL);
	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);

	kztab = KZ_TAB_LABEL(g_object_new(KZ_TYPE_TAB_LABEL, NULL));

	kztab->kz = kz;
	kztab->kzembed = kzembed;
	g_signal_connect(G_OBJECT(kzembed), "title",
			 G_CALLBACK(title_changed_cb), kztab);
	g_signal_connect(G_OBJECT(kzembed), "net-start",
			 G_CALLBACK(net_start_cb), kztab);
	g_signal_connect(G_OBJECT(kzembed), "net-stop",
			 G_CALLBACK(net_stop_cb), kztab);
	g_signal_connect(G_OBJECT(kzembed), "destroy",
			 G_CALLBACK(destroy_cb), kztab);

	/* label */
	kztab->label = gtk_label_new("no title");
	gtk_widget_set_size_request(kztab->label, 50, -1);
	gtk_widget_show (kztab->label);

	/* button */
	kztab->close_button = button = gtk_button_new();
	gtk_widget_set_size_request(button, 14, 14);
	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	g_signal_connect(G_OBJECT(button), "clicked", 
			 G_CALLBACK(kz_tab_label_close_cb), kztab);

	pixmap = gdk_pixmap_colormap_create_from_xpm_d(GTK_WIDGET(kz)->window,
						       gtk_widget_get_colormap(GTK_WIDGET(kztab)),
						       &mask,
						       NULL,
						       small_close);
	image = gtk_image_new_from_pixmap(pixmap,mask);
	gdk_pixmap_unref(pixmap);
	gdk_bitmap_unref(mask);
	gtk_container_add(GTK_CONTAINER(button), image);

	gtk_box_pack_start(GTK_BOX(kztab), kztab->label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(kztab), kztab->close_button, FALSE, FALSE, 0);

	gtk_widget_show(image);
	gtk_widget_show (button);

	return GTK_WIDGET (kztab);
}

void
kz_tab_label_show_close_button (KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	gtk_widget_show(kztab->close_button);
}

void
kz_tab_label_hide_close_button (KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	gtk_widget_hide(kztab->close_button);
}

void
kz_tab_label_set_text(KzTabLabel *kztab, const gchar *text)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	gtk_label_set_text(GTK_LABEL(kztab->label), text);
}

void
kz_tab_label_set_state (KzTabLabel *kztab,
			KzTabLabelState state)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	switch(state) {
	case KZ_TAB_LABEL_STATE_LOADING:
		gtk_widget_modify_style(GTK_WIDGET(kztab->label),
					kz_tab_label_style_loading);
		break;
	case KZ_TAB_LABEL_STATE_NORMAL:
	default:
		gtk_widget_modify_style(GTK_WIDGET(kztab->label),
					kz_tab_label_style_normal);
		break;
	}
}

KzTabLabelState
kz_tab_label_get_state (KzTabLabel *kztab)
{
	g_return_val_if_fail(KZ_IS_TAB_LABEL(kztab), KZ_TAB_LABEL_STATE_NORMAL);
	return kztab->state;
}

static void
kz_tab_label_color_init ()
{
	int i;
	
	/* new rc styles */
	kz_tab_label_style_loading = gtk_rc_style_new();
	kz_tab_label_style_normal  = gtk_rc_style_new();

	/* set colors */
	for(i = 0; i< 5; i++)
	{
		kz_tab_label_style_loading->color_flags[i]   = 1;
		kz_tab_label_style_loading->fg[i].red   = 0xffff;
		kz_tab_label_style_loading->fg[i].blue  = 0x0000;
		kz_tab_label_style_loading->fg[i].green = 0x0000;
		kz_tab_label_style_normal->color_flags[i]   = 1;
		kz_tab_label_style_normal->fg[i].red   = 0x0000;
		kz_tab_label_style_normal->fg[i].blue  = 0x0000;
		kz_tab_label_style_normal->fg[i].green = 0x0000;
	}
}


/* callbacks for mozilla */

static void
title_changed_cb (GtkMozEmbed *embed, KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_MOZ_EMBED(embed));
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	gtk_label_set_text(GTK_LABEL(kztab->label),
			   (gchar *) kz_moz_embed_get_title(kztab->kzembed));
}

static void
net_start_cb (GtkMozEmbed *embed, KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	kz_tab_label_set_state(kztab, KZ_TAB_LABEL_STATE_LOADING);
}

static void
net_stop_cb (GtkMozEmbed *embed, KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	kz_tab_label_set_state(kztab, KZ_TAB_LABEL_STATE_NORMAL);
}

static void
kz_tab_label_close_cb (GtkWidget *button, KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	kz_window_close_tab(kztab->kz, GTK_WIDGET(kztab->kzembed));
}

static void
destroy_cb (GtkObject *object, KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	kztab->kzembed = NULL;
}

