/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <unistd.h>
#include <pthread.h>
#include <libxml/nanohttp.h>
#include <gtkmozembed.h>

#include "config.h"
#include "kz-window.h"
#include "kz-mozembed.h"
#include "prefs.h"
#include "rss.h"
#include "gtk-utils.h"

static void     rss_file_fetch(RSS *rss);
static gboolean rss_item_update(RSS *rss);

static void     rss_item_menu_update(RSS *rss);
static void     rss_item_menu_button_press_cb(GtkWidget *widget,
					      GdkEventButton *event,
					      RSSItem *item);
static void     rss_item_menu_new(KzWindow *kz, RSS *rss);

static void     rss_button_press_cb(GtkWidget *widget, GdkEvent *event,
				    RSS *rss);

typedef enum {
	STATE_START   = (1 << 0),
	STATE_ROOT    = (1 << 1),
	STATE_CHANNEL = (1 << 2),
	STATE_ITEM    = (1 << 3),
	STATE_TITLE   = (1 << 4),
	STATE_LINK    = (1 << 5),
	STATE_DESC    = (1 << 6),
	STATE_END     = (1 << 7),
	STATE_ERROR   = (1 << 8)
} ParseState;

typedef struct _ParseContext ParseContext;
struct _ParseContext
{
	/* parser state information */
	ParseState state;
	ParseState current_state;

	RSS *rss;
};

static void
start_element_handler (GMarkupParseContext *context,
		       const gchar         *element_name,
		       const gchar        **attribute_names,
		       const gchar        **attribute_values,
		       gpointer             user_data,
		       GError             **error)
{
	ParseContext *ctx = user_data;

	ctx->current_state = STATE_ERROR;
	
	switch (element_name[0])
	{
	 case 'r':
		if (ctx->state == STATE_START && 
		    (!strcmp(element_name, "rdf:RDF") || 
		     !strcmp(element_name, "rss")))
		{
			ctx->state = STATE_ROOT;
			ctx->current_state = STATE_ROOT;
		}
		break;
	 case 'i':
		if ((ctx->state & STATE_ROOT) && !strcmp(element_name, "item") 
		    && strcmp(element_name, "items"))
		{
			ctx->state |= STATE_ITEM;
			ctx->current_state = STATE_ITEM;
			/* append new RSSItem */
			RSSItem *item = g_new0(RSSItem, 1);
			item->rss = ctx->rss;
			item->title = NULL;
			item->link = NULL;
			item->description = NULL;
			
			if (!ctx->rss->items)
				ctx->rss->items = g_slist_append(ctx->rss->items, item);
			else
				g_slist_append(ctx->rss->items,item);
		}
		break;
	 case 'c':
		if ((ctx->state & STATE_ROOT) && !strcmp(element_name, "channel"))
		{
			ctx->state |= STATE_CHANNEL;
			ctx->current_state = STATE_CHANNEL;
		}
		break;
	 case 't':
		if (!strcmp(element_name, "title"))
		{
			ctx->state |= STATE_TITLE;
			ctx->current_state = STATE_TITLE;
		}
		break;
	 case 'l':
		if (!strcmp(element_name, "link"))
		{
			ctx->state |= STATE_LINK;
			ctx->current_state = STATE_LINK;
		}
		break;
	 case 'd':
		if (!strcmp(element_name, "description"))
		{
			ctx->state |= STATE_DESC;
			ctx->current_state = STATE_DESC;
		}
		break;
	}
}

static void
end_element_handler (GMarkupParseContext *context,
		     const gchar         *element_name,
		     gpointer             user_data,
		     GError             **error)
{
	ParseContext *ctx = user_data;
	
	ctx->state = ctx->state & ~(ctx->current_state);
	ctx->current_state = STATE_ERROR;
}

static void 
text(GMarkupParseContext *context,
     const gchar          *text,
     gsize                 text_len,
     gpointer              user_data,
     GError              **error)
{
	ParseContext *ctx = user_data;
	
	if (ctx->current_state == STATE_ERROR)
		return;
	if (ctx->state & STATE_ITEM)
	{
		guint pos = g_slist_length(ctx->rss->items);
		RSSItem *item = g_slist_nth_data(ctx->rss->items, pos - 1);
		switch (ctx->current_state)
		{
		 case STATE_TITLE:
			item->title = g_strndup(text, text_len);
			break;
		 case STATE_LINK:
			item->link = g_strndup(text, text_len);
			break;
		 case STATE_DESC:
			item->description = g_strndup(text, text_len);
			break;
		}
			
	}
	else if (ctx->state & STATE_CHANNEL)
	{
		switch (ctx->current_state)
		{
		 case STATE_TITLE:
			if (ctx->rss->rss_title)
				g_free(ctx->rss->rss_title);
			ctx->rss->rss_title = g_strndup(text, text_len);
			break;
		 case STATE_LINK:
			if (ctx->rss->rss_link)
				g_free(ctx->rss->rss_link);
			ctx->rss->rss_link = g_strndup(text, text_len);
			break;
		 case STATE_DESC:
			if (ctx->rss->rss_description)
				g_free(ctx->rss->rss_description);
			ctx->rss->rss_description = g_strndup(text, text_len);
			break;
		}
	}
}

static GMarkupParser ui_parser = {
	start_element_handler,
	end_element_handler,
	text,
	NULL,
	NULL
};

static gboolean
rss_parse_from_string (RSS *rss, gpointer user_data,
		       const gchar *buffer, guint length,
		       GError **error)
{
	ParseContext ctx = { 0 };
	GMarkupParseContext *context;
	gboolean res = TRUE;

	g_return_val_if_fail(buffer != NULL, FALSE);

	ctx.state = STATE_START;
	ctx.current_state = STATE_START;
	ctx.rss = rss;

	context = g_markup_parse_context_new(&ui_parser, 0, &ctx, NULL);
	if (length < 0)
		length = strlen(buffer);

	if (g_markup_parse_context_parse(context, buffer, length, error))
	{
		if (!g_markup_parse_context_end_parse(context, error))
			res = FALSE;
	}
	else
		res = FALSE;

	g_markup_parse_context_free (context);

	return res;
}

static gboolean
rss_parse_from_file (RSS *rss, gpointer user_data,
		     const gchar *filename, GError **error)
{
	gchar *buffer;
	gint length;
	gboolean res;

	if (!g_file_get_contents (filename, &buffer, &length, error))
		return FALSE;

	res = rss_parse_from_string(rss, user_data, buffer, length, error);
	g_free(buffer);

	return res;
}

static void
rss_file_fetch(RSS *rss)
{
	gchar *contentType;

	xmlNanoHTTPScanProxy(prefs.proxy);
	if (g_file_test(rss->localfile, G_FILE_TEST_EXISTS))
		remove (rss->localfile);
	xmlNanoHTTPFetch(rss->uri, rss->localfile, &contentType);
}

static void
rss_item_menu_new(KzWindow *kz, RSS *rss)
{
	GtkWidget *menu_item;
	GSList *items;
	guint item_num, n;
	RSSItem *item = NULL;
	GtkTooltips *tooltip = NULL;
	
	if (rss->SubMenu != NULL)
	{
		gtk_widget_destroy(rss->SubMenu);
		gtk_menu_item_remove_submenu(GTK_MENU_ITEM(rss->MenuItem));
	}

	rss->SubMenu = gtk_menu_new();
	
	items = rss->items;
	item_num = g_slist_length(rss->items);

	for (n = 0; n < item_num; n++)
	{
		item = (RSSItem*) g_slist_nth_data(items, n);

		menu_item = gtk_menu_item_new_with_label(item->title);
		gtk_menu_shell_append(GTK_MENU_SHELL(rss->SubMenu),
				     menu_item);
		item->rss = rss;
		g_signal_connect(G_OBJECT(menu_item), "button_press_event",
				 G_CALLBACK(rss_item_menu_button_press_cb), item);
		tooltip = gtk_tooltips_new();
		gtk_tooltips_set_tip(tooltip, menu_item, item->description, NULL);
	}
	gtk_menu_item_set_submenu (GTK_MENU_ITEM(rss->MenuItem), rss->SubMenu);
	gtk_widget_show_all(rss->SubMenu);
}

static void 
rss_item_menu_update(RSS *rss)
{
	KzWindow *kz = rss->kz;
	rss_file_fetch(rss);
	if (rss->items)
	{
		g_slist_free(rss->items);
		rss->items = NULL;
	}
	rss_parse_from_file(rss, NULL, rss->localfile, NULL);
	rss_item_menu_new(kz, rss);
	gtk_widget_show_all(rss->MenuItem);
}

static gboolean
rss_item_update(RSS *rss)
{
	pthread_t t;
	pthread_create(&t, NULL, (void *)rss_item_menu_update, (void *)rss);
	pthread_detach(t);
	return TRUE;
}

static void
rss_item_menu_button_press_cb(GtkWidget *widget, GdkEventButton *event,
			      RSSItem *item)
{
	KzWindow *kz;
	kz = item->rss->kz;
	
	if (!item->link) return;

	switch (event->button) {
	case 1:
	case 3:
		kz_window_load_url(kz, item->link);
		break;
	case 2:
		kz_window_open_new_tab(kz, item->link);
		break;
	}
}

/* create a new rss menu */
GtkWidget *
rss_menu_new(KzWindow *kz, gchar *uri, gchar *title, guint update_interval)
{
	RSS *rss=NULL;	
	gchar *temp, *pos;

	rss = g_new0(RSS,1);
	rss->MenuItem        = NULL;
	rss->SubMenu         = NULL;
	rss->items           = NULL;
	rss->rss_title       = NULL;
	rss->rss_link        = NULL;
	rss->rss_description = NULL;
	rss->rss_date        = NULL;
	
	rss->kz              = kz;
	rss->uri             = g_strdup(uri);
	rss->title           = g_strdup(title);
	
	temp = g_strdup(rss->uri);
	/* I'm not sure this robustness. */
	while ((pos = strstr (temp, "/")) != NULL) *pos = '_';
	while ((pos = strstr (temp, "&")) != NULL) *pos = '_';
	while ((pos = strstr (temp, "?")) != NULL) *pos = '_';
	while ((pos = strstr (temp, "|")) != NULL) *pos = '_';
	temp = g_strdelimit(temp, NULL, '_');
	rss->localfile = g_strdup_printf("%s/.%s/rss/%s", g_get_home_dir(), PACKAGE, temp);
	g_free(temp);

	rss->MenuItem = gtk_menu_item_new_with_label(rss->title);

	g_signal_connect(G_OBJECT(rss->MenuItem), "button_press_event",
			 G_CALLBACK(rss_button_press_cb), rss);

	rss_item_update(rss);
	if (update_interval != 0)
	{
		rss->update_interval = update_interval;
		gtk_timeout_add(60000 * rss->update_interval, (void *)&rss_item_update, rss);
	}
	
	return (rss->MenuItem);
}

static void
rss_button_press_cb(GtkWidget *widget, GdkEvent *event,
		    RSS *rss)
{
	KzWindow *kz;
	GdkEventButton *event_button;
	GSList *items;
	guint item_num, n;
	RSSItem *item = NULL;
	
	kz = rss->kz;
	
	if (event->type == GDK_BUTTON_PRESS) 
	{
		event_button = (GdkEventButton *)event;
		switch(event_button->button)
		{
		 case 1:
			gtk_menu_popup(GTK_MENU(rss->SubMenu), NULL, NULL, 
				       menu_position_under_widget,
				       widget,
				       event_button->button, event_button->time);
			break;
		 case 2:
			items = rss->items;
			item_num = g_slist_length(rss->items);
			for (n = 0; n < item_num; n++)
			{
				item = (RSSItem*) g_slist_nth_data(items, n);
				if (item->link)
					kz_window_open_new_tab(kz, item->link);
			}
			break;
		 case 3:
			kz_window_load_url(kz, rss->rss_link);
			break;
		}
	}
}
