/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-link.h"

#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <regex.h>

#include "kazehakase.h"
#include "kz-profile.h"
#include "utils.h"

#ifndef REG_NOERROR
#define REG_NOERROR (0)
#endif

static void kz_link_finalize (KzLink *kzlink);


KzLink *
kz_link_new (void)
{
	KzLink *kzlink = g_new0(KzLink, 1);
	kzlink->title     = NULL;
	kzlink->url       = NULL;
	kzlink->ref_count = 1;

	return kzlink;
}


KzLink *
kz_link_ref (KzLink *kzlink)
{
	g_return_val_if_fail(kzlink, NULL);

	kzlink->ref_count++;

	return kzlink;
}


void
kz_link_unref (KzLink *kzlink)
{
	g_return_if_fail(kzlink);

	kzlink->ref_count--;

	if (kzlink->ref_count < 1)
		kz_link_finalize(kzlink);
}


static void
kz_link_finalize (KzLink *kzlink)
{
	g_return_if_fail(kzlink);

	g_free(kzlink->title);
	g_free(kzlink->url);
	g_free(kzlink);
}


typedef enum {
	KEY_NONE,
	KEY_REGEX_HREF,
	KEY_REGEX_ELEMENT
} FilterType;


FilterType
detect_filter_type (const gchar *key)
{
	const gchar *known_keys[] = {
		NULL,
		"filter_href_regex",
		"filter_element_regex"
	};
	gint i;

	g_return_val_if_fail(key && *key, KEY_NONE);

	for (i = 1; i < G_N_ELEMENTS(known_keys); i++)
	{
		if (key_seems_sequential (key, known_keys[i]))
			return i;
	}

	return KEY_NONE;
}


gboolean
kz_link_filter (const KzLink *kzlink)
{
	regex_t preg;
	regmatch_t mat;
	int ret;
	GList *list, *node;

	g_return_val_if_fail(kzlink, FALSE);

	list = kz_profile_enum_key(kz_global_profile, "ExtractLinks");

	for (node = list; node; node = g_list_next(node))
	{
		const gchar *key = node->data;
		const gchar *str = NULL;
		gchar *exp;
		FilterType key_type;

		if (!key || !*key) continue;

		key_type = detect_filter_type(key);

		switch (key_type) {
		case KEY_REGEX_HREF:
			str = kzlink->url;
			break;
		case KEY_REGEX_ELEMENT:
			str = kzlink->title;
			break;
		default:
			break;
		}

		if (!str || !*str) continue;

		exp = kz_profile_get_string(kz_global_profile,
					    "ExtractLinks", key);

		ret = regcomp(&preg, exp, 0);
		if (ret != REG_NOERROR) continue;

		ret = regexec(&preg, kzlink->url, 1, &mat, 0);
		if (ret == 0)
		{
			g_free(exp);
			regfree(&preg);
			return TRUE;
		}

		g_free(exp);
		regfree(&preg);
	}

	return FALSE;
}
