/*
 * The MIT License
 *
 * Copyright 2013 Dra0211.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package kinugasa.contents.graphics;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.ConvolveOp;
import java.awt.image.Kernel;
import java.awt.image.RasterFormatException;
import java.util.Arrays;
import static kinugasa.contents.graphics.ImageUtil.*;

/**
 * BufferedImageɑ΂鍂xȕҏW@\񋟂郆[eBeBNXł.
 * <br>
 * ̃NXł́Au\[X摜ṽsNZf[^AڕύX\̂@\`Ă܂B<br>
 * ̃NXɒ`ꂽ\bh́AQ[ɂĎgpׂł͂܂B\[XƂďقptH[}Xサ܂B<br>
 * <br>
 *
 * @version 1.0.0 - 2013/01/13_2:18:18<br>
 * @author Dra0211<br>
 */
public final class ImageEditor {

	/**
	 * [eBeBNX̂߃CX^Xł܂.
	 */
	private ImageEditor() {
	}

	/**
	 * FtgtARGBnewARGBŒuV摜dstɊi[ĕԂ܂.
	 *
	 * @param src Fu\[X摜w肵܂B̉摜̃sNZ͕ύX܂B<br>
	 * @param tgtARGB src̒ȗΏۂƂȂFARGBJ[Ŏw肵܂B<br>
	 * @param newARGB u̐FARGBJ[Ŏw肵܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return srctgtARGB̃sNZnewARGBɒu摜Ԃ܂B<br>
	 */
	public static BufferedImage replaceColor(BufferedImage src, int tgtARGB, int newARGB, BufferedImage dst) {
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[] pix = getPixel(src);
		for (int i = 0; i < pix.length; i++) {
			if (pix[i] == tgtARGB) {
				pix[i] = newARGB;
			}
		}
		setPixel(dst, pix);
		return dst;
	}

	/**
	 * srcOCXP[ϊ摜dstɊi[ĕԂ܂. ̃\bh̓sNZRGBς̂܂ܐݒ肵܂B NTSCndϖ@͎gp܂B<br>
	 *
	 * @param src Fu\[X摜w肵܂B̉摜̃sNZ͕ύX܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return src̃sNZ𖾈Âɒu摜Ԃ܂B<br>
	 */
	public static BufferedImage grayScale(BufferedImage src, BufferedImage dst) {
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[] pix = getPixel(src);
		for (int i = 0; i < pix.length; i++) {
			int average = ARGBColor.getRGBAverage(pix[i]);
			pix[i] = (pix[i] & ARGBColor.ARGB_ALPHA_MASK)
					| (average << 16)
					| (average << 8)
					| average;
		}
		setPixel(dst, pix);
		return dst;
	}

	/**
	 * NTSCdϖ@KpOCXP[ϊs܂.
	 *
	 * @param src Fu\[X摜w肵܂B̉摜̃sNZ͕ύX܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return src̃sNZ𖾈Âɒu摜Ԃ܂B<br>
	 */
	public static BufferedImage weightedGrayScale(BufferedImage src, BufferedImage dst) {
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[] pix = getPixel(src);
		for (int i = 0; i < pix.length; i++) {
			int brightness = (int) (ARGBColor.getRed(pix[i]) * 0.298912f)
					+ (int) (ARGBColor.getGreen(pix[i]) * 0.586611f)
					+ (int) (ARGBColor.getBlue(pix[i]) * 0.114478f);
			pix[i] = pix[i] & ARGBColor.ARGB_ALPHA_MASK
					| (brightness << 16)
					| (brightness << 8)
					| brightness;
		}
		setPixel(dst, pix);

		return dst;
	}

	/**
	 * src𔒍mN[ϊ摜dstɊi[ĕԂ܂.
	 *
	 * @param src Fu\[X摜w肵܂B̉摜̃sNZ͕ύX܂B<br>
	 * @param center ƂȂ閾x0255Ŏw肵܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return src̖xcenter𒴂sNZ𔒂ɁAłȂsNZɒu摜Ԃ܂B<br>
	 *
	 * @throws IllegalArgumentException center0܂255𒴂ꍇɓ܂B<br>
	 */
	public static BufferedImage monochrome(BufferedImage src, int center, BufferedImage dst)
			throws IllegalArgumentException {
		if (center < 0 || center > 255) {
			throw new IllegalArgumentException("center is over color range : center=[" + center + "]");
		}
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[] pix = getPixel(src);
		for (int i = 0; i < pix.length; i++) {
			pix[i] = (ARGBColor.getRGBAverage(pix[i]) > center)
					? (pix[i] & ARGBColor.ARGB_ALPHA_MASK)
					| 0x00FFFFFF
					: (pix[i] & ARGBColor.ARGB_ALPHA_MASK);
		}
		setPixel(dst, pix);
		return dst;
	}

	/**
	 * src̃sNZ̖x(right*100)%ɕϊ摜dstɊi[ĕԂ܂.
	 *
	 * @param src Fu\[X摜w肵܂B̉摜̃sNZ͕ύX܂B<br>
	 * @param right x̕ύX銄w肵܂B0w肷邱Ƃ͂ł܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return src̃sNZ̖x(right*100)%ɕϊ摜Ԃ܂B<br>
	 *
	 * @throws IllegalArgumentException right0̂Ƃɓ܂B<br>
	 */
	public static BufferedImage brightness(BufferedImage src, float right, BufferedImage dst)
			throws IllegalArgumentException {
		if (right == 1f) {
			return dst = copy(src);
		}
		if (right < 0) {
			throw new IllegalArgumentException("right < 0 : right=[" + right + "]");
		}
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[] pix = getPixel(src);
		int a, r, g, b;
		for (int i = 0; i < pix.length; i++) {
			a = ARGBColor.getAlpha(pix[i]);
			r = (int) (ARGBColor.getRed(pix[i]) * right);
			if (r > 255) {
				r = 255;
			} else if (r < 0) {
				r = 0;
			}
			g = (int) (ARGBColor.getGreen(pix[i]) * right);
			if (g > 255) {
				g = 255;
			} else if (g < 0) {
				g = 0;
			}
			b = (int) (ARGBColor.getBlue(pix[i]) * right);
			if (b > 255) {
				b = 255;
			} else if (b < 0) {
				b = 0;
			}
			pix[i] = ARGBColor.toARGB(a, r, g, b);
		}
		setPixel(dst, pix);
		return dst;
	}

	/**
	 * src̐F𔽓]摜dstɊi[ĕԂ܂.
	 *
	 * @param src Fu\[X摜w肵܂B̉摜̃sNZ͕ύX܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return src̃sNZ̐F𔽓]摜Ԃ܂B<br>
	 */
	public static BufferedImage reverseColor(BufferedImage src, BufferedImage dst) {
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[] pix = getPixel(src);
		int a, r, g, b;
		for (int i = 0; i < pix.length; i++) {
			a = pix[i] & ARGBColor.ARGB_ALPHA_MASK;
			r = (255 - ARGBColor.getRed(pix[i])) << 16;
			g = (255 - ARGBColor.getGreen(pix[i])) << 8;
			b = ARGBColor.getBlue(pix[i]);
			pix[i] = ARGBColor.toARGB(a, r, g, b);
		}
		setPixel(dst, pix);
		return dst;
	}

	/**
	 * srcɊȈՓIȃUCN{摜dstɊi[ĕԂ܂.
	 *
	 * @param src Fu\[X摜w肵܂B̉摜̃sNZ͕ύX܂B<br>
	 * @param size UCÑ^C̃TCYsNZPʂŎw肵܂B1w肷邱Ƃ͂ł܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return srcsizesNZƂɋ؂̗̈̈̂Ƃ̃sNZ̐FœhԂ摜Ԃ܂B<br>
	 *
	 * @throws IllegalArgumentException size1̏ꍇɓ܂B<br>
	 * @throws RasterFormatException size摜傫ꍇɓ܂B<br>
	 */
	public static BufferedImage mosaic(BufferedImage src, int size, BufferedImage dst)
			throws IllegalArgumentException, RasterFormatException {
		if (size < 1) {
			throw new IllegalArgumentException("size < 1 : size=[" + size + "]");
		}
		if (size > src.getWidth() || size > src.getHeight()) {
			throw new RasterFormatException("size is over image bounds : size=[" + size + "]");
		}
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[][] pix = getPixel2D(src);
		for (int y = 0, imageHeight = src.getHeight(); y + size < imageHeight; y += size) {
			for (int x = 0, imageWidth = src.getWidth(); x + size < imageWidth; x += size) {
				int argb = pix[y][x];
				for (int mosaicY = y, i = 0; i < size; mosaicY++, i++) {
					for (int mosaicX = x, j = 0; j < size; mosaicX++, j++) {
						if (mosaicX <= imageWidth && mosaicY <= imageHeight) {
							pix[mosaicY][mosaicX] = argb;
						}
					}
				}
			}
		}
		setPixel2D(dst, pix);
		return dst;
	}

	/**
	 * srcvdegx]摜dstɊi[ĕԂ܂.
	 *
	 * @param src Fu\[X摜w肵܂B̉摜̃sNZ͕ύX܂B<br>
	 * @param deg ]pxx@Ŏw肵܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return src]摜Ԃ܂Bdeg1̂Ƃsrc̃Rs[Ԃ܂B]ɉ摜̈ OɏoꍇA󂢂͍̈(Color.BLACK)ƂȂ܂B<br>
	 */
	public static BufferedImage rotate(BufferedImage src, float deg, BufferedImage dst) {
		if (deg == 0) {
			return dst = copy(src);
		}
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		Graphics2D g = createGraphics2D(dst, RenderingPolicy.QUALITY);
		g.setClip(0, 0, dst.getWidth(), dst.getHeight());
		g.setColor(Color.BLACK);
		g.fillRect(0, 0, dst.getWidth(), dst.getHeight());
		g.rotate(Math.toRadians(deg), dst.getWidth() / 2, dst.getHeight() / 2);
		g.drawImage(src, 0, 0, null);
		g.dispose();
		return dst;
	}

	/**
	 * \[X摜̑SẴsNZ̓ߓxtpZ摜Ԃ܂. ̃\bhł̓sNZ̓ߓx0255𒴂ꍇ͗L͈͓ɐ؂l߂܂B<br>
	 * tp0̏ꍇAߓxύXKvȂ߁APsrc̃Rs[dstɊi[ĕԂ܂B<br>
	 *
	 * @param src ߓxύX\[X摜w肵܂B̉摜̃sNZf[^͑삳܂B<br>
	 * @param tp Z铧ߓxw肵܂Be܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return src̑SẴsNZ̓ߓxtpZdstɊi[ĉ摜Ԃ܂B<br>
	 */
	public static BufferedImage addTransparent(BufferedImage src, int tp, BufferedImage dst) {
		if (tp == 0) {
			return dst = copy(src);
		}
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[] pix = getPixel(src);
		for (int i = 0; i < pix.length; i++) {
			int alpha = ARGBColor.getAlpha(pix[i]);
			alpha += tp;
			if (alpha < 0) {
				alpha = 0;
			} else if (alpha > 255) {
				alpha = 255;
			}
			pix[i] = (alpha << 24) & ARGBColor.ARGB_ALPHA_MASK | pix[i] & ARGBColor.CLEAR_WHITE;
		}
		setPixel(dst, pix);
		return dst;
	}

	/**
	 * 摜̊esNZw肳ꂽTCYɃVtgč\z摜Ԃ܂B<br>
	 *
	 * @param src 摜B<br>
	 * @param dst nullłȂꍇ̃CX^XɕҏWʂi[B<br>
	 * @param shiftPixNum c̊esNZ̃Vtg.src̍ɖȂꍇ͂𖞂܂ŌJԂ.wł<br>
	 * @param insertARGB Vtg,󂢂̈ɑ}FARBG`Ŏw肷B<br>
	 *
	 * @return nullłȂꍇÄɌʂi[܂B<br>
	 */
	public static BufferedImage rasterScroll(BufferedImage src, BufferedImage dst, int[] shiftPixNum, int insertARGB) {
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[] sPix = shiftPixNum;
		if (shiftPixNum.length != dst.getHeight()) {
			sPix = new int[dst.getHeight()];
			for (int i = 0, spi = 0; i < sPix.length; i++) {
				sPix[i] = shiftPixNum[spi];
				spi = (spi < shiftPixNum.length - 1) ? spi + 1 : 0;
			}
		}
		int[][] pix = getPixel2D(dst);
		for (int y = 0; y < pix.length; y++) {
			if (sPix[y] == 0) {
				continue;
			}
			final int[] ROW = new int[pix[y].length];
			System.arraycopy(pix[y], 0, ROW, 0, ROW.length);
			int x = 0, lineIdx = 0;
			if (sPix[y] > 0) {
				Arrays.fill(pix[y], 0, sPix[y], insertARGB);
				x += sPix[y];
			} else {
				lineIdx += Math.abs(sPix[y]);
			}
			System.arraycopy(ROW, lineIdx, pix[y], x, ROW.length - Math.abs(sPix[y]));
			x += ROW.length - Math.abs(sPix[y]);
			Arrays.fill(pix[y], x, pix[y].length, insertARGB);
		}
		setPixel2D(dst, pix);
		return dst;
	}

	/**
	 * ѐɃu[GtFNg摜Ԃ܂.
	 *
	 * @param src ʂ\[X摜łB̉摜̃sNZf[^͑삳܂B<br>
	 * @param width ̃u[͈͂𑗐M܂B1ȏ̒l𑗐MłA傫ȒlقǕsNȌʂɂȂ܂B<br>
	 * @param height ̃u[͈͂w肵܂B1ȏ̒l𑗐MłA傫ȒlقǕsNȌʂɂȂ܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return ѐɃu[GtFNg摜dstɊi[ĕԂ܂B<br>
	 */
	public static BufferedImage blur2D(BufferedImage src, int width, int height, BufferedImage dst) {
		return blur(blur(src, width, false, dst), height, true, null);
	}

	/**
	 * 摜̐͐Ƀu[ʂ܂.
	 *
	 * @param src ʂ\[X摜łB̉摜̃sNZf[^͑삳܂B<br>
	 * @param rad 1̃sNZɑ΂Au[̌ʔ͈͂w肵܂B1ȏ̒l𑗐MłA傫ȒlقǕsNȌʂɂȂ܂B<br>
	 * @param hrz truêƂu[̕ɂȂ܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return ͐Ƀu[GtFNg摜dstɊi[ĕԂ܂B<br>
	 *
	 * @throws IllegalArgumentException rad1̏ꍇɓ܂B<br>
	 */
	public static BufferedImage blur(BufferedImage src, int rad, boolean hrz, BufferedImage dst)
			throws IllegalArgumentException {
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		if (rad < 1) {
			throw new IllegalArgumentException("radius < 1 :  rad=[" + rad + "]");
		}

		int size = rad * 2 + 1;
		float[] data = new float[size];
		float sigma = rad * 3f;
		float twoSigmaSq = 2f * sigma * sigma;
		float sigmaRoot = (float) Math.sqrt(twoSigmaSq * Math.PI);
		float total = 0f;

		for (int i = -rad; i < rad; i++) {
			float dist = i * i;
			int index = i + rad;
			data[index] = (float) Math.exp(-dist / twoSigmaSq) / sigmaRoot;
			total += data[index];
		}
		for (int i = 0; i < data.length; i++) {
			data[i] /= total;
		}
		Kernel kernel = hrz ? new Kernel(size, 1, data) : new Kernel(1, size, data);
		ConvolveOp op = new ConvolveOp(kernel, ConvolveOp.EDGE_NO_OP, null);
		return op.filter(src, dst);
	}

	/**
	 * \[X摜̓ߓxύX摜Ԃ܂. ̃\bh͉摜̑SẴsNZ̃At@lψtp*100%ɕύX܂B<br>
	 * \[X摜̊SɓȃsNZ͂̂܂܊SɓȃsNZƂăRs[܂B<br>
	 *
	 * @param src ߓxύX\[X摜w肵܂B̉摜̃sNZf[^͑삳܂B<br>
	 * @param tp ύX̓ߓx̌Ww肵܂B0.0f1.0f̒lwł܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 *
	 * @return src̊SɓłȂsNZ̓ߓxtp*100%ɕύX摜dstɊi[ĕԂ܂B<br>
	 *
	 * @throws IllegalArgumentException tp1𒴂ꍇ0̏ꍇɓ܂B<br>
	 */
	public static BufferedImage transparent(
			BufferedImage src, float tp, BufferedImage dst)
			throws IllegalArgumentException {
		if (tp > 1f || tp < 0f) {
			throw new IllegalArgumentException("ߒlł [tp > 1 || tp < 0]łKv܂ tp=[" + tp + "]");
		}
		if (dst == null || dst == src) {
			dst = copy(src);
		}
		int[] pix = getPixel(src);
		int alpha = (int) (255 * tp) << 24;
		for (int i = 0; i < pix.length; i++) {
			pix[i] = (pix[i] & ARGBColor.ARGB_ALPHA_MASK) != 0x00000000
					? alpha | (pix[i] & ARGBColor.CLEAR_WHITE)
					: pix[i] & ARGBColor.CLEAR_WHITE;
		}
		setPixel(dst, pix);
		return dst;
	}

	/**
	 * w肳ꂽ摜g^kV摜Ԃ܂B<br>
	 *
	 * @param src \[X摜B<br>
	 * @param scale gXP[.1.0fw肵ꍇ́Asrc̃Rs[ԂB<br>
	 *
	 * @return \[X摜̃XP[OʂԂ.XP[OW1.0f̂Ƃ̓\[X摜̃Rs[ԂB<br>
	 */
	public static BufferedImage resize(BufferedImage src, float scale) {
		if (Float.compare(scale, 1.0f) == 0) {
			return copy(src);
		}
		int newWidth = (int) (src.getWidth() * scale);
		int newHeight = (int) (src.getHeight() * scale);
		BufferedImage dst = newImage(newWidth, newHeight);
		Graphics2D g2 = createGraphics2D(dst, RenderingPolicy.SPEED);
		g2.drawImage(src, 0, 0, newWidth, newHeight, null);
		g2.dispose();
		return dst;
	}

	public static BufferedImage[] resizeAll(BufferedImage[] images, float scale) {
		BufferedImage[] result = new BufferedImage[images.length];
		for (int i = 0; i < result.length; i++) {
			if (images[i] == null) {
				continue;
			}
			result[i] = resize(images[i], scale);
		}
		return result;
	}
}
