/*
 * The MIT License
 *
 * Copyright 2013 Dra0211.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package kinugasa.contents.graphics;

import java.awt.AWTException;
import java.awt.Graphics2D;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsEnvironment;
import java.awt.Rectangle;
import java.awt.Robot;
import java.awt.Transparency;
import java.awt.image.BufferedImage;
import java.awt.image.RasterFormatException;
import java.io.File;
import java.io.IOException;
import java.lang.ref.SoftReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import javax.imageio.ImageIO;
import kinugasa.contents.resource.ContentsFileNotFoundException;
import kinugasa.contents.resource.ContentsIOException;
import kinugasa.game.GameLog;
import kinugasa.util.StopWatch;

/**
 * 摜IOȈՕҏWs[eBeBNXł.
 * <br>
 * ̃NX烍[h摜́Aʏ̕@Ń[hꂽ摜 ɕ`ł\܂B ܂ÃNX̃[h@\́At@CpXw肷 摜CX^XԂ܂B<br>
 * <br>
 *
 * @version 1.0.0 - 2013/01/13_2:08:33<br>
 * @version 1.1.0 - 2013/04/28_23:16<br>
 * @author Dra0211<br>
 */
public final class ImageUtil {

	/**
	 * ftHg̃EChEVXeT|[g摜̐@\AOtBbNX̐ݒł.
	 */
	private static final GraphicsConfiguration gc
			= GraphicsEnvironment.getLocalGraphicsEnvironment().
			getDefaultScreenDevice().getDefaultConfiguration();
	/**
	 * [h摜LbV邽߂̃}bvł.
	 */
	private static final HashMap<String, SoftReference<BufferedImage>> IMAGE_CACHE
			= new HashMap<String, SoftReference<BufferedImage>>(32);

	/**
	 * CXN[̃foCXݒ擾܂B<br>
	 *
	 * @return foCX̐ݒB̃CX^X摜쐬ł܂B<br>
	 */
	public static GraphicsConfiguration getGraphicsConfiguration() {
		return gc;
	}

	/**
	 * [eBeBNX̂߃CX^Xł܂.
	 */
	private ImageUtil() {
	}

	//------------------------------------------------------------------------------------------------------------
	/**
	 * VBufferedImage𐶐܂. 쐬ꂽ摜͑SẴsNZSɓȍ(0x00000000)łB<br>
	 *
	 * @param width 摜̕sNZPʂŎw肵܂B<br>
	 * @param height 摜̍sNZPʂŎw肵܂B<br>
	 *
	 * @return BufferedImage̐VCX^XԂ܂B<br>
	 */
	public static BufferedImage newImage(int width, int height) {
		return gc.createCompatibleImage(width, height, Transparency.TRANSLUCENT);
	}

	/**
	 * BufferedImage̕VCX^XƂĕԂ܂.
	 *
	 * @param src Rs[摜B<br>
	 *
	 * @return srcƓ摜̐VCX^XԂ܂B<br>
	 */
	public static BufferedImage copy(BufferedImage src) {
		return copy(src, (BufferedImage) null);
	}

	/**
	 * BufferedImage̕쐬AdstɊi[܂.
	 *
	 * @param src Rs[摜B<br>
	 * @param dst nullłȂꍇ̃CX^XɌʂi[B<br>
	 *
	 * @return nullłȂꍇÄɌʂi[܂B<br>
	 */
	public static BufferedImage copy(BufferedImage src, BufferedImage dst) {
		if (dst == null || dst == src) {
			dst = newImage(src.getWidth(), src.getHeight());
		}
		Graphics2D g2 = dst.createGraphics();
		g2.setRenderingHints(RenderingPolicy.QUALITY.getRenderingHints());
		g2.drawImage(src, 0, 0, null);
		g2.dispose();
		return dst;
	}

	/**
	 * BufferedImaget@C쐬܂. ̃\bh͂łɈxvꂽ摜ēxvꍇACX^XԂ܂B<br>
	 * mɕʂ̃CX^X擾ꍇ͂̃\bh̖߂lɑ΂ẴNXcopy\bhgpĂB<br>
	 *
	 * @param filePath ǂݍރt@CpXB<br>
	 *
	 * @return ǂݍ܂ꂽ摜.łɈxǂݍ܂Ăꍇ̓LbVf[^̓摜CX^XԂB<br>
	 *
	 * @throws ContentsFileNotFoundException t@C݂ȂꍇɓB<br>
	 * @throws ContentsIOException t@C[hłȂꍇɓ܂B<br>
	 */
	public static BufferedImage load(String filePath) throws ContentsFileNotFoundException, ContentsIOException {
		StopWatch watch = new StopWatch().start();
		SoftReference<BufferedImage> cacheRef = IMAGE_CACHE.get(filePath);
		//LbV&GCs
		if (cacheRef != null && cacheRef.get() != null) {
			watch.stop();
			GameLog.printInfoIfUsing("ImageUtil LbVĂ܂ filePath=[" + filePath + "](" + watch.getTime() + " ms)");
			return cacheRef.get();
		}
		//GCsĂ邩LbVȂΐV[hăLbVɒǉ
		File file = new File(filePath);
		if (!file.exists()) {
			watch.stop();
			throw new ContentsFileNotFoundException("t@C݂܂ : filePath=[" + filePath + "](" + watch.getTime() + " ms)");
		}
		BufferedImage dst = null;
		try {
			dst = ImageIO.read(file);
		} catch (IOException ex) {
			watch.stop();
			GameLog.print(Level.WARNING, "摜[hł܂ filePath=[" + filePath + "](" + watch.getTime() + " ms)");
			throw new ContentsIOException(ex);
		}
		if (dst == null) {
			watch.stop();
			GameLog.print(Level.WARNING, "摜nullł filePath=[" + filePath + "](" + watch.getTime() + " ms)");
			throw new ContentsIOException("image is null");
		}
		//݊摜ɒu
		dst = copy(dst, newImage(dst.getWidth(), dst.getHeight()));
		IMAGE_CACHE.put(filePath, new SoftReference<BufferedImage>(dst));
		watch.stop();
		GameLog.printInfoIfUsing("ImageUtil [h܂ filePath=[" + filePath + "](" + watch.getTime() + " ms)");
		return dst;
	}

	/**
	 * BufferedImaget@Cɕۑ܂. 摜`͓PNG摜ƂȂ܂B<br>
	 *
	 * @param filePath ރt@CpX.㏑͊mFꂸAgqCӁB<br>
	 * @param image މ摜B<br>
	 *
	 * @throws ContentsIOException t@C߂ȂꍇɓB<br>
	 */
	public static void save(String filePath, BufferedImage image) throws ContentsIOException {
		File file = new File(filePath);
		try {
			ImageIO.write(image, "PNG", file);
		} catch (IOException ex) {
			throw new ContentsIOException(ex);
		}
	}

	/**
	 * BufferedImagẽsNZf[^zƂĎ擾܂.
	 *
	 * @param image sNZf[^擾摜𑗐M܂B<br>
	 *
	 * @return w肳ꂽ摜̃sNZf[^ꎟzƂĕԂ܂B ̔z͉摜ɐݒ肳ĂsNZ̃N[łB<br>
	 */
	public static int[] getPixel(BufferedImage image) {
		return image.getRGB(0, 0, image.getWidth(), image.getHeight(), null, 0, image.getWidth());
	}

	/**
	 * BufferedImagẽsNZf[^񎟌zƂĎ擾܂.
	 *
	 * @param image sNZf[^擾摜𑗐M܂B<br>
	 *
	 * @return w肳ꂽ摜̃sNZf[^񎟌zƂĕԂ܂B ̔z͉摜ɐݒ肳ĂsNZ̃N[łB<br>
	 */
	public static int[][] getPixel2D(BufferedImage image) {
		int[] pix = getPixel(image);
		int[][] pix2 = new int[image.getHeight()][image.getWidth()];
		for (int i = 0, row = 0, WIDTH = image.getWidth(); i < pix2.length; i++, row += WIDTH) {
			System.arraycopy(pix, row, pix2[i], 0, WIDTH);
		}
		return pix2;
	}

	/**
	 * BufferedImageɃsNZf[^ݒ肵܂. ̃\bh̓sNZƉ摜̎ۂ̃sNZقȂꍇ̓͒`Ă܂B<br>
	 *
	 * @param image sNZf[^ݒ肷摜B<br>
	 * @param pix ݒ肷sNZf[^B<br>
	 */
	public static void setPixel(BufferedImage image, int[] pix) {
		image.setRGB(0, 0, image.getWidth(), image.getHeight(), pix, 0, image.getWidth());
	}

	/**
	 * BufferedImageɃsNZf[^ݒ肵܂. ̃\bh̓sNZƉ摜̎ۂ̃sNZقȂꍇ̓͒`Ă܂B<br>
	 *
	 * @param image 摜B<br>
	 * @param pix ݒ肷sNZf[^B<br>
	 */
	public static void setPixel2D(BufferedImage image, int[][] pix) {
		int[] newPix = new int[getPixel(image).length];
		for (int i = 0; i < pix.length; i++) {
			System.arraycopy(pix[i], 0, newPix, i * pix[0].length, pix[i].length);
		}
		image.setRGB(0, 0, image.getWidth(), image.getHeight(), newPix, 0, image.getWidth());
	}

	public static int getPixel(BufferedImage image, int x, int y) {
		return getPixel2D(image)[y][x];
	}

	/**
	 * 摜ɏނ߂̃OtBNXReLXg쐬܂.
	 *
	 * @param image OtBbNXReLXg擾摜w肵܂B <br>
	 * @param renderingPolicy nullłȂꍇÃ_Oݒ肪OtBbNXReLXgɓKp܂B<br>
	 *
	 * @return w肵摜ɏނ߂̃OtBbNXReLXg쐬ĕԂ܂B<br>
	 */
	public static Graphics2D createGraphics2D(BufferedImage image, RenderingPolicy renderingPolicy) {
		Graphics2D g = image.createGraphics();
		if (renderingPolicy != null) {
			g.setRenderingHints(renderingPolicy.getRenderingHints());
		}
		g.setClip(0, 0, image.getWidth(), image.getHeight());
		return g;
	}

	/**
	 * BudderdImage0, y w, h̃TCYŉɉ摜𕪊AzƂĕԂ܂.
	 *
	 * @param src 摜B<br>
	 * @param y YWB<br>
	 * @param w ؂oB<br>
	 * @param h ؂oB<br>
	 *
	 * @return srcw̕Ő؂ỏ摜B<br>
	 *
	 * @throws RasterFormatException W܂̓TCYsȏꍇɓB<br>
	 */
	public static BufferedImage[] rows(BufferedImage src, int y, int w, int h) throws RasterFormatException {
		BufferedImage[] dst = new BufferedImage[src.getWidth() / w];
		for (int i = 0, x = 0; i < dst.length; i++, x += w) {
			dst[i] = src.getSubimage(x, y, w, h);
		}
		return dst;
	}

	/**
	 * BudderdImagex, 0 w, h̃TCYŏcɉ摜𕪊AzƂĕԂ܂.
	 *
	 * @param src 摜B<br>
	 * @param x XWB<br>
	 * @param w ؂oB<br>
	 * @param h ؂oB<br>
	 *
	 * @return srcch̍Ő؂ỏ摜B<br>
	 *
	 * @throws RasterFormatException W܂̓TCYsȏꍇɓB<br>
	 */
	public static BufferedImage[] columns(BufferedImage src, int x, int w, int h) throws RasterFormatException {
		BufferedImage[] dst = new BufferedImage[src.getHeight() / h];
		for (int i = 0, y = 0; i < dst.length; i++, y += h) {
			dst[i] = src.getSubimage(x, y, w, h);
		}
		return dst;
	}

	/**
	 * BufferedImage0, 0,w, h̃TCYœ񎟌ɉ摜𕪊AXgƂĕԂ܂.
	 *
	 * Ԃ郊Xg1ŁA摜̍ォEׂ֕܂B<br>
	 *
	 * @param src 摜B<br>
	 * @param w ؂oB<br>
	 * @param h ؂oB<br>
	 *
	 * @return srcw肳ꂽTCYŐ؂ỏ摜B<br>
	 *
	 * @throws RasterFormatException W܂̓TCYsȏꍇɓB<br>
	 */
	public static List<BufferedImage> splitAsList(BufferedImage src, int w, int h) throws RasterFormatException {
		int columns = src.getHeight() / h;
		List<BufferedImage> dst = new ArrayList<BufferedImage>(columns * (src.getWidth() / w));
		for (int i = 0; i < columns; i++) {
			dst.addAll(Arrays.asList(rows(src, i * h, w, h)));
		}
		return dst;
	}

	/**
	 * BufferedImage0, 0,w, h̃TCYœ񎟌ɉ摜𕪊AzƂĕԂ܂.
	 *
	 * @param src 摜B<br>
	 * @param w ؂oB<br>
	 * @param h ؂oB<br>
	 *
	 * @return srcw肳ꂽTCYŐ؂ỏ摜B<br>
	 *
	 * @throws RasterFormatException W܂̓TCYsȏꍇɓB<br>
	 */
	public static BufferedImage[][] splitAsArray(BufferedImage src, int w, int h) throws RasterFormatException {
		BufferedImage[][] dst = new BufferedImage[src.getHeight() / h][src.getWidth() / w];
		for (int i = 0, y = 0; i < dst.length; i++, y += h) {
			dst[i] = rows(src, y, w, h);
		}
		return dst;
	}

	/**
	 * BufferedImage0, 0,w, h̃TCYœ񎟌ɉ摜𕪊A}bvƂĕԂ܂. evf̖K0x[X[c̗vfԍ][̗vfԍ]̓񂯂̐ƂȂ܂B<br>
	 * vf2ɖȂꍇ0n̂悤ɐ`܂B<br>
	 *
	 * @param src 摜B<br>
	 * @param w ؂oB<br>
	 * @param h ؂oB<br>
	 *
	 * @return srcw肳ꂽTCYŐ؂ỏ摜B<br>
	 *
	 * @throws RasterFormatException W܂̓TCYsȏꍇɓB<br>
	 */
	public static Map<String, BufferedImage> splitAsMap(BufferedImage src, int w, int h) throws RasterFormatException {
		HashMap<String, BufferedImage> dst = new HashMap<String, BufferedImage>(src.getWidth() / w * src.
				getHeight() / h);
		BufferedImage[][] dist = splitAsArray(src, w, h);
		for (int i = 0; i < dist.length; i++) {
			for (int j = 0; j < dist[i].length; j++) {
				String y = Integer.toString(i);
				String x = Integer.toString(j);
				if (y.length() == 1) {
					y = "0" + y;
				}
				if (x.length() == 1) {
					x = "0" + x;
				}
				dst.put(y + x, dist[i][j]);
			}
		}
		return dst;
	}

	/**
	 * BufferedImage0, 0,w, h̃TCYœ񎟌ɉ摜𕪊A}bvƂĕԂ܂. evf̖K0x[X[c̗vfԍ][̗vfԍ]digit̐ƂȂ܂B<br>
	 * vf2ɖȂꍇ0n̂悤ɐ`܂B<br>
	 *
	 * @param src 摜B<br>
	 * @param w ؂oB<br>
	 * @param h ؂oB<br>
	 * @param digit 摜̃CfbNX̌B0߂B<br>
	 *
	 * @return srcw肳ꂽTCYŐ؂ỏ摜B<br>
	 *
	 * @throws RasterFormatException W܂̓TCYsȏꍇɓB<br>
	 */
	public static Map<String, BufferedImage> splitAsMapN(BufferedImage src, int w, int h, int digit) throws RasterFormatException {
		HashMap<String, BufferedImage> dst = new HashMap<String, BufferedImage>(src.getWidth() / w * src.
				getHeight() / h);
		BufferedImage[][] dist = splitAsArray(src, w, h);
		for (int i = 0; i < dist.length; i++) {
			for (int j = 0; j < dist[i].length; j++) {
				String y = Integer.toString(i);
				String x = Integer.toString(j);
				while (y.length() < digit) {
					y = "0" + y;
				}
				while (x.length() < digit) {
					x = "0" + x;
				}
				dst.put(y + x, dist[i][j]);
			}
		}
		return dst;
	}

	/**
	 * w肳ꂽ̈̃Lv`w肳ꂽt@Cɕۑ܂. ̃\bh͐VXbh[kgf screen shot]NÃXbhŉ摜쐬ĕۑ܂B<br>
	 * 摜̏㏑mF͍s܂BIɏ㏑܂B<br>
	 *
	 * @param FILE_PATH t@CpXLq܂B<br>
	 * @param BOUNDS Lv`̈.foCX̃O[oWŎw肵܂B<br>
	 *
	 * @throws ContentsIOException 摜ۑłȂꍇуXN[Vbg擾łȂꍇ ܂B<br>
	 */
	public static void screenShot(final String FILE_PATH, final Rectangle BOUNDS) throws ContentsIOException {
		new Thread("kgf screen shot") {
			@Override
			public void run() {
				try {
					BufferedImage image = new Robot().createScreenCapture(BOUNDS);
					ImageUtil.save(FILE_PATH, image);
					GameLog.printInfoIfUsing("XN[VbgBe܂ FILE_PATH=[" + FILE_PATH + "]");
				} catch (AWTException ex) {
					throw new ContentsIOException(ex);
				}
			}
		}.start();
	}

	/**
	 * \[X摜w肳ꂽɕׂ摜쐬܂.
	 *
	 * @param src ^CO\[X摜w肵܂B̉摜̃sNZf[^͑삳܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 * @param xNum Xɕׂ鐔w肵܂B<br>
	 * @param yNum Yɕׂ鐔w肵܂B<br>
	 *
	 * @return \[X摜2ɌԂȂׂ摜dstɊi[ĕԂ܂B<br>
	 */
	public static BufferedImage tiling(BufferedImage src, BufferedImage dst, int xNum, int yNum) {
		if (dst == null || dst == src) {
			dst = newImage(src.getWidth() * xNum, src.getHeight() * yNum);
		}
		Graphics2D g2 = dst.createGraphics();
		for (int y = 0, width = src.getWidth(), height = src.getHeight(); y < yNum; y++) {
			for (int x = 0; x < xNum; x++) {
				g2.drawImage(src, x * width, y * height, null);
			}
		}
		g2.dispose();
		return dst;
	}

	/**
	 * \[X摜w肳ꂽɕׂ摜쐬܂.
	 *
	 * @param src ^CO\[X摜w肵܂B̉摜̃sNZf[^͑삳܂B<br>
	 * @param dst nullłȂꍇÄɌʂi[܂B<br>
	 * @param xNum Xɕׂ鐔w肵܂B<br>
	 * @param yNum Yɕׂ鐔w肵܂B<br>
	 * @param drawWidth 摜`悷ۂ̃TCYw肵܂B<br>
	 * @param drawHeight 摜`悷ۂ̃TCYw肵܂B<br>
	 * @return \[X摜2ɌԂȂׂ摜dstɊi[ĕԂ܂B<br>
	 */
	public static BufferedImage tiling(BufferedImage src, BufferedImage dst, int xNum, int yNum,
			int drawWidth, int drawHeight) {
		if (dst == null || dst == src) {
			dst = newImage(xNum * drawWidth, yNum * drawHeight);
		}
		Graphics2D g2 = dst.createGraphics();
		for (int y = 0; y < yNum; y++) {
			for (int x = 0; x < xNum; x++) {
				g2.drawImage(src, x * drawWidth, y * drawHeight, drawWidth, drawHeight, null);
			}
		}
		g2.dispose();
		return dst;
	}

	/**
	 * w肵̈̉摜VCX^XƂĕԂ܂B<br>
	 *
	 * @param src
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @return
	 * @throws RasterFormatException
	 */
	public static BufferedImage trimming(BufferedImage src, int x, int y, int width, int height)
			throws RasterFormatException {
		return src.getSubimage(x, y, width, height);
	}

	/**
	 * 1̉摜̓ߓxinitialTpdecTpύX摜zƂĕԂ܂. ̃\bhł́A\[X摜̊SɓȃsNZ͂̂܂ܓȃsNZƂăRs[܂B<br>
	 *
	 * @param image ߓxύX\[X摜B<br>
	 * @param initialTp ߓx̏lłB
	 * @param addTp ߓxɉZlłBʏ͕gp܂B<br>
	 *
	 * @return \[X摜́AXɓߓxς摜zƂĕԂ܂B<br>
	 *
	 * @throws IllegalArgumentException initailTp01𒴂Ƃɓ܂B<br>
	 */
	public static BufferedImage[] transparentArray(BufferedImage image, float initialTp, float addTp)
			throws IllegalArgumentException {
		if (initialTp > 1f || initialTp < 0f) {
			throw new IllegalArgumentException("initialTp > 1 || initialTp < 0 : initialTp=[" + initialTp + "], addTp=[" + addTp + "]");
		}
		BufferedImage src = ImageUtil.copy(image);
		List<BufferedImage> result = new ArrayList<BufferedImage>((int) (Math.abs(initialTp) / Math.
				abs(addTp)));
		for (float tp = initialTp; tp > 0; tp += addTp) {
			result.add(ImageUtil.copy(src));
			src = ImageEditor.transparent(src, tp, null);
		}
		return result.toArray(new BufferedImage[result.size()]);
	}

	/**
	 * 摜z𐅕ɕׂV摜쐬ĕԂ܂.
	 *
	 * @param images gp摜1ȏ㑗M܂B<br>
	 *
	 * @return images̏Ԃō琅ɌԂȂׂV摜Ԃ܂B<br>
	 *
	 * @throws IllegalArgumentException images̒0̂Ƃɓ܂B<br>
	 */
	public static BufferedImage lineUp(BufferedImage... images)
			throws IllegalArgumentException {
		if (images.length == 0) {
			throw new IllegalArgumentException("images is empty : images.length=[" + images.length + "]");
		}
		int maxHeight = images[0].getHeight();
		int width = 0;
		for (int i = 0; i < images.length; i++) {
			if (images[i].getHeight() > maxHeight) {
				maxHeight = images[i].getHeight();
			}
			width += images[i].getWidth();
		}
		BufferedImage result = newImage(width, maxHeight);
		Graphics2D g = result.createGraphics();
		g.setRenderingHints(RenderingPolicy.QUALITY.getRenderingHints());
		for (int i = 0, widthSum = 0; i < images.length; i++) {
			g.drawImage(images[i], widthSum, 0, null);
			widthSum += images[i].getWidth();
		}
		return result;
	}
}
