/*
 * The MIT License
 *
 * Copyright 2013 Dra0211.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package kinugasa.contents.sound;

import java.util.HashMap;
import java.util.Set;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.BooleanControl;
import javax.sound.sampled.Clip;
import javax.sound.sampled.Control;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.FloatControl;
import kinugasa.contents.resource.ContentsIOException;
import kinugasa.contents.resource.NotYetLoadedException;
import kinugasa.game.GameLog;
import kinugasa.util.StopWatch;

/**
 * LbV̃TEh̎ł.
 * <br>
 * WAVEt@CĐۂ́AłʓIȃTEh̎ƂȂ܂B<br>
 *
 * <br>
 *
 * @version 2.0.0 - 2013/01/13_18:46:43<br>
 * @author Dra0211<br>
 */
public class CachedSound implements Sound {

	/**
	 * TEh̃LbVf[^.
	 */
	private static final HashMap<SoundBuilder, CachedSound> CACHE = new HashMap<SoundBuilder, CachedSound>(16);

	/**
	 * VLbVTEh\z܂.
	 *
	 * @param b TEh̍\zɎgpr_.<br>
	 *
	 * @return r_̐ݒō쐬ꂽLbVTEh.<br>
	 */
	static CachedSound create(SoundBuilder b) {
		if (CACHE.containsKey(b) && !b.isNewFile()) {
			GameLog.printInfoIfUsing("CachedSound LbVԂ܂ name=[" + b.getName() + "]");
			return CACHE.get(b);
		}
		CachedSound sc = new CachedSound(b);
		CACHE.put(b, sc);
		GameLog.printInfoIfUsing("CachedSound CX^XVK쐬܂ name=[" + b.getName() + "]");
		return sc;
	}
	/**
	 * ̃TEh\zr_.
	 */
	private SoundBuilder builder;
	/**
	 * ̃TEh̃Xg[.
	 */
	private transient Clip clip;
	/**
	 * ̃TEh̃[vݒ.
	 */
	private LoopPoint lp;

	/**
	 * TEh쐬. ̃RXgN^ł́Af.exisť͂Ȃ.
	 *
	 * @param builder t@CCX^X.<br>
	 */
	private CachedSound(SoundBuilder builder) {
		this.builder = builder;
		if (Float.compare(builder.getMasterGain(), 1.0f) != 0) {
			setControl(FloatControl.Type.MASTER_GAIN, (float) Math.log10(builder.getMasterGain()) * 20);
		}
		if (Float.compare(builder.getVolume(), 1.0f) != 0) {
			setControl(FloatControl.Type.VOLUME, builder.getVolume());
		}
		if (Float.compare(builder.getPan(), 0f) != 0) {
			setControl(FloatControl.Type.PAN, builder.getPan());
		}
		if (!builder.getReverbModel().equals(ReverbModel.NO_USE)) {
			setControl(FloatControl.Type.REVERB_RETURN, builder.getReverbModel().getRet());
			setControl(FloatControl.Type.REVERB_SEND, builder.getReverbModel().getSend());
			setControl(BooleanControl.Type.APPLY_REVERB, builder.getReverbModel().isUse() ? 1f : 0f);
		}
		if (builder.getLoopPoint() != null && !builder.getLoopPoint().equals(LoopPoint.NO_USE)) {
			setLoopPoints(builder.getLoopPoint());
		}
		if (Float.compare(builder.getSampleRate(), 0f) != 0) {
			setControl(FloatControl.Type.SAMPLE_RATE, builder.getSampleRate());
		}
	}
	/**
	 * [hɓKpRg[.
	 */
	private HashMap<Control.Type, Float> ctrls = new HashMap<Control.Type, Float>(8);

	/**
	 * Rg[obt@O.
	 *
	 * @param t Rg[̃^Cv.<br>
	 * @param val l.<br>
	 */
	private void setControl(Control.Type t, float val) {
		ctrls.put(t, val);
	}

	/**
	 * [vʒuݒ肵܂.
	 *
	 * @param p [vʒu.<br>
	 */
	private void setLoopPoints(LoopPoint p) {
		this.lp = p;
	}

	@Override
	public void setVolume(float vol) {
		FloatControl control = (FloatControl) clip.getControl(FloatControl.Type.MASTER_GAIN);
		control.setValue((float) Math.log10(vol) * 20);
	}

	@Override
	public void play() throws NotYetLoadedException {
		if (!isLoaded()) {
			throw new NotYetLoadedException("sound " + this + " is not yet loaded.");
		}
		if (lp != LoopPoint.NO_USE) {
			clip.loop(Clip.LOOP_CONTINUOUSLY);
		} else {
			clip.start();
		}
	}

	@Override
	public void again() {
		stop();
		play();
	}

	@Override
	public long getFramePosition() {
		return clip == null ? -1 : clip.getLongFramePosition();
	}

	@Override
	public long getFrameLength() {
		return clip == null ? -1 : clip.getFrameLength();
	}

	@Override
	public void stop() {
		if (clip != null) {
			clip.stop();
			clip.setFramePosition(0);
		}
	}

	@Override
	public void pause() {
		if (clip != null) {
			clip.stop();
		}
	}

	@Override
	public boolean isRunning() {
		return clip == null ? false : clip.isRunning();
	}

	@Override
	public CachedSound load() throws ContentsIOException, SoundStreamException {
		if (isLoaded()) {
			return this;
		}
		StopWatch watch = new StopWatch().start();
		AudioInputStream stream = null;
		try {
			stream = AudioSystem.getAudioInputStream(builder.getFile());
			DataLine.Info dInfo = new DataLine.Info(Clip.class, stream.getFormat());
			clip = (Clip) AudioSystem.getLine(dInfo);
			clip.open(stream);
			Set<Control.Type> types = ctrls.keySet();
			for (Control.Type t : types) {
				float val = ctrls.get(t);
				try {
					FloatControl control = (FloatControl) clip.getControl(t);
					control.setValue(val);
				} catch (IllegalArgumentException i) {
					GameLog.printInfoIfUsing("! > SoundClip : [" + getName() + "] : UN SUPPORTED CONTROL : Type=[" + t + "]");
				}
			}
			if (lp != null) {
				clip.setLoopPoints(lp.getTo(), lp.getFrom());
			}
		} catch (Exception ex) {
			throw new SoundStreamException(ex);
		} finally {
			try {
				if (stream != null) {
					stream.close();
				}
			} catch (java.io.IOException ex) {
				throw new ContentsIOException(ex);
			}
		}
		watch.stop();
		GameLog.printInfoIfUsing("CachedSound [h܂ name=[" + getName() + "](" + watch.getTime() + " ms)");
		return this;
	}

	@Override
	public CachedSound free() {
		stop();
		if (clip != null) {
			clip.flush();
			clip.close();
		}
		clip = null;
		GameLog.printInfoIfUsing("CachedSound : [" + getName() + "] : FREE");
		return this;
	}

	@Override
	public boolean isLoaded() {
		return !(clip == null);
	}

	@Override
	public String toString() {
		return "CachedSound{" + "name=" + getName() + ", lp=" + lp + ", load=" + isLoaded() + ", run=" + isRunning() + '}';
	}

	@Override
	public String getName() {
		return builder.getName();
	}

	/**
	 * ̃TEh\zr_擾܂. r_ւ̐ݒ́AӖ܂B<br>
	 *
	 * @return ̃TEh\z邽߂ɍ쐬ꂽr_̃CX^XԂ܂B<br>
	 */
	public SoundBuilder getBuilder() {
		return builder;
	}
}
