import sys
sys.path.append((sys.path[0] or ".") + "/..")
import unittest
import os.path
import marshal

from bbs.topic import Topic
import bbs.topic as topic
import mockhttp
import bbs.http as http
from bbs.message import Message
import bbs.board as board
import config


class TopicTestCase(unittest.TestCase):
    def setUp(self):
        topic.http = mockhttp
        self.host = "kittywalk.sourceforge.jp"
        self.path = "test"
        self.id = "0000000001"
        self.info_file_path = os.path.join(config.CACHEDIR,
                                           self.host,
                                           self.path,
                                           self.id)
        self.topic = Topic(self.host, self.path, self.id)

    def tearDown(self):
        topic.http = http
        if os.path.exists(self.info_file_path):
            os.remove(self.info_file_path)
        if os.path.exists(self.topic._data_file_path):
            os.remove(self.topic._data_file_path)

    def test_title(self):
        """Topic.title() extracts HTML entity"""
        self.topic._title = "&gt;"
        self.assertEqual(self.topic.title(), u">")

    def test_summary(self):
        """Summary is modified after update() call"""
        size = 23
        content = "name<>mail<>time<>body<>\n" * size
        mockhttp.set_data([{"status":200,
                            "content":content,
                            "last-modified":"now"}])
        self.topic.update()
        s = board.Summary(self.topic)
        self.assertEqual(s.get(self.topic,"size"), size)

    def test_equality(self):
        """topic1==topic2 if url match"""
        t1 = Topic(self.host, self.path, self.id, title="foo")
        t2 = Topic(self.host, self.path, self.id, title="bar")
        self.assert_(t1==t2)
    
    def test_messages_empty(self):
        """Topic.messages() returns [] if local file does not exist"""
        if os.path.exists(self.topic._data_file_path):
            os.remove(self.topic._data_file_path)
        self.assertEqual([m for m in self.topic.messages()], [])

    def test_messages_non_empty(self):
        """Topic.messages() returns a list of Message instances"""
        name1,mail1,time1,body1 = "1","2","3","4"
        name2,mail2,time2,body2 = "5","6","7","8"
        data = "%s<>%s<>%s<>%s<>title\n" % (name1,mail1,time1,body1) +\
               "%s<>%s<>%s<>%s<>" % (name2,mail2,time2,body2)
        file(self.topic._data_file_path,"w").write(data)
        m1 = Message(1, name1,mail1,time1,body1)
        m2 = Message(2, name2,mail2,time2,body2)
        self.assertEqual([m for m in self.topic.messages()], [m1, m2])
        
    def test_update_with_old_data_full(self):
        """Topic.update() downloads and save data"""
        old_data = "old"
        new_data = "new"
        file(self.topic._data_file_path,"w").write(old_data)
        mockhttp.set_data([{"status":http.OK,"content":"new"}])
        self.topic.update()
        self.assertEqual(file(self.topic._data_file_path,"r").read(),
                         new_data)

    def test_update_with_old_data_partial(self):
        """Topic.update() downloads partial data and save data"""
        old_data = "old"
        new_data = "new"
        file(self.topic._data_file_path,"w").write(old_data)
        mockhttp.set_data([{"status":http.PARTIAL_CONTENT,"content":"new"}])
        self.topic.update()
        self.assertEqual(file(self.topic._data_file_path,"r").read(),
                         old_data+new_data)

    def test_update_with_latest_data(self):
        """Topic.update() does not download if data file is up to date"""
        current_data = "current_data"
        file(self.topic._data_file_path,"w").write(current_data)
        mockhttp.set_data([{"status":http.NOT_MODIFIED,"content":"wrong"}])
        self.topic.update()
        self.assertEqual(file(self.topic._data_file_path,"r").read(),
                         current_data)

    def test_update_without_datafile(self):
        """Topic.update() downloads from bbs if data file does not exist"""
        f = self.topic._data_file_path
        if os.path.exists(f):
            os.remove(f)
        mockhttp.set_data([{"status":200,
                            "content":"hello",
                            "last-modified":"now"}])
        self.topic.update()
        self.assert_(os.path.exists(f))

    def test_update_error(self):
        """Topic.update() raises DownloadError when bad status"""
        mockhttp.set_data([{"status":301,
                            "content":"foo",
                            "last-modified":"now"}])
        self.assertRaises(topic.DownloadError, self.topic.update)
  
    def test_save_info(self):
        """Topic._save_info(info)"""
        d = {"last-modified":"now"}
        if os.path.exists(self.info_file_path):
            os.remove(self.info_file_path)
        self.topic._save_info(d)
        self.assertEqual(marshal.load(file(self.info_file_path,"r")), d)

    def test_save_info_empty(self):
        """Topic._save_info({}) does not create file"""
        if os.path.exists(self.info_file_path):
            os.remove(self.info_file_path)
        self.topic._save_info({})
        self.failIf(os.path.exists(self.info_file_path))

    def test_load_info(self):
        """Topic._load_info() returns saved info"""
        info = {"key":"val"}
        marshal.dump(info, file(self.topic._info_file_path,"w"))
        self.assertEqual(self.topic._load_info(),info)

    def test_load_info_empty(self):
        """Topic._load_info() returns {} if info is not saved"""
        if os.path.exists(self.topic._info_file_path):
            os.remove(self.topic._info_file_path)
        self.assertEqual(self.topic._load_info(),{})


if __name__ == "__main__":
    unittest.main()
