from qt import *

from utils import *
from boardtreewidget import BoardTreeWidget
from topiclistwidget import TopicListWidget
from messagewidget import MessageWidget
from ui import ICONPATH
import config
from version import kittywalk_version


class MainWindow(QMainWindow):
    def __init__(self):
        QMainWindow.__init__(self)
        self._splitter = QSplitter(self)
        self._board_tree_widget = BoardTreeWidget(self._splitter)
        self._topic_list_widget = TopicListWidget(self._splitter)
        self._message_widget = MessageWidget(self._splitter)
        self._actions = self._build_actions()
        self._init_tool_bar()
        self._init_menu_bar()
        self._init_status_bar()
        self._connect()
        self._initial_config()

    def display_board_tree(self, bbs):
        self._board_tree_widget.display(bbs)

    def _build_actions(self):
        actions = {}
        actions["toggle-board-tree-widget"] =\
          QAction(u("掲示板一覧"),
                  QIconSet(QPixmap(os.path.join(ICONPATH, "leftpane.png"))),
                  u("掲示板一覧"),
                  0, self, "", True)
        actions["toggle-topic-list-widget"] =\
          QAction(u("スレッド一覧"),
                  QIconSet(QPixmap(os.path.join(ICONPATH, "midpane.png"))),
                  u("スレッド一覧"),
                  0, self, "", True)
        actions["toggle-message-widget"] =\
          QAction(u("書込内容"),
                  QIconSet(QPixmap(os.path.join(ICONPATH, "rightpane.png"))),
                  u("書込内容"),
                  0, self, "", True)
        return actions

    def _init_tool_bar(self):
        toobar = QToolBar(self)
        self._actions["toggle-board-tree-widget"].addTo(toobar)
        self._actions["toggle-topic-list-widget"].addTo(toobar)
        self._actions["toggle-message-widget"].addTo(toobar)

    def _init_menu_bar(self):
        menubar = self.menuBar()
        # File
        file_menu = QPopupMenu(menubar)
        file_menu.insertItem(u("終了(&X)"), self.close)
        menubar.insertItem(u("ファイル(&F)"), file_menu)
        # View
        view_menu = QPopupMenu(menubar)
        view_menu.insertItem(u("フォント(&F)"), self._change_font)
        menubar.insertItem(u("表示(&V)"), view_menu)
        # Help
        help_menu = QPopupMenu(menubar)
        help_menu.insertItem(u("KittyWalk について(&A)"), show_about_dialog)
        menubar.insertItem(u("ヘルプ(&H)"), help_menu)

    def _init_status_bar(self):
        self.statusBar().message("")

    def _connect(self):
        self.connect(self._board_tree_widget,
                    PYSIGNAL("requestOpen"),
                    PYSIGNAL("requestOpenTopicList"))
        self.connect(self._actions["toggle-board-tree-widget"],
                     SIGNAL("toggled(bool)"),
                     self._toggle_board_tree_widget)
        self.connect(self._actions["toggle-topic-list-widget"],
                     SIGNAL("toggled(bool)"),
                     self._toggle_topic_list_widget)
        self.connect(self._actions["toggle-message-widget"],
                     SIGNAL("toggled(bool)"),
                     self._toggle_message_widget)
        self.connect(self._topic_list_widget,
                     PYSIGNAL("requestSyncTopicList"),
                     PYSIGNAL("requestSyncTopicList"))
        self.connect(self._topic_list_widget,
                     PYSIGNAL("requestOpenMessages"),
                     PYSIGNAL("requestOpenMessages"))
        self.connect(self._message_widget,
                     PYSIGNAL("requestSyncMessages"),
                     PYSIGNAL("requestSyncMessages"))

    def _initial_config(self):
        self.setCaption("KittyWalk Qt")
        self.setCentralWidget(self._splitter)
        self.setIcon(QPixmap(os.path.join(ICONPATH, "kittywalk.png")))
        self._board_tree_widget.show()
        self._topic_list_widget.hide()
        self._message_widget.hide()
        self._actions["toggle-board-tree-widget"].setOn(True)
        self._actions["toggle-topic-list-widget"].setOn(False)
        self._actions["toggle-message-widget"].setOn(False)
        self.resize(config.get("gui-mainwindow-width", 640),
                    config.get("gui-mainwindow-height", 480))
        self._splitter.setSizes(config.get("gui-mainwidget-sizes",[640,0,0]))

    def echo(self, text):
        self.statusBar().message(text)

    def display_topic_list(self, board, topics, newtab):
        self._topic_list_widget.display(board, topics, newtab)
        self._actions["toggle-topic-list-widget"].setOn(True)

    def display_messages(self, topic, messages, newtab):
        self._message_widget.display(topic, messages, newtab)
        self._actions["toggle-message-widget"].setOn(True)

    def set_message_font(self, font):
        self._message_widget.set_font(font)

    def _toggle_board_tree_widget(self, on):
        self._toggle_widget(self._board_tree_widget, on)

    def _toggle_topic_list_widget(self, on):
        self._toggle_widget(self._topic_list_widget, on)

    def _toggle_message_widget(self, on):
        self._toggle_widget(self._message_widget, on)

    def _toggle_widget(self, widget, on):
        if on:
            widget.show()
        else:
            widget.hide()

    def close(self, alsoDelete):
        config.set("gui-mainwindow-width", self.width())
        config.set("gui-mainwindow-height", self.height())
        config.set("gui-mainwidget-sizes", self._splitter.sizes())
        return QWidget.close(self, alsoDelete)

    def _change_font(self):
        self.emit(PYSIGNAL("requestChangeFont"),
                  (self._message_widget.font(),))


def show_about_dialog():
    caption = u("KittyWalk について")
    text = "\n\n".join(["KittyWalk " + kittywalk_version(),
                        "http://kittywalk.sourceforge.jp",
                        "onefifty@users.sourceforge.jp"])
    QMessageBox.about(None, caption, text)

