"""Board"""

from __future__ import generators
import re
import marshal

from utils import *
import config
import http
from topic import Topic
from summary import Summary

class DownloadError(Exception): pass


class Board(object):

    _topic_entry_re = re.compile(u"""(?P<id>.+)[.]dat<>
                                     (?P<title>.+)\s*
                                     \((?P<size>\d+)\)""", re.X)

    def __init__(self, host="", path="", title=""):
        assert isinstance(host, str)
        assert isinstance(path, str)
        assert isinstance(title, (str,unicode))

        self._host = host
        self._path = path
        self._title = j(title)
        self._data_file_path = \
          os.path.join(config.CACHEDIR,host,  path,"subject.txt")
        self._info_file_path = \
          os.path.join(config.CACHEDIR,host,path,"subject")

    def host(self): return self._host
    def path(self): return self._path
    def title(self): return self._title

    def topics(self):
        """Iterate over topics in this board

        This method read the data file locally stored, parse the data
        and yield every Topic instances.

        """
        if os.path.exists(self._data_file_path):
            fin = file(self._data_file_path)
            for line in fin.readlines():
                m = self._topic_entry_re.match(line)
                if m:
                    yield Topic(self._host,
                                self._path,
                                m.group("id"),
                                m.group("title"),
                                int(m.group("size")))
            fin.close()

    def update(self):
        """Update local data

        This method downloads board data and save them in local file.
        When requesting http GET method, "If-Modified-Since" field is
        used to avoid unnecessary transfer.  "Accept-Encoding: gzip"
        is used too.

        """
        # Download data
        info = self._load_info()
        headers = {"Accept-Encoding": "gzip",
                   "If-Modified-Since": info.get("last-modified","")}
        response = http.get(self._host, "/%s/subject.txt"%self._path, headers)
        stat = response["status"]
        if stat==http.OK:
            # Save files
            d = os.path.dirname(self._data_file_path)
            if not os.path.exists(d):
                os.makedirs(d)
            file(self._data_file_path,"w").write(response["content"])
            info.update({"last-modified":response["last-modified"]})
            self._save_info(info)
        elif stat==http.NOT_MODIFIED:
            pass
        else:
            raise DownloadError, stat

    def is_empty(self):
        """Return if board has locally stored data"""
        return not os.path.exists(self._data_file_path)
        
    def _load_info(self):
        """Return info dictionary

       This method extracts the content of info file by marshal and
        return the resulting dictionary.  If the file does not exists,
        an empty dictionary is returned.
        """
        if os.path.exists(self._info_file_path):
            fin = file(self._info_file_path)
            return marshal.load(fin)
        else:
            return {}

    def _save_info(self, info):
        """Save info

        If info is a non empty dictionary, this method saves it using
        marshal.  If info is empty, nothing is saved.
        """
        assert isinstance(info, dict)
        if info=={}:
            return

        d = os.path.dirname(self._info_file_path)
        if not os.path.exists(d):
            os.makedirs(d)
        marshal.dump(info, file(self._info_file_path,"w"))
        
    def __eq__(self, other):
        return ((self._host==other._host) and
                (self._path==other._path))

    def __hash__(self):
        return hash((self._host,self._path))
    
