"""Topic -- Topic (or thread) in a board"""

from __future__ import generators
import os
import re
import marshal

from utils import *
import config
import http
from message import Message
from summary import Summary
import Logger


class DownloadError(Exception): pass

class Topic(object):
    """Topic in board, or thread"""
    _message_entry_re = re.compile(u"""(?P<name>.*?)<>
                                       (?P<mail>.*?)<>
                                       (?P<time>.*?)<>
                                       (?P<body>.*?)<>""", re.X)

    def __init__(self, host, path, id, title="", remote_size=None):
        assert isinstance(host, str)
        assert isinstance(path, str)
        assert isinstance(id, str)
        assert isinstance(title, (str,unicode))
        assert isinstance(remote_size, (int, type(None)))

        self._host = host       # "pc2.2ch.net"
        self._path = path       # "tech"
        self._id = id           # "0123456789"
        self._title = j(title)  # "[indent] Python Python [muneo]"
        self._remote_size = remote_size
        
        self._data_file_path = os.path.join(config.CACHEDIR,host,path,id+".dat")
        self._info_file_path = os.path.join(config.CACHEDIR,host,path,id)
        self._data_url_path = "/%s/dat/%s.dat" % (path,id)

        info = self._load_info()
        self._local_size = info.get("local-size", None)
        self._remote_size = max((self._remote_size or 0),
                                info.get("remote-size"))

    def __eq__(self, other):
        return ((self._host==other._host) and
                (self._path==other._path) and
                (self._id==other._id))

    def host(self): return self._host
    def path(self): return self._path
    def id(self): return self._id
    def local_size(self): return self._local_size
    def remote_size(self): return self._remote_size
    def title(self): return extract_html_entities(self._title)
    def messages(self):
        """Iterate over messages in this topic (thread)

        This method reads data file stored in local disk, parse the
        content, and yield Message instaces.
        
        """
        if os.path.exists(self._data_file_path):
            i = 1
            for line in file(self._data_file_path).xreadlines():
                d = self._message_entry_re.match(line).groupdict()
                yield Message(i,
                              re.sub("</?b>","",d["name"]),
                              d["mail"],
                              d["time"],
                              d["body"])
                i += 1

    def update(self):
        """Update local data

        This method downloads message data (.dat file) and store it
        locally.  Last-Modified and Range fields are used to minimize
        data transfer.  

        If the local file exists, the file size is set to http's
        "Range:" header, and "Last-Modified" value, stored in info
        file, is set to "If-Modified-Since" header. If the local file
        does not exists, these header fields are blank.

        After the header preparation, message data chunk is downloaded
        and saved locally.  If partial data are transfered, they are
        appended to the existing local data.  "Last-Modified" is save
        in info file.

        Also, this method revises summary so that Board instance is able
        to know about topic.

        DownloadError raises if failed.

        """
        # Set up headers
        info = self._load_info()
        headers = {}
        if os.path.exists(self._data_file_path):
            size = os.stat(self._data_file_path).st_size
            headers["Range"] = "%d-" % size
        headers["If-Modified-Since"] = info.get("last-modified","")
        # Download data
        r = http.get(self._host, self._data_url_path, headers)
        # Save data and info
        status = r["status"]
        mode = {http.OK:"wb", http.PARTIAL_CONTENT:"ab"}
        if status==http.OK or status==http.PARTIAL_CONTENT:
            # Create directory if necessary
            if not os.path.exists(os.path.dirname(self._data_file_path)):
                os.makedirs(os.path.dirname(self._data_file_path))
            # Save files
            file(self._data_file_path,mode[status]).write(r["content"])
            size = r["content"].count("\n")
            info.update({"last-modified": r.get("last-modified",""),
                         "local-size": size,
                         "remote-size":size})
            self._save_info(info)
            Summary(self).set(self, "size", size)
            # Update size
            self._local_size = size
            self._remote_size = size
        elif status==http.NOT_MODIFIED:
            pass
        else:
            raise DownloadError, str(status)

    def is_empty(self):
        return not os.path.exists(self._data_file_path)

    def _load_info(self):
        """Return info dictionary

        This method extracts the content of info file by marshal and
        return the resulting dictionary.  If the file does not exists,
        an empty dictionary is returned.
        """
        try:
            if os.path.exists(self._info_file_path):
                return marshal.load(file(self._info_file_path))
            else:
                return {}
        except EOFError, why:
            Logger.log("EOFError in bbs.Topic._load_info() %s" % why)
            return {}

    def _save_info(self, info):
        """Save info

        If info is a non empty dictionary, this method saves it using
        marshal.  If info is empty, nothing is saved.
        """
        assert isinstance(info, dict)
        if info=={}:
            return

        d = os.path.dirname(self._info_file_path)
        if not os.path.exists(d):
            os.makedirs(d)
        marshal.dump(info, file(self._info_file_path,"w"))
        

