import unittest
import os.path
import sys
import marshal
sys.path.append((sys.path[0] or ".") + "/../")
from bbs.board import Board
import bbs.board as board
import bbs.http as http
import mockhttp
import bbs.topic as topic
from config import *
import config

class BoardTestCase(unittest.TestCase):
    def setUp(self):
        host = "kittywalk.sourceforge.jp"
        path = "test"
        self.board = Board(host,path)
        board.http = mockhttp
        self.info_file_path = os.path.join(config.CACHEDIR,host,path,"subject")
        if os.path.exists(self.info_file_path):
            os.remove(self.info_file_path)

    def tearDown(self):
        board.http = http
        if os.path.exists(self.info_file_path):
            os.remove(self.info_file_path)

    def test_messages_empty(self):
        """Board.topics() returns empty iterater if no data is stored"""
        if os.path.exists(self.board._data_file_path):
            os.remove(self.board._data_file_path)
        self.assertEqual([t for t in self.board.topics()], [])

    def test_message_not_empty(self):
        """Board.tipics() iterates over saved topics"""
        host = "kittywalk.sourceforge.jp"
        path = "test"
        id1, id2 = "0000000001", "0000000002"
        title1, title2 = "spam", "ham"
        size1, size2 = 1000,1
        t1 = topic.Topic(host, path, id1, title1)
        t2 = topic.Topic(host, path, id2, title2)
        data = "%s.dat<>%s (%d)\n" % (id1,title1,size1) + \
               "%s.dat<>%s (%d)" % (id2,title2,size2)
        file(self.board._data_file_path,"w").write(data)
        self.assertEqual([t for t in self.board.topics()],[t1,t2])
        
    def test_update_without_datafile(self):
        """Board.update() saves new data if local file does not exist"""
        data = "foo"
        mockhttp.set_data([{"status":200,
                            "last-modified":"past",
                            "content":data}])
        if os.path.exists(self.board._data_file_path):
            os.remove(self.board._data_file_path)
        self.board.update()
        self.assertEqual(file(self.board._data_file_path,"r").read(),
                         data)

    def test_update_saves_lastmodified(self):
        """Board.update() saves Last-Modified field"""
        lastmodified = "yesterday"
        if os.path.exists(self.board._data_file_path):
            os.remove(self.board._data_file_path)
        mockhttp.set_data([{"status":200,
                            "last-modified":lastmodified,
                            "content":"foo"}])
        self.board.update()
        self.assertEqual(marshal.load(file(self.info_file_path)),
                         {"last-modified":lastmodified})
        
    def test_update_with_new_datafile(self):
        """update saves new data if modified since last download"""
        olddata = "old"
        newdata = "new"
        file(self.board._data_file_path, "w").write(olddata)
        mockhttp.set_data([{"status":http.NOT_MODIFIED,
                            "last-modified":"tomorrow",
                            "content":newdata}])
        self.board.update()
        self.assertEqual(file(self.board._data_file_path).read(),
                         olddata)

    def test_update_with_old_datafile(self):
        """update does not save new data if not modified since last download"""
        olddata = "old"
        newdata = "new"
        file(self.board._data_file_path, "w").write(olddata)
        mockhttp.set_data([{"status":http.OK,
                            "last-modified":"tomorrow",
                            "content":newdata}])
        self.board.update()
        self.assertEqual(file(self.board._data_file_path).read(),
                         newdata)

    def test_update_failuer(self):
        """update raises exception if failed"""
        mockhttp.set_data([{"status":404,
                            "last-modified":"tomorrow",
                            "content":"data"}])
        self.assertRaises(board.DownloadError, self.board.update)

    def test_equality(self):
        """Board("a","b","c") == Board("a","b","X")"""
        self.assert_(Board("a","b","c")==Board("a","b","X"))


if __name__ == "__main__": unittest.main()
