from qt import *

from bbs import Board, Category
from utils import *


class BoardTreeWidget(QListView):
    """Board tree view"""

    class _ContextMenu(QPopupMenu):
        def __init__(self, parent):
            QPopupMenu.__init__(self, parent)
            self._view = parent
            self._target_item = None
            self.insertItem(u("新しいタブで開く"), self._open_in_new_tab)
            self.insertItem(u("今のタブで開く"), self._open_in_current_tab)

        def popup(self, item, position):
            self._target_item = item
            QPopupMenu.popup(self, position)

        def _open_in_new_tab(self):
            self._view._request_open(self._target_item, True)

        def _open_in_current_tab(self):
            self._view._request_open(self._target_item, False)

    def __init__(self, parent):
        QListView.__init__(self, parent)
        self._context_menu = self._ContextMenu(self)
        # Configure widgets
        self.setRootIsDecorated(True)
        self.addColumn(u("掲示板一覧"))
        self.setSorting(-1)  # sort disabled
        self.header().setClickEnabled(False)
        self.connect(self,
                     SIGNAL("doubleClicked(QListViewItem *)"),
                     self._request_open)
        self.connect(self,
                     SIGNAL("returnPressed(QListViewItem *)"),
                     self._request_open)
        self.connect(
          self,
          SIGNAL("rightButtonClicked(QListViewItem *,const QPoint &, int)"),
          self._popup_context_menu)

    def display(self, bbs):
        """Display given bbs tree"""
        self.clear()
        # Add every category
        last_citem = None
        for category in bbs.categories():
            last_citem = BoardTreeViewItem(self,last_citem,category)
            last_citem.setText(0, category.title())
            # Add every board
            last_bitem = None
            for board in category.boards():
                last_bitem = BoardTreeViewItem(last_citem, last_bitem, board)

    def _popup_context_menu(self, item, position, column):
        if item.board() != None:
            self._context_menu.popup(item, position)

    def _request_open(self, item, newtab=None):
        board = item.board()
        if isinstance(board, Board):
            self.emit(PYSIGNAL("requestOpen"), (board, newtab))


class BoardTreeViewItem(QListViewItem):
    """Subclass of QListViewItem used by BoardTreeView

    This class has _board attribute to keep a Board instance
    """
    def __init__(self, parent, after, bbs_element):
        assert isinstance(bbs_element, (Board, Category))
        if isinstance(bbs_element, Board):
            self._board = bbs_element
        else:
            self._board = None
        QListViewItem.__init__(self, parent, after)
        self.setText(0, bbs_element.title())

    def board(self):
        return self._board


