"""Widgets inheriting from Qt widgets"""

from qt import *
from Queue import Queue

from utils import *


class KittyTabWidget(QTabWidget):
    _tab_menu_spec = ((u("更新"),   "requestReload"),
                      (u("閉じる"), "requestClose"))

    def __init__(self, parent=None):
        QTabWidget.__init__(self, parent)
        self.setTabBar(KittyTabBar(self))
        self._tab_menu = self._TabMenu(self._tab_menu_spec, self)
        self._newtab = True
        self._model_view_registry = self._ModelViewRegistry()
        self._queue = Queue()
        self._timer = QTimer()
        self.connect(self._timer, SIGNAL("timeout()"), self._reduce_queue)
        self._timer.start(100, False)

    def _reduce_queue(self):
        if not self._queue.empty():
            self._display(*self._queue.get())

    def _display(self, *args):
        """Actually display

        This method must be inherited.
        """
        pass

    class _TabMenu(QPopupMenu):
        def __init__(self, spec, *args):
            QPopupMenu.__init__(self, *args)
            self._taget_widget = None
            self._signal = {}
            for text, signal in spec:
                id = self.insertItem(text)
                self._signal[id] = signal
            self.connect(self, SIGNAL("activated(int)"), self._activated)

        def popup(self, widget, position):
            self._target_widget = widget
            QPopupMenu.popup(self, position)

        def _activated(self, id):
            self.emit(PYSIGNAL(self._signal[id]), (self._target_widget,))


    class _ModelViewRegistry(dict):
        """Stores Model and View relations"""
        # This is a dictionary that key is model and value is view.

        def views(self):
            """Return list of views"""
            return self.values()

        def models(self):
            """Return list of models"""
            return self.keys()

        def has_model(self, model):
            """Return whether model is contained"""
            return self.has_key(model)

        def view(self, model):
            """Return view associated with model"""
            return self[model]

        def model(self, view):
            """Return model associated with view"""
            d = dict([(v,m) for (m,v) in self.items()])
            return d.get(view, None)

        def size(self):
            return len(self)

        def set(self, model, view):
            self[model] = view

        def remove(self, view):
            """Remove view"""
            del self[self.model(view)]


class KittyTabBar(QTabBar):
    """TabBar subclass

    KittyTabBar is subclass of QTabBar with few extensions.  This class 
    provides tabRightPressed() and similar PYSIGNAL's.  This wheel was 
    invented to use Qt2.3, while you don't need this class with Qt3.
    """

    def __init__(self, parent):
        QTabBar.__init__(self, parent)
        self._parent = parent

    def mousePressEvent(self, event):
        if event.button()==Qt.LeftButton:
            QTabBar.mousePressEvent(self, event)
        elif event.button()==Qt.RightButton:
            self._parent.emit(PYSIGNAL("tabRightPressed"),
                              (self.pointedPage(event),
                               self.mapToGlobal(event.pos())))
        elif event.button()==Qt.MidButton:
            self._parent.emit(PYSIGNAL("tabMidPressed"),
                              (self.pointedPage(event),
                               self.mapToGlobal(event.pos())))

    def mouseDoubleClickEvent(self, event):
        if event.button()==Qt.LeftButton:
            self._parent.emit(PYSIGNAL("tabLeftDoubleClicked"),
                              (self.pointedPage(event),
                               self.mapToGlobal(event.pos())))
        elif event.button()==Qt.RightButton:
            self._parent.emit(PYSIGNAL("tabRightDoubleClicked"),
                              (self.pointedPage(event),
                               self.mapToGlobal(event.pos())))
        elif event.button()==Qt.MidButton:
            self._parent.emit(PYSIGNAL("tabMidDoubleClicked"),
                              (self.pointedPage(event),
                               self.mapToGlobal(event.pos())))

    def pointedPage(self, event):
        current_id = self.currentTab()
        tab = self.selectTab(event.pos())
        self.setCurrentTab(tab)
        page = self._parent.currentPage()
        self.setCurrentTab(current_id)
        return page


class KittyTextBrowser(QTextBrowser):
    """Rich text browser

    This class inherits QTextBrowser.  Actual acutions of setText() and 
    ensureVisible() methods are queued to make sure you can scroll to a 
    certain position just after setText() call.  This widget emits 
    linkMouseMressEvent pysignal with (mouse_event, url_string).  Note 
    that any link-clicking, except #name, causes this emittion and all 
    navigation must be done explicitly.
    """
    def __init__(self, parent=None):
        QTextBrowser.__init__(self, parent)
        self._url = ""
        self._contents_height = self.contentsHeight()
        self._is_rendering = False
        self._queue = Queue()
        self.connect(self,
                     SIGNAL("highlighted(const QString &)"),
                     self._store_url)
        # Auto update
        self.timer = QTimer(self)
        self.connect(self.timer, SIGNAL("timeout()"), self._auto_update)
        self.timer.start(100, False)

    def setText(self, text):
        self._queue.put((QTextBrowser.setText, (self,text)))

    def ensureVisible(self, x, y):
        self._queue.put((QTextBrowser.ensureVisible, (self,x,y)))

    def setContentsPos(self, x, y):
        self._queue.put((QTextBrowser.setContentsPos, (self,x,y)))

    def scrollBy(self, dx, dy):
        self._queue.put((QTextBrowser.scrollBy, (self,dx,dy)))

    def viewportMousePressEvent(self, ev):
        if self._url:
            if self._url.startswith("#"):
                QTextBrowser.scrollToAnchor(self, self._url[1:])
            else:
                self.emit(PYSIGNAL("linkMousePressEvent"), (ev, self._url))
        else:
            QTextBrowser.viewportMousePressEvent(self, ev)

    def _auto_update(self):
        self._update_rendering_state()
        if self._is_rendering:
            return
        if not self._queue.empty():
            func, args = self._queue.get()
            func(*args)

    def _update_rendering_state(self):
        if self._is_rendering and self._contents_height==self.contentsHeight():
            self._is_rendering = False
        elif ((not self._is_rendering) and
              self._contents_height != self.contentsHeight()):
            self._is_rendering = True
        self._contents_height = self.contentsHeight()

    def _store_url(self, href):
        self._url = str(href)


class KittyWidgetStack(QWidget):
    def __init__(self, *args):
        QWidget.__init__(self, *args)
        self._combo_box = QComboBox(self)
        self._widget_stack = QWidgetStack(self)
        v = QVBoxLayout(self)
        v.addWidget(self._combo_box)
        v.addWidget(self._widget_stack)
        self.connect(self._combo_box, SIGNAL("activated(int)"), self._raise)

    def add(self, widget, name):
        assert isinstance(widget, QWidget)
        assert isinstance(name, (str, unicode))
        self._combo_box.insertItem(name)
        self._widget_stack.addWidget(widget, self._combo_box.count() - 1)

    def raiseWidget(self, *args):
        self._widget_stack.raiseWidget(*args)

    def _raise(self, index):
        self._widget_stack.raiseWidget(index)
