from qt import *

from utils import *
from boardtreewidget import BoardTreeWidget
from topiclistwidget import TopicListWidget
from messagewidget import MessageWidget
from BookmarkListWidget import BookmarkListWidget
from kittywidget import *
from ui import ICONPATH
import config
from version import kittywalk_version


class MainWindow(QMainWindow):
    def __init__(self):
        QMainWindow.__init__(self)
        self._main_splitter = QSplitter(self)
        self._bbs_widget = KittyWidgetStack(self._main_splitter)
        self._right_splitter = QSplitter(self._main_splitter)
        self._topic_list_widget = TopicListWidget(self._right_splitter)
        self._message_widget = MessageWidget(self._right_splitter)
        self._board_tree_widget = BoardTreeWidget(self._main_splitter)
        self._bookmark_list_widget = BookmarkListWidget(self._main_splitter)
        self._actions = self._build_actions()
        self._init_tool_bar()
        self._init_menu_bar()
        self._init_status_bar()
        self._connect()
        self._initial_config()

    def display_board_tree(self, bbs):
        self._board_tree_widget.display(bbs)

    def _build_actions(self):
        actions = {}
        actions["toggle-bbs-widget"] =\
          \
          QAction(u("掲示板一覧"),
                  QIconSet(QPixmap(os.path.join(ICONPATH, "leftpane.png"))),
                  u("掲示板一覧"),
                  0, self, "", True)
        actions["toggle-topic-list-widget"] =\
          QAction(u("スレッド一覧"),
                  QIconSet(QPixmap(os.path.join(ICONPATH, "toprightpane.png"))),
                  u("スレッド一覧"),
                  0, self, "", True)
        actions["toggle-message-widget"] =\
          QAction(u("書込内容"),
                  QIconSet(QPixmap(os.path.join(ICONPATH,
                                                "bottomrightpane.png"))),
                  u("書込内容"),
                  0, self, "", True)
        return actions

    def _init_tool_bar(self):
        toobar = QToolBar(self)
        self._actions["toggle-bbs-widget"].addTo(toobar)
        self._actions["toggle-topic-list-widget"].addTo(toobar)
        self._actions["toggle-message-widget"].addTo(toobar)

    def _init_menu_bar(self):
        menubar = self.menuBar()
        # File
        file_menu = QPopupMenu(menubar)
        file_menu.insertItem(u("掲示板一覧更新(&U)"), self._reload_board_tree)
        file_menu.insertSeparator()
        file_menu.insertItem(u("終了(&X)"), self.close)
        menubar.insertItem(u("ファイル(&F)"), file_menu)
        # View
        view_menu = QPopupMenu(menubar)
        self._actions["toggle-bbs-widget"].addTo(view_menu)
        self._actions["toggle-topic-list-widget"].addTo(view_menu)
        self._actions["toggle-message-widget"].addTo(view_menu)
        view_menu.insertItem(u("フォント変更(&F)"), self._change_font)
        menubar.insertItem(u("表示(&V)"), view_menu)
        # Help
        help_menu = QPopupMenu(menubar)
        help_menu.insertItem(u("KittyWalk について(&A)"), show_about_dialog)
        menubar.insertItem(u("ヘルプ(&H)"), help_menu)

    def _init_status_bar(self):
        self.statusBar().message("")

    def _connect(self):
        self.connect(self._board_tree_widget,
                    PYSIGNAL("requestOpen"),
                    PYSIGNAL("requestOpenTopicList"))
        self.connect(self._actions["toggle-bbs-widget"],
                     SIGNAL("toggled(bool)"),
                     self._toggle_bbs_widget)
        self.connect(self._actions["toggle-topic-list-widget"],
                     SIGNAL("toggled(bool)"),
                     self._toggle_topic_list_widget)
        self.connect(self._actions["toggle-message-widget"],
                     SIGNAL("toggled(bool)"),
                     self._toggle_message_widget)
        self.connect(self._topic_list_widget,
                     PYSIGNAL("requestReloadTopicList"),
                     PYSIGNAL("requestReloadTopicList"))
        self.connect(self._topic_list_widget,
                     PYSIGNAL("requestOpenMessages"),
                     PYSIGNAL("requestOpenMessages"))
        self.connect(self._message_widget,
                     PYSIGNAL("requestReloadMessages"),
                     PYSIGNAL("requestReloadMessages"))
        self.connect(self._message_widget,
                     PYSIGNAL("requestBookmark"),
                     PYSIGNAL("requestAppendBookmark"))
        self.connect(self._bookmark_list_widget,
                     PYSIGNAL("requestOpenMessages"),
                     PYSIGNAL("requestOpenMessages"))
        self.connect(self._bookmark_list_widget,
                     PYSIGNAL("requestRemoveBookmark"),
                     PYSIGNAL("requestRemoveBookmark"))
        self.connect(self._topic_list_widget,
                     PYSIGNAL("requestAppendBookmark"),
                     PYSIGNAL("requestAppendBookmark"))

    def _initial_config(self):
        self.setCaption("KittyWalk Qt")
        self.setCentralWidget(self._main_splitter)
        self._right_splitter.setOrientation(Qt.Vertical)
        self.setIcon(QPixmap(os.path.join(ICONPATH, "kittywalk.png")))
        self._bbs_widget.add(self._board_tree_widget, u("２ちゃんねる"))
        self._bbs_widget.add(self._bookmark_list_widget, u("ブックマーク"))
        self._bbs_widget.raiseWidget(0)
        self._topic_list_widget.hide()
        self._message_widget.hide()
        self._actions["toggle-bbs-widget"].setOn(True)
        self._actions["toggle-topic-list-widget"].setOn(True)
        self._actions["toggle-message-widget"].setOn(True)
        self.resize(config.get("gui-mainwindow-width", 640),
                    config.get("gui-mainwindow-height", 480))
        self._main_splitter.setSizes(config.get("gui-mainwidget-sizes",[1,4]))
        self._right_splitter.setSizes(
          config.get("gui-rightpane-sizes",[1,4]))

    def echo(self, text):
        self.statusBar().message(text)

    def display_bookmark_list(self, bookmark):
        self._bookmark_list_widget.display(bookmark)

    def display_topic_list(self, board, topics, newtab):
        self._topic_list_widget.display(board, topics, newtab)
        self._actions["toggle-topic-list-widget"].setOn(True)

    def display_messages(self, topic, messages, newtab):
        self._message_widget.display(topic, messages, newtab)
        self._actions["toggle-message-widget"].setOn(True)

    def set_message_font(self, font):
        self._message_widget.set_font(font)

    def _toggle_bbs_widget(self, on):
        self._toggle_widget(self._bbs_widget, on)

    def _toggle_topic_list_widget(self, on):
        self._toggle_widget(self._topic_list_widget, on)

    def _toggle_message_widget(self, on):
        self._toggle_widget(self._message_widget, on)

    def _toggle_widget(self, widget, on):
        if on:
            widget.show()
        else:
            widget.hide()

    def _reload_board_tree(self):
        self.emit(PYSIGNAL("requestReloadBoardTree"), ())

    def close(self, alsoDelete):
        config.set("gui-mainwindow-width", self.width())
        config.set("gui-mainwindow-height", self.height())
        config.set("gui-mainwidget-sizes", self._main_splitter.sizes())
        config.set("gui-rightpane-sizes", self._right_splitter.sizes())
        return QWidget.close(self, alsoDelete)

    def _change_font(self):
        self.emit(PYSIGNAL("requestChangeFont"),
                  (self._message_widget.font(),))


def show_about_dialog():
    caption = u("KittyWalk について")
    text = "\n\n".join(["KittyWalk " + kittywalk_version(),
                        "http://kittywalk.sourceforge.jp",
                        "onefifty@users.sourceforge.jp"])
    QMessageBox.about(None, caption, text)

