# -*- coding: utf8 -*-

import os
from Queue import Queue

from qt import *

from bbs import Topic
from renderer import Renderer
from kittywidget import *
from utils import *
import config


class MessageWidget(KittyTabWidget):

    _position_file = os.path.join(config.RCDIR, "gui-messagewidget-pos")
    _tab_menu_spec = ((u("更新"),   "requestReload"),
                      (u("ブックマークする"), "requestBookmark"),
                      (u("閉じる"), "requestClose"))

    def __init__(self, parent):
        KittyTabWidget.__init__(self, parent)
        self._font = QFont(*config.get("gui-message-font", ()))
        self._view_pos = MarshalDict(self._position_file)
        self._renderer = Renderer()
        # Connect
        self.connect(self,
                     PYSIGNAL("tabRightPressed"),
                     self._tab_menu.popup)
        self.connect(self,
                     PYSIGNAL("tabMidPressed"),
                     self._close_pointed_page)
        self.connect(self,
                     PYSIGNAL("tabLeftDoubleClicked"),
                     self._reload_current_page)
        self.connect(self._tab_menu, PYSIGNAL("requestClose"), self._close)
        self.connect(self._tab_menu,
                     PYSIGNAL("requestReload"),
                     self._reload_pointed_page)
        self.connect(self._tab_menu, PYSIGNAL("requestBookmark"),
                     self._bookmark_current_topic)

    def display(self, topic, messages, newtab=None):
        """Interface to display messages in topic
        
        This method actually submit messages into the local queue.  If 
        newtab is True, the new tab will be created for this topic.
        """
        assert isinstance(topic, Topic)
        if newtab==None:
            newtab = self._newtab
        self._queue.put((topic, messages, newtab))

    def set_font(self, font):
        self._font = font
        for view in self._model_view_registry.views():
            view.setFont(self._font)
            view.setText(view.text())
        config.set("gui-message-font",
                   (unicode(font.family()), font.pointSize()))

    def font(self):
        return self._font

    def _display(self, topic, messages, newtab):
        """Display the messages in topic
        
        If this window is already displaying the topic, the tab is
        raised.  Otherwise, create a new tab and display the list in
        the tab.
        """
        # If topic is being shown, use existing tab. Otherwise, create
        # new tab.
        if self._model_view_registry.has_model(topic):
            view = self._model_view_registry.view(topic)
            self._save_position(view)
        elif newtab==False and self._model_view_registry.size()>0:
            view = self.currentPage()
            self.changeTab(view, topic.title())
            self._save_position(view)
            self._model_view_registry.remove(self.currentPage())
        else:
            view = KittyTextBrowser(self)
            view.setFont(self._font)
            self.connect(view,
                         SIGNAL("highlighted(const QString &)"),
                         self._hover)
            self.addTab(view, topic.title())
            pass

        self._model_view_registry.set(topic, view)
        self.showPage(view)
        view.setText(self._renderer.render(topic, messages))
        self._restore_position(view)

    def _close_pointed_page(self, page, event):
        """Close page pointed by mouse cursor"""
        self._close(page)

    def _close(self, page_widget):
        """Close tab that contains page_widget

        In addition to call removePage() method, this method removes the 
        page_widget from model-view registry.
        """
        self._save_position(page_widget)
        self.removePage(page_widget)
        self._model_view_registry.remove(page_widget)
        page_widget.close()

    def _reload_pointed_page(self, page):
        self._reload(self._model_view_registry.model(page))

    def _reload_current_page(self, page):
        self._reload(self._current_topic())

    def _reload(self, topic):
        if topic!=None:
            self.emit(PYSIGNAL("requestReloadMessages"), (topic,))

    def _current_topic(self):
        widget = self.currentPage()
        return self._model_view_registry.model(widget)

    def _hover(self, href):
        pass

    def _save_position(self, view):
        topic = self._model_view_registry.model(view)
        self._view_pos[topic.id()] = (view.contentsX(),view.contentsY())

    def _restore_position(self, view):
        topic = self._model_view_registry.model(view)
        view.setContentsPos(0,0)
        view.scrollBy(*self._view_pos.get(topic.id(), (0,0)))

    def _bookmark_current_topic(self, view):
        topic = self._model_view_registry.model(view)
        self.emit(PYSIGNAL("requestBookmark"), (topic,))

