#!/usr/bin/perl
# ----------------------------------------------------------------------------
#
# komazou.pl --- Photozou Uploader Command Line Version
#
# This program uploads pictures from command line to
# Photozou(http://photozou.jp), a picture Social Networking Service.
# Photozou is a trademark of Unoh Inc.(http://www.unoh.net)
#
# Copyright (c) 2005 Junya Ishihara <jishiha@users.sourceforge.jp>
#
#   This file is part of Komazou
#
#   Komazou is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   Komazou is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with Komazou; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
#   Modified by    Junya Ishihara <jishiha@users.sourceforge.jp>
#
#   Special thanks to:
#   - Hiroshi Yuki (The original code of this program is very much based on
#  Hatena Diary Writer(http://www.hyuki.com/techinfo/hatena_diary_writer.html))
#
# ----------------------------------------------------------------------------

use strict;
my $VERSION = "0.2";

use LWP::UserAgent;
use HTTP::Request::Common;
use HTTP::Cookies;
use File::Basename;
use Getopt::Std;

#my $enable_encode = eval('use Encode; 1');
my $enable_encode = 1;
use Encode;
use Encode::JP;

# Prototypes.
sub login();
sub logout();
sub upload($$$$$$$$$);
sub get_list($);
sub convert_encoding($);
sub check_version();
sub print_debug(@);
sub print_message(@);
sub error_exit(@);
sub load_config();

# Photozou mail address (if empty, I will ask you later).
my $mail_address = '';
# Photozou password (if empty, I will ask you later).
my $password = '';

# Default file names.
my $cookie_file = 'cookie.txt';
my $config_file = 'config.txt';

# Photozou URL.
my $photozou_url = 'http://photozou.jp';
my $photozou_api_url = 'http://photozou.jp/photo_uploader.php';

# Komazou URL.
my $komazou_version_url = 'http://komazou.sourceforge.jp/version.txt';

# Option for LWP::UserAgent.
my %ua_option = (
    agent => "Komazou/$VERSION",
    timeout => 180,
);

# Other variables.
my $cookie_jar;
my $user_agent;
#my $dummyinput = '';
my $dummyinput = '';
my $title;
my $album;
my $group;
my $album_id;
my $perm_type = 'allow';
my $perm_type2 = 'all';
my $perm_id;

my $client_encoding = 'Shift_JIS';
my $server_encoding = 'EUC-JP';

my %album_hash;
my %group_hash;

my ($day, $month, $year) = (localtime)[3,4,5];

# Handle command-line option.
my %cmd_opt = (
    'd' => 0,   # "debug" flag.
    'm' => "",  # "mail_address" option.
    'p' => "",  # "password" option.
    'A' => "",  # "agent" option.
    'T' => "",  # "timeout" option.
    'c' => 0,   # "cookie" flag.
    'n' => "",  # "config file" option.
    't' => "",  # "title" option.
    'a' => "",  # "album" option.
    'g' => ""   # "group" option.
);

$Getopt::Std::STANDARD_HELP_VERSION = 1;
getopts("dm:p:A:T:cn:t:a:g:", \%cmd_opt) or error_exit("Unknown option.");

if ($cmd_opt{d}) {
    print_debug("Debug flag on.");
    print_debug("Cookie flag on.") if $cmd_opt{c};
    &VERSION_MESSAGE();
}

# Override config file name (before load_config).
$config_file = $cmd_opt{n} if $cmd_opt{n};

# Override global vars with config file.
load_config() if -e($config_file);

# Override global vars with command-line options.
$mail_address = $cmd_opt{m} if $cmd_opt{m};
$password = $cmd_opt{p} if $cmd_opt{p};
$ua_option{agent} = $cmd_opt{A} if $cmd_opt{A};
$ua_option{timeout} = $cmd_opt{T} if $cmd_opt{T};
$title = $cmd_opt{t} if $cmd_opt{t};
$album = $cmd_opt{a} if $cmd_opt{a};
$group = $cmd_opt{g} if $cmd_opt{g};

# Start.
&main;

# no-error exit.
exit(0);

# Main sequence.
sub main {
    my $count = 0;
    my @files = @ARGV;
    my ($basename, $dir, $ext);

    $user_agent = LWP::UserAgent->new(%ua_option);

    # Check the latest version of komazou.
    check_version() if ($user_agent);

    # Login if necessary.
#    login() unless ($user_agent);
    login() if ($user_agent);

    # Get album/group hashes from album/group lists.
    %album_hash = get_list("photo_album") if ($album);
    %group_hash = get_list("user_group") if ($group);

    # Convert encodings of album name and group name.
    my ($org_album, $org_group) = ($album, $group);

    print_debug("Convert encoding of album name.");
    $album = convert_encoding($album);
    print_debug("Convert encoding of group name.");
    $group = convert_encoding($group);

    if ($album and ($album_hash{$album} == "")) {
        print_message("$org_album is not found in your album list. Will upload the picture to 'Unspecified' album.");
    }

    if ($group and ($group_hash{$group} == "")) {
        print_message("$org_group is not found in your group list. Will upload the picture with 'Allow to all' permission.");
    }

    # Set album id and group id.
    $album_id = $album_hash{$album};
    $perm_id = $group_hash{$group};
    print_debug("main: \$album_id = $album_id, \$perm_id = $perm_id.");

    # Process it.
    my $file;
    foreach $file (@files) {

        # Parse file name.
        ($basename, $dir, $ext) = fileparse($file, '\..*');
        print_debug("main: Parse file. \$basename: $basename, \$dir: $dir, \$ext: $ext.");

        # Check file name.
#        next unless ($file =~ /.+\.jpg$/);

        # Check if it is a file.
        next unless (-f $file);

        sleep(1);

        unless ($title) {
            print "Title \[$basename\]: ";
            chomp($title = <STDIN>);
            $title = $basename if ($title eq '');

            $title = convert_encoding($title);
        }

        $perm_type2 = 'user_group' if $perm_id;

        print_debug("main: upload($file, $album_id, $title, $year, $month, $day, $perm_type, $perm_type2, $perm_id)");
        upload($file, $album_id, $title, $year, $month, $day, $perm_type, $perm_type2, $perm_id);

        $count++;
    }

    # Logout if necessary.
    logout() if ($user_agent);

    print_message("No files are posted.") if ($count == 0);
}

# Login using photozou simple API.
sub login() {
    my $login_url = $photozou_api_url;

    # Ask username if not set.
    unless ($mail_address) {
        print "Mail Address: ";
        chomp($mail_address = <STDIN>);
    }

    # Ask password if not set.
    unless ($password) {
        print "Password: ";
        chomp($password = <STDIN>);
    }

    my %form;

    $form{mode} = 'login';
    $form{email} = $mail_address;
    $form{password} = $password;

    my $r; # Response.

    print_message("Login to $photozou_url as $form{email}.");

    $r = $user_agent->simple_request(
        HTTP::Request::Common::POST("$login_url", \%form, 'Content_Type' => 'multipart/form-data')
    );

    print_debug("login: " . $r->status_line);

    print_debug("login: \$r = " . $r->content());

    unless ($r->content() =~ /0 Succeeded to login/) {
        error_exit("login: Unexpected response: ", $r->status_line);
    }

    print_message("Login OK.");

    print_debug("login: Making cookie jar.");

    $cookie_jar = HTTP::Cookies->new;
    $cookie_jar->extract_cookies($r);
    $cookie_jar->save($cookie_file) if $cmd_opt{c};

    print_debug("login: \$cookie_jar = " . $cookie_jar->as_string);
}

# Logout using photozou simple API.
sub logout() {
    $user_agent->cookie_jar($cookie_jar);

    my $logout_url = $photozou_api_url;

    my %form;

    $form{mode} = 'logout';

    my $r; # Response.

    print_message("Logout from $photozou_url.");

    $r = $user_agent->simple_request(
        HTTP::Request::Common::POST("$logout_url", \%form, 'Content_Type' => 'multipart/form-data')
    );

    print_debug("logout: " . $r->status_line);

    print_debug("logout: \$r = " . $r->content());

    unless ($r->content() =~ /0 Succeeded to logout/) {
        error_exit("logout: Unexpected response: ", $r->status_line);
    }

    print_message("Logout OK.");

    print_debug("logout: returns 1 (OK).");
    return 1;
}

# Upload picture using photozou simple API.
sub upload($$$$$$$$$) {
    my ($file, $album_id, $title, $year, $month, $day, $perm_type, $perm_type2, $perm_id) = @_;

    print_debug("upload: upload($file, $album_id, $title, $year, $month, $day, $perm_type, $perm_type2, $perm_id)");

    print_message("Uploading...");

    $user_agent->cookie_jar($cookie_jar);

    my $uploadurl = $photozou_api_url;

    my $r = $user_agent->simple_request(
        HTTP::Request::Common::POST($uploadurl,
            Content_Type => 'multipart/form-data',
            Content => [
                dummyinput => $dummyinput,
                mode => 'add_photo',
                album_id => $album_id,
                photo_title => $title,
                photo => [$file],
                year => $year + 1900,
                month => $month + 1,
                day => $day,
                perm_type => $perm_type,
                perm_type2 => $perm_type2,
                perm_id => $perm_id
            ]
        )
    );

    print_debug("upload: ", $r->status_line);

    print_debug("upload: ", $r->content());

    unless ($r->content() =~ /0 Succeeded to regist/) {
        error_exit("upload: Unexpected response: ", $r->status_line);
    }

    print_message("Upload OK.");

    print_debug("upload: returns 1 (OK).");
    return 1;
}

# Fetch object name and id from the list, then return hash.
# @object{$object_name} = $object_id
sub get_list($) {
    my ($list_name) = @_;
    my %object;

    print_debug("get_list: Getting $list_name list.");

    $user_agent->cookie_jar($cookie_jar);

    my $get_list_url = $photozou_api_url;

    my %form;

    $form{mode} = 'get_'.$list_name.'_list';

    my $r; # Response.

    $r = $user_agent->simple_request(
        HTTP::Request::Common::POST("$get_list_url", \%form, 'Content_Type' => 'multipart/form-data')
    );

    print_debug("get_list: " . $r->status_line);

    print_debug("get_list: \$r = " . $r->content());

    unless ($r->content() =~ /0 Succeeded/) {
        error_exit("get_list: Unexpected response: ", $r->status_line);
    }

    my $line;
    foreach $line (split(/\n/, $r->content())){
        next if ($line =~ /0 Succeeded/);
        if ($line =~ /^(\d+) (.*)$/) {
            $object{$2} = $1;
        }
    }

    # Return hash. @object{$object_name} = $object_id
    return %object;
}

# Convert encoding.
sub convert_encoding($) {
    my ($char) = @_;

    if ($enable_encode and ($client_encoding ne $server_encoding))  {
        print_debug("convert_encoding: Convert from $client_encoding to $server_encoding.");
        print_debug("convert_encoding: Before: $char");

        Encode::from_to($char, $client_encoding, $server_encoding);

        print_debug("convert_encoding: After: $char");
    }

    return $char;
}

sub check_version() {
    my $url = $komazou_version_url;
    my ($major_ver, $minor_ver);
    my ($latest_major_ver, $latest_minor_ver);
    my $ver_index;
    my $latest_ver_index;

    print_debug("check_version: Check the current version of komazou.");

    ($major_ver, $minor_ver) = split (/\./, $VERSION);
    $ver_index = $major_ver * 100 + $minor_ver;

    print_debug("check_version: \$major_ver=$major_ver, \$minor_ver=$minor_ver, \$ver_index=$ver_index");

    print_debug("check_version: Check the latest version of komazou.");

    my $r = $user_agent->get("$url");
    print_debug("check_version: " . $r->status_line);
    print_debug("check_version: " . $r->content());

    if ($r->content() =~ /version:(\d+)\.(\d+)/) {
        ($latest_major_ver, $latest_minor_ver) = ($1, $2);
        $latest_ver_index = $latest_major_ver * 100 + $latest_minor_ver;
        print_debug("check_version: \$latest_ver_index = $latest_ver_index");
        if ($latest_ver_index > $ver_index) {
            print_message("### Newer version, Komazou $1.$2, is available.");
            print_message("### Please check http://komazou.sourceforge.jp");
        }
    } else {
        print_message("WARNING: Can't get the version info from $url.");
    }
}

# Show version message. This is called by getopts.
sub VERSION_MESSAGE {
    print <<"EOD";
Komazou Version $VERSION
Copyright (C) 2005 Junya Ishihara

Komazou comes with ABSOLUTELY NO WARRANTY.
This is free software, and you are welcome to redistribute it
under certain conditions; see the file named LICENSE in the Komazou
ditribution for details.
EOD
}

# Debug print.
sub print_debug(@) {
    if ($cmd_opt{d}) {
        print "DEBUG: ", @_, "\n";
    }
}

# Print message.
sub print_message(@) {
    print @_, "\n";
}

# Error exit.
sub error_exit(@) {
    print "ERROR: ", @_, "\n";
    unlink($cookie_file);
    exit(1);
}

# Show help message. This is called by getopts.
sub HELP_MESSAGE {
    my $command;

    if ($0 =~ /komazou.pl/) {
        $command = "perl $0";
    }else{
        $command = "$0";
    }

    print <<"EOD";

Usage: $command [Options] [Picture File]

Options:
    --version       Show version.
    --help          Show this message.
    -d              Debug. Use this switch for verbose log.
    -m mail_address Mail address. Specify mail address.
    -p password     Password. Specify password.
    -A agent        User agent. Default value is Komazou/$VERSION.
    -T seconds      Timeout. Default value is 180.
    -c              Cookie. Skip login/logout if $cookie_file exists.
    -n config_file  Config file. Default value is $config_file.
    -t title        Title. Specify the title of the picture.
    -a album        Album. Specify the album that the picture belongs to.
    -g group        Group. Specify the group you give the permission to access the picture.

Config file example:
#
# $config_file
#
mail_address:yourmailaddress
password:yourpassword
cookie:cookie.txt
title:title
album:albumname
group:groupname
client_encoding:Shift_JIS
server_encoding:EUC-JP
EOD
}

# Load config file.
sub load_config() {
    print_debug("load_config: Loading config file ($config_file).");
    if (not open(CONF, $config_file)) {
        error_exit("Can't open $config_file.");
    }
    while (<CONF>) {
        chomp;
        if (/^\#/) {
            # skip comment.
        } elsif (/^$/) {
            # skip blank line.
        } elsif (/^mail_address:([^:]+)$/) {
            $mail_address = $1;
            print_debug("load_config: mail_address:$mail_address");
        } elsif (/^password:(.*)$/) {
            $password = $1;
            print_debug("load_config: password:********");
        } elsif (/^cookie:(.*)$/) {
            $cookie_file = glob($1);
            $cmd_opt{c} = 1; # If cookie file is specified, Assume '-c' is given.
            print_debug("load_config: cookie:$cookie_file");
        } elsif (/^title:(.*)$/) {
            $title = $1;
            print_debug("load_config: title:$title");
        } elsif (/^album:(.*)$/) {
            $album = $1;
            print_debug("load_config: album:$album");
        } elsif (/^group:(.*)$/) {
            $group = $1;
            print_debug("load_config: group:$group");
        } elsif (/^client_encoding:(.*)$/) {
            $client_encoding = $1;
            print_debug("load_config: client_encoding:$client_encoding");
        } elsif (/^server_encoding:(.*)$/) {
            $server_encoding = $1;
            print_debug("load_config: server_encoding:$server_encoding");
        } else {
            error_exit("Unknown command '$_' in $config_file.");
        }
    }
    close(CONF);
}
__END__
