/****************************************************************************
 * KONOHA COPYRIGHT, LICENSE NOTICE, AND DISCRIMER
 *
 * Copyright (c)  2010-      Konoha Team konohaken@googlegroups.com
 * All rights reserved.
 *
 * You may choose one of the following two licenses when you use konoha.
 * See www.konohaware.org/license.html for further information.
 *
 * (1) GNU Lesser General Public License 3.0 (with KONOHA_UNDER_LGPL3)
 * (2) Konoha Software Foundation License 1.0
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************/

// **************************************************************************
// LIST OF CONTRIBUTERS
//
// **************************************************************************


#include <oniguruma.h>
#include <konoha.h>

#ifdef __cplusplus 
extern "C" {
#endif

typedef struct onig_regex_t {
	regex_t *reg;
	OnigErrorInfo *einfo;
} onig_regex_t;

static knh_regex_t* onig_regex_malloc(Ctx *ctx, knh_String_t *s)
{
	onig_regex_t *r = (onig_regex_t*) KNH_MALLOC(ctx, sizeof(onig_regex_t));
	r->einfo = NULL; // memset
	r->reg = NULL;
	return (knh_regex_t*) r;
}

static int onig_regex_parse_cflags(Ctx *ctx, const char *option)
{
	//fprintf(stderr, "parse_compile_option = '%s'\n", option);
	OnigOptionType cflags = ONIG_OPTION_DEFAULT;
	int i, optlen = strlen(option);
	for (i = 0; i < optlen; i++) {
		switch(option[i]) {
		case 'i':
			ONIG_OPTION_ON(cflags, ONIG_OPTION_IGNORECASE);
			break;
		case 'm':
			ONIG_OPTION_ON(cflags, ONIG_OPTION_MULTILINE);
			break;
		case 'x':
			ONIG_OPTION_ON(cflags, ONIG_OPTION_EXTEND);
			break;
		default: break;
		}
	}
	return (int)cflags;
}

static int onig_regex_parse_eflags(Ctx *ctx, const char *option)
{
	//fprintf(stderr, "parse_exec_option = '%s'\n", option);
	OnigOptionType eflags = ONIG_OPTION_NONE;
	int i, optlen = strlen(option);
	for (i = 0; i < optlen; i++) {
		switch(option[i]) {
		case 'g':
			ONIG_OPTION_ON(eflags, ONIG_OPTION_NOTBOL);
			break;
		default: break;
		}
	}	
	return eflags;
}

static size_t onig_regex_regerror(int res, knh_regex_t *reg, char* ebuf, size_t ebuf_size)
{
	onig_regex_t *oreg = (onig_regex_t*)reg;
	return onig_error_code_to_str((UChar*)ebuf, res, oreg->einfo);
}

static int onig_regex_regcomp(Ctx *ctx, knh_regex_t *reg, const char *pattern, int cflag)
{ 
	//fprintf(stderr, "pattern = '%s'\n",pattern);
	onig_regex_t *oreg = (onig_regex_t*)reg;
	UChar* upatt = (UChar*) pattern;
	UChar* end = upatt + strlen(pattern);
	int cmp = onig_new(&(oreg->reg), upatt, end, cflag,
					   ONIG_ENCODING_UTF8, ONIG_SYNTAX_DEFAULT, oreg->einfo);
	return (cmp == ONIG_NORMAL) ? 0 : 1;
}

static void onig_regex_regmatch_init(knh_regmatch_t p[], size_t nmatch)
{
	size_t i;
	for (i = 0; i < nmatch; i++) {
		p[i].rm_so = -1;
		p[i].rm_eo = -1;
		p[i].rm_name.text = NULL;
		p[i].rm_name.len = 0;
	}
}

static void onig_regex_regmatch_copy_from_region(OnigRegion *region, knh_regmatch_t p[], size_t nmatch, size_t poffs)
{
	size_t i, j, mlen = region->num_regs;
	for (i = poffs, j = 0; i < nmatch && j < mlen; i++, j++) {
		p[i].rm_so = region->beg[j];
		p[i].rm_eo = region->end[j];
		p[i].rm_name.text = NULL;
		p[i].rm_name.len = 0;
		//fprintf(stderr, "%zd: (%d, %d)\n", i, region->beg[j], region->end[j]);
	}
}

static int onig_regex_regexec(Ctx *ctx, knh_regex_t *reg, const char *str, size_t nmatch, knh_regmatch_t p[], int eflag)
{
	//fprintf(stderr, "souce='%s'\n", str);
	onig_regex_t *oreg = (onig_regex_t*)reg;
	UChar* ustr = (UChar*) str;
	UChar* end = ustr + strlen(str);
	OnigRegion* region = onig_region_new();
	onig_regex_regmatch_init(p, nmatch);
	int res = 0;
	if (eflag & ONIG_OPTION_NOTBOL) {
		/* global loop */
		ONIG_OPTION_OFF(eflag, ONIG_OPTION_NOTBOL);
		UChar *head = ustr;
		size_t poffs = 0;
		while (res >= 0) {
			//fprintf(stderr, "%zd, -> %s\n", poffs, head);
			res = onig_search(oreg->reg, ustr, end, head, end, region, eflag);
			if (res >= 0) {
				onig_regex_regmatch_copy_from_region(region, p, nmatch, poffs);
				poffs++;
				head = ustr + p[poffs-1].rm_eo;
				onig_region_clear(region);
			}
		}
		if (res == ONIG_MISMATCH) res = 0;
	}
	else {
		res = onig_search(oreg->reg, ustr, end, ustr, end, region, eflag);
		if (res >= 0) {
			onig_regex_regmatch_copy_from_region(region, p, nmatch, 0);
		}
	}
	onig_region_free(region, 1); /* 1:free self, 0:free contents only */
	return (res >= 0) ? 0 : res; /* >=0: not error(matched bytes), <0:error */
}

static void onig_regex_regfree(Ctx *ctx, knh_regex_t *reg)
{
	onig_regex_t *oreg = (onig_regex_t*)reg;
	regex_t *r = oreg->reg;
	onig_free(r);
	KNH_FREE(ctx, oreg, sizeof(onig_regex_t));
}

static knh_RegexSPI_t ONIG_REGEXSPI = {
	"oniguruma",
	onig_regex_malloc,
	onig_regex_parse_cflags,
	onig_regex_parse_eflags,
	onig_regex_regcomp,
	onig_regex_regexec,
	onig_regex_regerror,
	onig_regex_regfree
};

KNHAPI(int) kcheck(void)
{
	return K_BUILDID;
}

KNHAPI(void) init(Ctx *ctx, const knh_PackageLoaderAPI_t *kapi, char *dname, int isOVERRIDE)
{
	kapi->setRegexSPI(ctx, &ONIG_REGEXSPI);
}

#ifdef __cplusplus
}
#endif
