/*
 call table redirection
 Copyright (c) 2005,2006 Hitachi,Ltd.,
 Created by Satoru Moriya <s-moriya@sdl.hitachi.co.jp>
 Updated by Masami Hiramatsu <hiramatu@sdl.hitachi.co.jp>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <linux/module.h>
#include <asm/unistd.h>
#include "ctr.h"
#include "kstrax_ioc.h"
#include "kstrax_syscall_list.h"

/*
 *  global variable
 */
static syscall_handler_t pre_handler = NULL;
static syscall_handler_t post_handler = NULL;
static call_table_t *true_sys_call_addr = NULL;
call_table_t true_sys_call_table[512] = { NULL, };

/*
 *  global variable and parameter
 */
static unsigned long addr;
module_param(addr, ulong, 0444);
MODULE_PARM_DESC(addr, "Address of system call table(sys_call_table).");

static void trace_all_syscall(call_table_t *);
static void trace_file_syscall(call_table_t *);
static void trace_network_syscall(call_table_t *);
static void trace_ipc_syscall(call_table_t *);
static void trace_process_syscall(call_table_t *);
static void trace_signal_syscall(call_table_t *);

extern int sys_call_stub(void);

void pre_sys_call(const struct pt_regs regs)
{
	if (pre_handler)
		pre_handler(regs);
}

void post_sys_call(const struct pt_regs regs)
{
	if (post_handler)
		post_handler(regs);
}

static int change_sys_call_table(call_table_t *saddr, int type)
{
	void (*change_each_entry[])(call_table_t *) = {trace_all_syscall,
						       trace_file_syscall,
						       trace_network_syscall,
						       trace_ipc_syscall,
						       trace_process_syscall,
						       trace_signal_syscall};
	if (true_sys_call_addr != NULL)
		return -1;

	true_sys_call_addr = saddr;
	(*change_each_entry[type])(saddr);
	return 0;
}

static void restore_sys_call_table(void)
{
	int i;
	call_table_t *sys_call_table;

	if (true_sys_call_addr == NULL)
		return;
	sys_call_table = (void *)true_sys_call_addr;

	for (i = 0; i < NR_syscalls; i++)
		sys_call_table[i] = true_sys_call_table[i];

	true_sys_call_addr = NULL;
}


static void trace_all_syscall(call_table_t *saddr)
{
	int i;
	for (i = 0; i < NR_syscalls; i++) {
		true_sys_call_table[i] = saddr[i];
		saddr[i] = (call_table_t)sys_call_stub;
	}
}

static void trace_file_syscall(call_table_t *saddr)
{
	int i;
	for (i = 0; i < NR_syscalls; i++) {
		true_sys_call_table[i] = saddr[i];
		if (i == OPEN || i == CREAT || i == LINK || i == UNLINK || 
		    i == EXECVE || i == CHDIR || i == MKNOD || i == CHMOD || 
		    i == LCHOWN16 || i == STAT || i == MOUNT || i == OLDUMOUNT || 
		    i == UTIME || i == ACCESS || i == RENAME || i == MKDIR || 
		    i == RMDIR || i == ACCT || i == UMOUNT || i == CHROOT || 
		    i == SYMLINK || i == LSTAT || i == READLINK || i == USELIB ||
		    i == SWAPON || i == TRUNCATE || i == STATFS || i ==NEWSTAT || 
		    i == NEWLSTAT || i == PREAD64 || i == PWRITE64 ||
		    i == CHOWN16 || i == GETCWD || i == SENDFILE ||
		    i == TRUNCATE64 || i == FTRUNCATE64 || i == STAT64 || 
		    i == LSTAT64 || i == FSTAT64 || i == LCHOWN || i == CHOWN || 
		    i == PIVOT_ROOT || i == SETXATTR || i == LSETXATTR || 
		    i == GETXATTR || i == LGETXATTR || i == LISTXATTR ||
		    i == LLISTXATTR || i == REMOVEXATTR || i == LREMOVEXATTR || 
		    i == SENDFILE64 || i == FADVISE64 || i == STATFS64 ||
		    i == FSTATFS64 || i == UTIMES || i == FADVISE64_64)
			saddr[i] = (call_table_t)sys_call_stub;
	}
}

static void trace_network_syscall(call_table_t *saddr)
{
	int i;
	for (i = 0; i < NR_syscalls; i++) {
		true_sys_call_table[i] = saddr[i];
		if (i == SOCKETCALL)
			saddr[i] = (call_table_t)sys_call_stub;
	}
}

static void trace_ipc_syscall(call_table_t *saddr)
{
	int i;
	for (i = 0; i < NR_syscalls; i++) {
		true_sys_call_table[i] = saddr[i];
		if (i == IPC)
			saddr[i] = (call_table_t)sys_call_stub;
	}
}

static void trace_process_syscall(call_table_t *saddr)
{
	int i;
	for (i = 0; i < NR_syscalls; i++) {
		true_sys_call_table[i] = saddr[i];
		if (i == EXIT || i == FORK || i == WAITPID || i == EXECVE ||
		    i == WAIT4 || i == CLONE || i == VFORK || i == WAITID)
			saddr[i] = (call_table_t)sys_call_stub;
	}
}

static void trace_signal_syscall(call_table_t *saddr)
{
	int i;
	for (i = 0; i < NR_syscalls; i++) {
		true_sys_call_table[i] = saddr[i];
		if (i == PAUSE || i == KILL || i == SIGNAL || i == SIGACTION ||
		    i == SGETMASK || i == SSETMASK || i == SIGSUSPEND ||
		    i == SIGPENDING || i == SIGRETURN || i == SIGPROCMASK ||
		    i == RT_SIGRETURN || i == RT_SIGACTION || i == RT_SIGPROCMASK ||
		    i == RT_SIGPENDING || i == RT_SIGTIMEDWAIT || 
		    i == RT_SIGSUSPEND || i == RT_SIGQUEUEINFO || 
		    i == TKILL || i == TGKILL || i == SIGALTSTACK)
			saddr[i] = (call_table_t)sys_call_stub;
	}
}


/*------------------------------------------------------------------------
 *  module interface
 */
int register_ctr(syscall_handler_t pre, syscall_handler_t post)
{
	if (pre == NULL && post == NULL)
		return -EINVAL;
	pre_handler = pre;
	post_handler = post;
	return 0;
}

void unregister_ctr(void)
{
	pre_handler = NULL;
	post_handler = NULL;
}

int kstrax_syscall_spec(int type)
{
	/* check parameter */
	if ((void *)addr == NULL) {
		printk(KERN_ERR "error(init):Invalid parameter\n");
		return -EINVAL;
	}

	restore_sys_call_table();

	/* call table redirection */
	if (change_sys_call_table((void *)addr, type) != 0) {
		printk(KERN_ERR "error:Failed to change system call table\n");
		return -EINVAL;
	}
	return 0;
}

/*------------------------------------------------------------------------
 *  module init/exit
 */

#ifdef SPLIT_MODULE
#define CTR_INIT static int __init
#define CTR_EXIT static void
#else	/* SPLIT_MODULE */
#define CTR_INIT int
#define CTR_EXIT void
#endif	/* SPLIT_MODULE */

CTR_EXIT exit_ctr(void)
{
	restore_sys_call_table();
}

CTR_INIT init_ctr(void)
{
	/* check parameter */
	if ((void *)addr == NULL) {
		printk(KERN_ERR "error(init):Invalid parameter\n");
		return -EINVAL;
	}

	/* call table redirection */
	if (change_sys_call_table((void *)addr, TRACE_ALL) != 0) {
		exit_ctr();
		printk(KERN_ERR "error(init):Failed to change system call table\n");
		return -EINVAL;
	}
	try_module_get(THIS_MODULE);

	return 0;
}

#ifdef SPLIT_MODULE
module_init(init_ctr);
module_exit(exit_ctr);
MODULE_AUTHOR("S.Moriya <s-moriya@sdl.hitachi.co.jp>");
MODULE_LICENSE("GPL");
EXPORT_SYMBOL_GPL(register_ctr);
EXPORT_SYMBOL_GPL(unregister_ctr);
EXPORT_SYMBOL_GPL(kstrax_syscall_spec);
#endif	/*SPLIT_MODULE */
