/*
 mcontrol -- control module(event handler)
 Copyright (c) 2005,2006 Hitachi,Ltd.,
 Created by Satoru Moriya <s-moriya@sdl.hitachi.co.jp>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <linux/module.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <asm/atomic.h>
#include <asm/uaccess.h>
#include "ctr.h"
#include "mbuffer.h"
#include "mcontrol.h"
#include "kstrax_ioc.h"

#define DEVICE_NAME		"kstrax"
#define VERSION                 "0.0.1"
#define	DEV_COUNT		1
#define KERNEL_BUFFER_ENTRY	4000 * 40  /* multiples of 128 */

/*
 *  global variable
 */
static dev_t kstrax_number;
static struct cdev cdev_kstrax;
static int flag_file_open = 0;
int kstrax_force_flag = 0;
pid_t kstrax_owner = -1;

/*
 *  function prototype declaration
 */
static int kstrax_open(struct inode *, struct file *);
static int kstrax_release(struct inode *, struct file *);
static int kstrax_ioctl(struct inode *, struct file *,
			 unsigned int, unsigned long);

static struct file_operations kstrax_file_operation = {
	.owner = THIS_MODULE,
	.open = kstrax_open,
	.release = kstrax_release,
	.read = kstrax_read,
	.ioctl = kstrax_ioctl,
};

static int setup_kstrax_cdev(void)
{
	cdev_init(&cdev_kstrax, &kstrax_file_operation);
	cdev_kstrax.owner = THIS_MODULE;
	return (cdev_add(&cdev_kstrax, kstrax_number, DEV_COUNT));
}

/*
 *  open
 */
static int kstrax_open(struct inode *inode, struct file *flip)
{
	/* only 1 program can open */
	if (flag_file_open > 0) {
		printk(KERN_ERR "error(file open):only 1 command can open\n");
		return -EBUSY;
	}
	flag_file_open++;
	kstrax_force_flag = 0;
	return 0;
}

/*
 *  release
 */
static int kstrax_release(struct inode *inode, struct file *flip)
{
	if (flag_file_open <= 0) {
		printk(KERN_ERR "error(file close):already closed\n");
		return -EINVAL;
	}
	flag_file_open--;
	return 0;
}

/*------------------------------------------------------------------------------
 *  ioctl
 */
static int kstrax_status_ioctl(trace_stat_t *);
static int kstrax_set_cpu_ioctl(int *);
static int kstrax_init_index_ioctl(void);
static int kstrax_read_basetime_ioctl(int *);
static int kstrax_set_owner_ioctl(pid_t *);
static int kstrax_clr_owner_ioctl(void);
static int kstrax_pid_spec_ioctl(pid_t *);
static int kstrax_syscall_spec_ioctl(int *);

static int kstrax_ioctl(struct inode *inode, struct file *flip,
			 unsigned int cmd, unsigned long arg)
{
	int retval = 0;

	switch (cmd) {
	case KSTRAX_IOC_STATUS:
		retval = kstrax_status_ioctl((trace_stat_t *)arg);
		break;
	case KSTRAX_IOC_SET_CPU:
		retval = kstrax_set_cpu_ioctl((int *)arg);
		break;
	case KSTRAX_IOC_SET_FORCE_WAKEUP:
		kstrax_force_flag = 1;
		break;
	case KSTRAX_IOC_INIT_INDEX:
		retval = kstrax_init_index_ioctl();
		break;
	case KSTRAX_IOC_READ_BASETIME:
		retval = kstrax_read_basetime_ioctl((int *)arg);
		break;
	case KSTRAX_IOC_SET_OWNER:
		retval = kstrax_set_owner_ioctl((pid_t *)arg);
		break;
	case KSTRAX_IOC_CLR_OWNER:
		retval = kstrax_clr_owner_ioctl();
		break;
	case KSTRAX_IOC_PID_SPEC:
		retval = kstrax_pid_spec_ioctl((pid_t *)arg);
		break;
	case KSTRAX_IOC_SYSCALL_SPEC:
		retval = kstrax_syscall_spec_ioctl((int *)arg);
		break;
	default:
		retval = -1;
	}
	return retval;

}

static int kstrax_status_ioctl(trace_stat_t *user_status)
{
	return (kstrax_copy_status(user_status));
}

static int kstrax_set_cpu_ioctl(int *user_cpu)
{
	int cpu;
	if (copy_from_user(&cpu, user_cpu, sizeof(int))) {
		printk(KERN_ERR "error(ioctl):cannot copy cpu infomation\n");
		return -EINVAL;
	}

	if (set_cpus_allowed(current, cpumask_of_cpu(cpu)) < 0) {
		return -EINVAL;
	}
	return 0;
}

static int kstrax_init_index_ioctl(void)
{
	return (kstrax_reset_index());
}

/* user_buf is not pointer to int but sys_call_t */
static int kstrax_read_basetime_ioctl(int *user_buf)
{
	return (kstrax_read_basetime(user_buf));
}


static int kstrax_set_owner_ioctl(pid_t *user_pid)
{
	if (copy_from_user(&kstrax_owner, user_pid, sizeof(pid_t))) {
		printk(KERN_ERR "error(ioctl):cannot copy owner information\n");
		return -EINVAL;
	}
	if (kstrax_owner < 0) {
		kstrax_owner = -1;
		return -EINVAL;
	}
	return 0;
}

static int kstrax_clr_owner_ioctl(void)
{
	kstrax_owner = -1;
	return 0;
}

static int kstrax_pid_spec_ioctl(pid_t *user_pid)
{
	pid_t pid;
	if (copy_from_user(&pid, user_pid, sizeof(pid_t))) {
		printk(KERN_ERR "error(ioctl):cannot copy owner information\n");
		return -EINVAL;
	}
	return kstrax_pid_spec(pid);
}

static int kstrax_syscall_spec_ioctl(int *user_type)
{
	int type;
	if (copy_from_user(&type, user_type, sizeof(int))) {
		printk(KERN_ERR "error(ioctl):cannot copy syscall spec information\n");
		return -EINVAL;
	}
	return kstrax_syscall_spec_wrapper(type);
}

/*-------------------------------------------------------------------------------------
 *  module init/exit
 */

MODULE_AUTHOR("S.Moriya <s-moriya@sdl.hitachi.co.jp>");
MODULE_LICENSE("GPL");

static void exit_kstrax(void)
{
	kstrax_pid_spec(-1);
	kstrax_syscall_spec_wrapper(TRACE_ALL);

	/* release  /dev/kstrax */
	cdev_del(&cdev_kstrax);
	
	/* release device number */
	unregister_chrdev_region(kstrax_number, DEV_COUNT);

	/* unregister save functions */
	unregister_ctr();

	synchronize_kernel();

	clear_mbuffer();
	
/* undo system call table */
#ifndef SPLIT_MODULE
	exit_ctr();
#endif  /* SPLIT MODULE */
}

/*
 *  module parameter
 */
static int kbuffer_entry = KERNEL_BUFFER_ENTRY;
module_param(kbuffer_entry, int, 0444);
MODULE_PARM_DESC(kbuffer_entry, "number of kernel buffer entry.");

static int __init init_kstrax(void)
{
	int retval = 0;

#ifndef SPLIT_MODULE
	if (init_ctr() < 0) {
		return -EINVAL;
	};
#endif /* SPLIT MODULE */
	
	retval = init_mbuffer(kbuffer_entry);
	if (retval < 0) {
		printk(KERN_ERR "error(init):cannot allocate buffer\n");
		goto fail;
	}
	
	/* register recording functions */
	retval = register_ctr(pre_sys_call_record, post_sys_call_record);
	if (retval < 0) {
		printk(KERN_ERR "error(init):cannot register function\n");
		goto fail;
	}

	/* determine the module's major and minor number */
	retval = alloc_chrdev_region(&kstrax_number, 0, DEV_COUNT, DEVICE_NAME);
	if (retval < 0) {
		printk(KERN_ERR "error(init):cannot allocate device number\n");
		goto fail;
	}
	
	/* char device registration */
	retval = setup_kstrax_cdev();
	if (retval < 0) {
		printk(KERN_ERR "error(init):cannot add cdev\n");
		goto fail;
	}

	retval = kstrax_init_status(kbuffer_entry);
	if (retval < 0) {
		printk(KERN_ERR "error(init):cannot initialize status\n");
		goto fail;
	}

	printk("%s module version %s\n", DEVICE_NAME, VERSION);		
	return retval; // success
	
 fail:
	exit_kstrax();
	return retval;
}

module_init(init_kstrax);
module_exit(exit_kstrax);
