/*
 mbuffer -- buffer module
 Copyright (c) 2005,2006 Hitachi,Ltd.,
 Created by Satoru Moriya <s-moriya@sdl.hitachi.co.jp>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __MBUFFER_H__
#define __MBUFFER_H__

#include <linux/interrupt.h>
#include <asm/atomic.h>

#include "kstrax_ioc.h"
#include "kstrax_syscall_list.h"
#include "mcontrol.h"

#define SPECIFIED_FILE (kstrax_status.trace_syscall[KS_OPEN] == 1 && \
		       	kstrax_status.trace_syscall[KS_CREAT] == 1 && \
			kstrax_status.trace_syscall[KS_LINK] == 1 && \
			kstrax_status.trace_syscall[KS_UNLINK] == 1 && \
			kstrax_status.trace_syscall[KS_EXECVE] == 1 && \
			kstrax_status.trace_syscall[KS_CHDIR] == 1 && \
			kstrax_status.trace_syscall[KS_MKNOD] == 1 && \
			kstrax_status.trace_syscall[KS_CHMOD] == 1 && \
			kstrax_status.trace_syscall[KS_LCHOWN16] == 1 && \
			kstrax_status.trace_syscall[KS_STAT] == 1 && \
			kstrax_status.trace_syscall[KS_MOUNT] == 1 && \
			kstrax_status.trace_syscall[KS_OLDUMOUNT] == 1 && \
			kstrax_status.trace_syscall[KS_UTIME] == 1 && \
			kstrax_status.trace_syscall[KS_ACCESS] == 1 && \
			kstrax_status.trace_syscall[KS_RENAME] == 1 && \
			kstrax_status.trace_syscall[KS_MKDIR] == 1 && \
			kstrax_status.trace_syscall[KS_RMDIR] == 1 && \
			kstrax_status.trace_syscall[KS_ACCT] == 1 && \
			kstrax_status.trace_syscall[KS_UMOUNT] == 1 && \
			kstrax_status.trace_syscall[KS_CHROOT] == 1 && \
			kstrax_status.trace_syscall[KS_SYMLINK] == 1 && \
			kstrax_status.trace_syscall[KS_LSTAT] == 1 && \
			kstrax_status.trace_syscall[KS_READLINK] == 1 && \
			kstrax_status.trace_syscall[KS_USELIB] == 1 && \
			kstrax_status.trace_syscall[KS_SWAPON] == 1 && \
			kstrax_status.trace_syscall[KS_TRUNCATE] == 1 && \
			kstrax_status.trace_syscall[KS_STATFS] == 1 && \
			kstrax_status.trace_syscall[KS_NEWSTAT] == 1 && \
			kstrax_status.trace_syscall[KS_NEWLSTAT] == 1 && \
			kstrax_status.trace_syscall[KS_PREAD64] == 1 && \
			kstrax_status.trace_syscall[KS_PWRITE64] == 1 && \
			kstrax_status.trace_syscall[KS_CHOWN16] == 1 && \
			kstrax_status.trace_syscall[KS_GETCWD] == 1 && \
			kstrax_status.trace_syscall[KS_SENDFILE] == 1 && \
			kstrax_status.trace_syscall[KS_TRUNCATE64] == 1 && \
			kstrax_status.trace_syscall[KS_FTRUNCATE64] == 1 && \
			kstrax_status.trace_syscall[KS_STAT64] == 1 && \
			kstrax_status.trace_syscall[KS_LSTAT64] == 1 && \
			kstrax_status.trace_syscall[KS_FSTAT64] == 1 && \
			kstrax_status.trace_syscall[KS_LCHOWN] == 1 && \
			kstrax_status.trace_syscall[KS_CHOWN] == 1 && \
			kstrax_status.trace_syscall[KS_PIVOT_ROOT] == 1 && \
			kstrax_status.trace_syscall[KS_SETXATTR] == 1 && \
			kstrax_status.trace_syscall[KS_LSETXATTR] == 1 && \
			kstrax_status.trace_syscall[KS_GETXATTR] == 1 && \
			kstrax_status.trace_syscall[KS_LGETXATTR] == 1 && \
			kstrax_status.trace_syscall[KS_LISTXATTR] == 1 && \
			kstrax_status.trace_syscall[KS_LLISTXATTR] == 1 && \
			kstrax_status.trace_syscall[KS_REMOVEXATTR] == 1 && \
			kstrax_status.trace_syscall[KS_LREMOVEXATTR] == 1 && \
			kstrax_status.trace_syscall[KS_SENDFILE64] == 1 && \
			kstrax_status.trace_syscall[KS_FADVISE64] == 1 && \
			kstrax_status.trace_syscall[KS_STATFS64] == 1 && \
			kstrax_status.trace_syscall[KS_FSTATFS64] == 1 && \
			kstrax_status.trace_syscall[KS_UTIMES] == 1 && \
			kstrax_status.trace_syscall[KS_FADVISE64_64])

#define SPECIFIED_PROCESS (kstrax_status.trace_syscall[KS_EXIT] == 1 && \
			   kstrax_status.trace_syscall[KS_FORK] == 1 && \
			   kstrax_status.trace_syscall[KS_WAITPID] == 1 && \
			   kstrax_status.trace_syscall[KS_EXECVE] == 1 && \
			   kstrax_status.trace_syscall[KS_WAIT4] == 1 && \
			   kstrax_status.trace_syscall[KS_CLONE] == 1 && \
			   kstrax_status.trace_syscall[KS_VFORK] == 1 && \
			   kstrax_status.trace_syscall[KS_WAITID] == 1)
    
#define SPECIFIED_SIGNAL (kstrax_status.trace_syscall[KS_PAUSE] == 1 && \
			  kstrax_status.trace_syscall[KS_KILL] == 1 && \
			  kstrax_status.trace_syscall[KS_SIGNAL] == 1 && \
			  kstrax_status.trace_syscall[KS_SIGACTION] == 1 && \
			  kstrax_status.trace_syscall[KS_SGETMASK] == 1 && \
			  kstrax_status.trace_syscall[KS_SSETMASK] == 1 && \
			  kstrax_status.trace_syscall[KS_SIGSUSPEND] == 1 && \
			  kstrax_status.trace_syscall[KS_SIGPENDING] == 1 && \
			  kstrax_status.trace_syscall[KS_SIGRETURN] == 1 && \
			  kstrax_status.trace_syscall[KS_SIGPROCMASK] == 1 && \
			  kstrax_status.trace_syscall[KS_RT_SIGRETURN] == 1 && \
			  kstrax_status.trace_syscall[KS_RT_SIGACTION] == 1 && \
			  kstrax_status.trace_syscall[KS_RT_SIGPROCMASK] == 1 && \
			  kstrax_status.trace_syscall[KS_RT_SIGPENDING] == 1 && \
			  kstrax_status.trace_syscall[KS_RT_SIGTIMEDWAIT] == 1 && \
			  kstrax_status.trace_syscall[KS_RT_SIGSUSPEND] == 1 && \
			  kstrax_status.trace_syscall[KS_RT_SIGQUEUEINFO] == 1 && \
			  kstrax_status.trace_syscall[KS_TKILL] == 1 && \
			  kstrax_status.trace_syscall[KS_TGKILL] == 1 && \
			  kstrax_status.trace_syscall[KS_SIGALTSTACK] == 1)

#define IS_FILE_SYSCALL(i) \
if (i == KS_OPEN || i == KS_CREAT || i == KS_LINK || i == KS_UNLINK || \
    i == KS_EXECVE || i == KS_CHDIR || i == KS_MKNOD || i == KS_CHMOD || \
    i == KS_LCHOWN16 || i == KS_STAT || i == KS_MOUNT || i == KS_OLDUMOUNT || \
    i == KS_UTIME || i == KS_ACCESS || i == KS_RENAME || i == KS_MKDIR || \
    i == KS_RMDIR || i == KS_ACCT || i == KS_UMOUNT || i == KS_CHROOT || \
    i == KS_SYMLINK || i == KS_LSTAT || i == KS_READLINK || i == KS_USELIB || \
    i == KS_SWAPON || i == KS_TRUNCATE || i == KS_STATFS || i== KS_NEWSTAT || \
    i == KS_NEWLSTAT || i == KS_PREAD64 || i == KS_PWRITE64 || i == KS_CHOWN16 || \
    i == KS_GETCWD || i == KS_SENDFILE || i == KS_TRUNCATE64 || \
    i == KS_FTRUNCATE64 || i == KS_STAT64 || i == KS_LSTAT64 || \
    i == KS_FSTAT64 || i == KS_LCHOWN || i == KS_CHOWN || i == KS_PIVOT_ROOT || \
    i == KS_SETXATTR || i == KS_LSETXATTR || i == KS_GETXATTR || \
    i == KS_LGETXATTR || i == KS_LISTXATTR || i == KS_LLISTXATTR || \
    i == KS_REMOVEXATTR || i == KS_LREMOVEXATTR || i == KS_SENDFILE64 || \
    i == KS_FADVISE64 || i == KS_STATFS64 || i == KS_FSTATFS64 || \
    i == KS_UTIMES || i == KS_FADVISE64_64)

#define IS_PROCESS_SYSCALL(i) \
if (i == KS_EXIT || i == KS_FORK || i == KS_WAITPID || i == KS_EXECVE || \
    i == KS_WAIT4 || i == KS_CLONE || i == KS_VFORK || i == KS_WAITID)

#define IS_SIGNAL_SYSCALL(i) \
if (i == KS_PAUSE || i == KS_KILL || i == KS_SIGNAL || i == KS_SIGACTION || \
    i == KS_SGETMASK || i == KS_SSETMASK || i == KS_SIGSUSPEND || \
    i == KS_SIGPENDING || i == KS_SIGRETURN || i == KS_SIGPROCMASK ||\
    i == KS_RT_SIGRETURN || i == KS_RT_SIGACTION || i == KS_RT_SIGPROCMASK || \
    i == KS_RT_SIGPENDING || i == KS_RT_SIGTIMEDWAIT || i == KS_RT_SIGSUSPEND || \
    i == KS_RT_SIGQUEUEINFO || i == KS_TKILL || i == KS_TGKILL || \
    i == KS_SIGALTSTACK)

#define KS_FILENAME    -11
#define KSTRAX_WAKEUP    1
#define KSTRAX_SLEEP     0
#define BASETIME        -1

typedef struct sys_call_info {
	pid_t pid;
	short sys_call_number;
	long time;
	long utime;
	union {
		long arg_1;
		long return_value;
	};
	long arg_2;
	long arg_3;
	long arg_4;
	long arg_5;
	long arg_6;
	int  serial; /* int */
	int  cpu;
} sys_call_t;

typedef struct kbuffer_head {
	sys_call_t *buffer;
	int size;
	int w_index;
	int r_index;
	int wakeup_index;
	int wakeup_flag;
	wait_queue_head_t wq_head;
	struct tasklet_struct kst_tasklet;
} buf_head_t;

extern trace_stat_t kstrax_status;
extern atomic_t kstrax_serial;
DECLARE_PER_CPU(buf_head_t, kernel_buffer_head);

/*
 *  function declaration
 */
static int init_mbuffer(int);
void clear_mbuffer(void);
int kstrax_reset_index(void);
int kstrax_read_basetime(int *);
int kstrax_copy_status(trace_stat_t *);
int kstrax_pid_spec(pid_t);
ssize_t kstrax_read(struct file *, char __user *, size_t, loff_t *);

void kstrax_trace_file(void);
void kstrax_trace_process(void);
void kstrax_trace_signal(void);
void test_flag_spec_syscall(void);

static inline int kstrax_init_status(int nr_entry)
{
	int i;
	
	kstrax_status.nr_cpu = num_online_cpus();
	kstrax_status.flag_spec_syscall = 0;
	for (i = 0; i < NR_syscalls; i++)
		kstrax_status.trace_syscall[i] = 0;
	for (i = 0; i < NR_SOCKETCALL; i++)
		kstrax_status.trace_socketcall[i] = 0;
	for (i = 0; i < NR_IPC; i++)
		kstrax_status.trace_ipc[i] = 0;
	for (i = 0; i < NR_TRACE_PROCESS_MAX; i++)
		kstrax_status.trace_pid[i] = INIT_PID;
	kstrax_status.nr_buf_entry = nr_entry;

	return 0;
}

static inline void set_basetime(sys_call_t *kst_time)
{
	struct timeval tv;
	
	/* get the system time (tsc & gettimeofday) */
	do_gettimeofday(&tv);
	rdtsc(kst_time->utime, kst_time->time);
	kst_time->arg_1 = tv.tv_sec;
	kst_time->arg_2 = tv.tv_usec;
	kst_time->arg_3 = cpu_khz;
	kst_time->pid = BASETIME;
}

static inline void kstrax_trace_all(void)
{
	int i;

	kstrax_status.flag_spec_syscall = 0;

	for (i = 0; i < NR_syscalls; i++)
		kstrax_status.trace_syscall[i] = 0;
	for (i = 0; i < NR_SOCKETCALL; i++)
		kstrax_status.trace_socketcall[i] = 0;
	for (i = 0; i < NR_IPC; i++)
		kstrax_status.trace_ipc[i] = 0;

	return;
}

static inline void set_each_socketcall(int syscall_num)
{
	int i;
	int nr_trace = 0;
	/* tag to index */
	/* see module/kstrax_ioc.h & src/syscall_name.c */
	int index = -syscall_num - 7; 
	
	if (kstrax_status.trace_syscall[KS_SOCKETCALL] == 0) {
		/* on */
		kstrax_status.trace_socketcall[index] = 1;
		kstrax_status.trace_socketcall[0] = 1;
		kstrax_status.trace_syscall[KS_SOCKETCALL] = 1;
	} else {
		if (kstrax_status.trace_socketcall[index] == 0) {
			/* on */
			kstrax_status.trace_socketcall[index] = 1;
			kstrax_status.trace_socketcall[0] = 1;
			kstrax_status.trace_syscall[KS_SOCKETCALL] = 1;
		} else {
			/* off */
			kstrax_status.trace_socketcall[index] = 0;
			for (i = 1; i < NR_SOCKETCALL; i++) {
				if (kstrax_status.trace_socketcall[i] == 1)
					nr_trace++;
			}
			if (nr_trace == 0) {
				kstrax_status.trace_syscall[KS_SOCKETCALL] = 0;
				kstrax_status.trace_socketcall[0] = 0;
			}
		}
	}
}

static inline void kstrax_trace_socketcall(int syscall_num)
{
	int i;
       
	if (syscall_num == KS_SOCKETCALL) {
		/* trace all of socketcall */
		if (kstrax_status.trace_syscall[KS_SOCKETCALL] == 1 &&
		    kstrax_status.trace_socketcall[0] == 0) {
			/* off */
			kstrax_status.trace_syscall[KS_SOCKETCALL] = 0;
			for (i = 0; i < NR_SOCKETCALL; i++)
				kstrax_status.trace_socketcall[i] = 0;
		} else {
			/* on */
			for (i = 0; i < NR_SOCKETCALL; i++)
				kstrax_status.trace_socketcall[i] = 0;
			kstrax_status.trace_syscall[KS_SOCKETCALL] = 1;
		}
	} else {
		/* trace each socketcall */
		set_each_socketcall(syscall_num);
	}
	test_flag_spec_syscall();
}
static inline void set_each_ipc(int syscall_num)
{
	int i;
	int nr_trace = 0;
	/* tag to index */
	/* see module/kstrax_ioc.h & src/syscall_name.c */
	int num = -syscall_num - 25;
	int index[12] = {1, 2, 3, 4, 11, 12, 13, 14, 21, 22, 23, 24};
	
	if (kstrax_status.trace_syscall[KS_IPC] == 0) {
		/* on */
		kstrax_status.trace_ipc[index[num]] = 1;
		kstrax_status.trace_ipc[0] = 1;
		kstrax_status.trace_syscall[KS_IPC] = 1;
	} else {
		if (kstrax_status.trace_ipc[index[num]] == 0) {
			/* on */
			kstrax_status.trace_ipc[index[num]] = 1;
			kstrax_status.trace_ipc[0] = 1;
			kstrax_status.trace_syscall[KS_IPC] = 1;
		} else {
			/* off */
			kstrax_status.trace_ipc[index[num]] = 0;
			for (i = 1; i < NR_IPC; i++) {
				if (kstrax_status.trace_ipc[i] == 1)
					nr_trace++;
			}
			if (nr_trace == 0) {
				kstrax_status.trace_syscall[KS_IPC] = 0;
				kstrax_status.trace_ipc[0] = 0;
			}
		}
	}
}

static inline void kstrax_trace_ipc(int syscall_num)
{
	int i;
	
	if (syscall_num == KS_IPC) {
		if (kstrax_status.trace_syscall[KS_IPC] == 1 &&
		    kstrax_status.trace_ipc[0] == 0) {
			/* off */
			kstrax_status.trace_syscall[KS_IPC] = 0;
			for (i = 0; i < NR_IPC; i++)
				kstrax_status.trace_ipc[i] = 0;
		} else {
			/* on */
			for (i = 0; i < NR_IPC; i++)
				kstrax_status.trace_ipc[i] = 0;
			kstrax_status.trace_syscall[KS_IPC] = 1;
		}
	} else {
		set_each_ipc(syscall_num);
	}
	test_flag_spec_syscall();
}

/*------------------------------
 *  ken-patu
 */

static inline void kstrax_trace_type_demo(void)
{
	int i;
	
	/* off specified syscall */
	if (kstrax_status.flag_spec_syscall == TRACE_DEMO) {
		for (i = 0; i < NR_syscalls; i++)
			kstrax_status.trace_syscall[i] = 0;
		kstrax_status.flag_spec_syscall = 0;
		return;
	}

	for (i = 0; i < NR_syscalls; i++) {
		if (i == KS_READ || i == KS_WRITE || i == KS_OPEN ||
		    i == KS_CLOSE || i == KS_EXECVE || i == KS_TIME ||
		    i == KS_GETTIMEOFDAY || i == KS_CLONE)
			kstrax_status.trace_syscall[i] = 1;
		else 
			kstrax_status.trace_syscall[i] = 0;
	}
	kstrax_status.flag_spec_syscall = TRACE_DEMO;

	return;
}

static inline int kstrax_syscall_spec(int syscall_num)
{
	int retval = 0;

	if (syscall_num < 0) { /* specified by syscall type */
		switch (syscall_num) {
		case TRACE_ALL:
			kstrax_trace_all();
			break;
		case TRACE_FILE:
			kstrax_trace_file();
			break;
		case TRACE_NETWORK:
			kstrax_trace_socketcall(KS_SOCKETCALL);
			break;
		case TRACE_IPC:
			kstrax_trace_ipc(KS_IPC);
			break;
		case TRACE_PROCESS:
			kstrax_trace_process();
			break;
		case TRACE_SIGNAL:
			kstrax_trace_signal();
			break;
		case TRACE_DEMO:
			kstrax_trace_type_demo();

		default:
			if (TRACE_SOCKET >= syscall_num &&
			    syscall_num >= TRACE_RECVMSG) {
				kstrax_trace_socketcall(syscall_num);
			} else if (TRACE_SEMOP >= syscall_num &&
				   syscall_num >= TRACE_SHMCTL) {
				kstrax_trace_ipc(syscall_num);
			} else {
				printk(KERN_ERR "error(syscall_spec):bad argument\n");
				return -EINVAL;
			}
		}
	} else { /* syscall_num >= 0 */
		if (syscall_num == KS_SOCKETCALL) {
			kstrax_trace_socketcall(syscall_num);
		} else if (syscall_num == KS_IPC) {
			kstrax_trace_ipc(syscall_num);
		} else {
			if (syscall_num > NR_syscalls) {
				printk("error(syscall_spec):bad argument\n");
				return -EINVAL;
			}
			
			if (kstrax_status.trace_syscall[syscall_num] == 0)
				kstrax_status.trace_syscall[syscall_num] = 1;
			else
				kstrax_status.trace_syscall[syscall_num] = 0;

			test_flag_spec_syscall();
		}
	}
	return retval;
}


static inline void pre_sys_call_record(const struct pt_regs regs)
{
	int i;
	buf_head_t *head;
	__u32 cpu;  /* unsigned int */
	struct list_head *w_list;
	sys_call_t *buf;

	/* check tracing pid or not */
	if (kstrax_status.trace_pid[0] != INIT_PID) {
		for (i = 0; i < NR_TRACE_PROCESS_MAX; i++) {
			if (current->pid == kstrax_status.trace_pid[i]) {
				break;
			} else if (kstrax_status.trace_pid[i] == INIT_PID) {
				return;
			}
		}
	}

	/* check tracing system call or not */
	if (kstrax_status.flag_spec_syscall != 0 &&
	    kstrax_status.trace_syscall[regs.orig_eax] == 0)
		return;

	if(regs.orig_eax == KS_SOCKETCALL) {
		if (kstrax_status.trace_socketcall[0] == 1) {
			if (regs.ebx < 1 || 17 < regs.ebx)
				return;

			if (kstrax_status.trace_socketcall[regs.ebx] == 0)
				return;
		}
	}
	
	if(regs.orig_eax == KS_IPC) {
		if (kstrax_status.trace_ipc[0] == 1) {
			if (regs.ebx < 1 ||
			    (4 < regs.ebx && regs.ebx < 11) ||
			    (14 < regs.ebx && regs.ebx < 21) ||
			    24 < regs.ebx)
				return;
			
			if(kstrax_status.trace_ipc[regs.ebx] == 0)
				return;
		}		
	}		

	/* buffer header of current cpu */
	cpu = get_cpu();
	head = &per_cpu(kernel_buffer_head, cpu);
	
	buf = &head->buffer[head->w_index];
	w_list = &head->wq_head.task_list;

	/* update index */	
	head->w_index = (head->w_index + 1) % head->size;
	if (head->w_index == head->r_index) {
		head->r_index = (head->r_index + 1) % head->size;
	}
	
	buf->pid = current->pid;
	rdtsc(buf->utime, buf->time);
	buf->sys_call_number = regs.orig_eax;
	buf->arg_1 = regs.ebx;
	buf->arg_2 = regs.ecx;
	buf->arg_3 = regs.edx;
	buf->arg_4 = regs.esi;
	buf->arg_5 = regs.edi;
	buf->arg_6 = regs.ebp;
	buf->serial = atomic_inc_return(&kstrax_serial); 
	buf->cpu = cpu;

	if (head->wakeup_flag == KSTRAX_SLEEP && 
	    (head->wakeup_index == head->w_index ||
	     kstrax_force_flag == 1)) {
		head->wakeup_flag = KSTRAX_WAKEUP;

		if (!list_empty(w_list)) {
			tasklet_schedule(&head->kst_tasklet);
		}
	}
	put_cpu();
}

static inline void init_buf_entry(sys_call_t *entry)
{
	entry->pid = 0;
	entry->sys_call_number = 0;
	entry->time = 0;
	entry->utime = 0;
	entry->arg_1 = 0;
	entry->arg_2 = 0;
	entry->arg_3 = 0;
	entry->arg_4 = 0;
	entry->arg_5 = 0;
	entry->arg_6 = 0;
	entry->serial = 0;
	entry->cpu = 0;
}

/* return entry */
static inline void post_sys_call_record(const struct pt_regs regs)
{
	int i;
	buf_head_t *head;
	__u32 cpu;  /* unsigned int */
	struct list_head *w_list;
	sys_call_t *buf, *name = NULL;
	char filename[256] = {'\0'};
	int nr_add = 0;
	int copy_size;

	/* check tracing pid or not */
	if (kstrax_status.trace_pid[0] != INIT_PID) {
		for (i = 0; i < NR_TRACE_PROCESS_MAX; i++) {
			if (current->pid == kstrax_status.trace_pid[i]) {
				break;
			} else if (kstrax_status.trace_pid[i] == INIT_PID) {
				return ;
			}
		}
	}

	if (kstrax_status.flag_spec_syscall != 0) {
		if (0 <= regs.orig_eax && regs.orig_eax < NR_syscalls) {
			if (kstrax_status.trace_syscall[regs.orig_eax] == 0)
				return;
		} else if (regs.orig_eax == -1) {
			if (kstrax_status.trace_syscall[KS_SIGRETURN] == 0 &&
			    kstrax_status.trace_syscall[KS_RT_SIGRETURN] == 0)
				return;
		}
	}

	if(regs.orig_eax == KS_SOCKETCALL) {
		if (kstrax_status.trace_socketcall[0] == 1) {
			if (regs.ebx < 1 || 17 < regs.ebx)
				return;

			if (kstrax_status.trace_socketcall[regs.ebx] == 0)
				return;
		}
	}

	if(regs.orig_eax == KS_IPC) {
		if (kstrax_status.trace_ipc[0] == 1) {
			if (regs.ebx < 1 ||
			    (4 < regs.ebx && regs.ebx < 11) ||
			    (14 < regs.ebx && regs.ebx < 21) ||
			    24 < regs.ebx)
				return;
			
			if(kstrax_status.trace_ipc[regs.ebx] == 0)
				return;
		}		
	}		

	/* buffer header of current cpu */
	cpu = get_cpu();
	head = &per_cpu(kernel_buffer_head, cpu);  
	
	buf = &head->buffer[head->w_index];
	w_list = &head->wq_head.task_list;

	copy_size = sizeof(sys_call_t) - ((unsigned long)&buf->time - \
					  (unsigned long)buf) -1;
	/* copy filename */
	if (regs.orig_eax == KS_OPEN) {
		int len;
		
		put_cpu();
		/* might sleep */
		strncpy_from_user(filename, (const char *)regs.ebx, 256);
		get_cpu();

		len = strnlen(filename, 256);
		if (len % copy_size)
			nr_add = len / copy_size + 1;
		else
			nr_add = len / copy_size;
		/* reload */
		buf = &head->buffer[head->w_index];
		name = &head->buffer[(head->w_index + 1) % head->size];

		for (i = 0; i < nr_add; i++) {
			init_buf_entry(name);
			name->pid = KS_FILENAME;
			name->sys_call_number = regs.eax;
			strncpy((char *)&name->time,
				(const char *)&filename[i * copy_size],
				copy_size);
			name++;
		}
	
		for (i = 0; i < nr_add; i++) {
			head->w_index = (head->w_index + 1) % head->size;
			if (head->w_index == head->r_index) {
				head->r_index = (head->r_index + 1) % head->size;
			}
		}
	}

	/* update index */
	head->w_index = (head->w_index + 1) % head->size;
	if (head->w_index == head->r_index) {
		head->r_index = (head->r_index + 1) % head->size;
	}

	buf->pid = current->pid;
	rdtsc(buf->utime, buf->time);

	if (regs.orig_eax == -1) /* sigreturn is special */
		buf->sys_call_number = -KS_RT_SIGRETURN;
	else 
		buf->sys_call_number = -regs.orig_eax;

	buf->return_value = regs.eax;
	buf->arg_2 = regs.ebx;  /* for ipc & socketcall */
	buf->serial = atomic_inc_return(&kstrax_serial);
	buf->cpu = cpu;

	if (head->wakeup_flag == KSTRAX_SLEEP &&
	    (head->wakeup_index == head->w_index ||
	     kstrax_force_flag == 1)) {
		head->wakeup_flag = KSTRAX_WAKEUP;
		
		if (!list_empty(w_list)) { 
			tasklet_schedule(&head->kst_tasklet);
		}
	}
	put_cpu();
}
#endif /* __MBUFFER_H__ */
