/*
 call table redirection for IA-64
 Copyright (c) 2006 Hitachi,Ltd.,
 Created by Satoru Moriya <satoru.moriya.br@hitachi.com>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <linux/module.h>
#include <linux/threads.h>
#include <asm/unistd.h>

#include "ctr.h"
#include "kstrax_syscall_list.h"

#define KS_NR_TABLE 2

struct ks_ia64_symbol {
	unsigned long faddr;
	unsigned long base;
	int flag_a9;
};

struct ks_ia64_asmdata {
	unsigned long faddr;
	unsigned long base;
	unsigned long pfs;
	unsigned long r15;
	unsigned long chain_r15;
};

/*
 *  global variable
 */
static syscall_handler_t pre_handler = NULL;
static syscall_handler_t post_handler = NULL;
static call_table_t *true_sys_call_addr[2] = {NULL, NULL};
struct ks_ia64_symbol true_sys_call_table[2][512];
struct ks_ia64_asmdata kstrax_true_return_addr[PID_MAX_DEFAULT];

/*
 *  global variable and parameter
 */
static unsigned long ks_gp;
module_param(ks_gp, ulong, 0444);
MODULE_PARM_DESC(ks_gp, "Global Pointer of Kernel");

static void init_kstrax_true_return_addr(void)
{
	int i;
	for (i = 0; i < PID_MAX_DEFAULT; i++) {
		kstrax_true_return_addr[i].faddr=-1;
		kstrax_true_return_addr[i].base=-1;
		kstrax_true_return_addr[i].pfs=-1;
		kstrax_true_return_addr[i].r15=-1;
		kstrax_true_return_addr[i].chain_r15=-1;
	}
}

asmlinkage int kstrax_set_return_addr(unsigned long ret_addr, unsigned long gp,
				       unsigned long pfs, unsigned long r15)
{
	int cur_pid = current->pid;

	if ((long)kstrax_true_return_addr[cur_pid].faddr != -1) {
		if (r15 < 1024)
			kstrax_true_return_addr[cur_pid].chain_r15 = r15;
		else
			kstrax_true_return_addr[cur_pid].chain_r15 = r15-1024;
		return -1;
	}

	kstrax_true_return_addr[cur_pid].faddr = ret_addr;
	kstrax_true_return_addr[cur_pid].base = gp;
	kstrax_true_return_addr[cur_pid].pfs = pfs;
	if (r15 < 1024)
		kstrax_true_return_addr[cur_pid].r15 = r15;
	else
		kstrax_true_return_addr[cur_pid].r15 = r15-1024;

	return 0;
}

asmlinkage struct ks_ia64_asmdata *kstrax_get_return_addr(void)
{
	return &kstrax_true_return_addr[current->pid];
}

unsigned long kstrax_get_syscall_num(void)
{
	return(kstrax_true_return_addr[current->pid].r15);
}

struct ks_ia64_symbol *ia64_get_true_syscall(void)
{
	unsigned long r15;
	int cpid = current->pid;

	if ((long)kstrax_true_return_addr[cpid].chain_r15 == -1) {
		r15 = kstrax_true_return_addr[cpid].r15;
		if (r15 == KS_RT_SIGRETURN || r15 == KS_EXIT || 
		    r15 == KS_EXIT_GROUP) {
			kstrax_true_return_addr[cpid].faddr=-1;
			kstrax_true_return_addr[cpid].base=-1;
			kstrax_true_return_addr[cpid].pfs=-1;
			kstrax_true_return_addr[cpid].r15=-1;
			kstrax_true_return_addr[cpid].chain_r15=-1;
		}
	} else {
		/* chain load */
		r15 = kstrax_true_return_addr[current->pid].chain_r15;
		kstrax_true_return_addr[current->pid].chain_r15 = -1;
	}

	return &true_sys_call_table[1][r15];
}

void pre_sys_call(const ctr_handler_t args)
{
	if (pre_handler)
		pre_handler(args);
}

void post_sys_call(const ctr_handler_t args)
{
	if (post_handler)
		post_handler(args);
}

int change_sys_call_table(call_table_t *s_addr, call_table_t *fs_addr, 
			  void **stb_addr, void **stb_addr_2)
{
	int i, j;

	for (i = 0; i < KS_NR_TABLE; i++) {
		if (true_sys_call_addr[i] != NULL)
			return -1;
	}
	
	true_sys_call_addr[0] = fs_addr;
	true_sys_call_addr[1] = s_addr;
       	for (i = 0; i < NR_syscalls; i++) {
		for (j = 0; j < KS_NR_TABLE; j++) {
			if (j == 0) {
				true_sys_call_table[j][i].faddr = (long)fs_addr[i];
			} else { 
				true_sys_call_table[j][i].faddr = (long)s_addr[i];
			}
			true_sys_call_table[j][i].base = ks_gp;
			
			/* flag for nr_argument = 9 */
			if (i == KS_PIPE || i == KS_PERFMONCTL || i == KS_PTRACE ||
			    i == KS_SIGALTSTACK)
				true_sys_call_table[j][i].flag_a9 = 1;
			else 
				true_sys_call_table[j][i].flag_a9 = 0;
		}
 
		if (i != KS_CLONE && i != KS_CLONE2) {
			fs_addr[i] = 
				(call_table_t)((unsigned long)(*(void **)stb_addr) | 1UL);
			s_addr[i] = 			
				(call_table_t)((unsigned long)(*(void **)stb_addr) | 1UL);
		} else if (i == KS_CLONE2 || i == KS_CLONE) {
			fs_addr[i] = 
				(call_table_t)((unsigned long)(*(void **)stb_addr_2) | 1UL);
			s_addr[i] = 			
				(call_table_t)((unsigned long)(*(void **)stb_addr_2) | 1UL);
		}	
	}
	return 0;
}

void restore_sys_call_table(void)
{
	int i, j;
	call_table_t *sys_call_table;

	for (i = 0; i < KS_NR_TABLE; i++) {
		if (true_sys_call_addr[i] == NULL)
			return ;
	}

	for (i = 0; i < KS_NR_TABLE; i++) {
		sys_call_table = (void *)true_sys_call_addr[i];

		for (j = 0; j < NR_syscalls; j++) {
			sys_call_table[j] = (call_table_t)true_sys_call_table[i][j].faddr;
		}
		true_sys_call_addr[i] = NULL;
	}
}

/*------------------------------------------------------------------------
 *  module interface
 */
int register_ctr(syscall_handler_t pre, syscall_handler_t post)
{
	if (pre == NULL && post == NULL)
		return -EINVAL;
	pre_handler = pre;
	post_handler = post;
	return 0;
}

void unregister_ctr(void)
{
	pre_handler = NULL;
	post_handler = NULL;
}

/*------------------------------------------------------------------------
 *  module init/exit
 */
#define CTR_INIT static int __init
#define CTR_EXIT static void

CTR_EXIT ia64_exit_ctr(void)
{
	return;
}

CTR_INIT ia64_init_ctr(void)
{
	init_kstrax_true_return_addr();
	return 0;
}

module_init(ia64_init_ctr)
module_exit(ia64_exit_ctr);
MODULE_AUTHOR("S.Moriya <satoru.moriya.br@hitachi.com>");
MODULE_LICENSE("GPL");
EXPORT_SYMBOL_GPL(register_ctr);
EXPORT_SYMBOL_GPL(unregister_ctr);

EXPORT_SYMBOL_GPL(kstrax_set_return_addr);
EXPORT_SYMBOL_GPL(kstrax_get_return_addr);
EXPORT_SYMBOL_GPL(kstrax_get_syscall_num);
EXPORT_SYMBOL_GPL(ia64_get_true_syscall);
EXPORT_SYMBOL_GPL(pre_sys_call);
EXPORT_SYMBOL_GPL(post_sys_call);
EXPORT_SYMBOL_GPL(change_sys_call_table);
EXPORT_SYMBOL_GPL(restore_sys_call_table);
