/*
 kstrax
 Copyright (c) 2005,2007 Hitachi,Ltd.,
 Created by Satoru Moriya <satoru.moriya.br@hitachi.com>
 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <string.h>
#include <time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "kstrax.h"
#include "kstrax_buf.h"
#include "kstrax_ioc.h"
#include "syscall_name.h"
#include "kstrax_syscall_list.h"

#include "kstrax_arch.h"

#define DEFAULT_NR_ENTRY 4000
#define DEFAULT_LIMIT_TIME 0 /* sec */
#define NAME       "kstrax"
#define COPYRIGHT  "COPYRIGHT (C) HITACHI,LTD. 2005,2006"
#define INPUT_FILE "/dev/kstrax"

/*
 *  global variable
 */
extern char *optarg;
int kstrax_cflag = 0, kstrax_kflag = 0, kstrax_rflag = 0;
int kstrax_Rflag = 0, kstrax_gflag = 0; 
static int kstrax_bflag = 0, kstrax_eflag = 0, kstrax_oflag = 0;
static int kstrax_pflag = 0, kstrax_sflag = 0;
static int kstrax_tflag = 0, kstrax_Tflag = 0;
static int ks_nr_syscalls = 0;
/*
 *  function declaration
 */
static void kstrax_set_output_filename(char *, const char *);
static void kstrax_set_default_bin_filename(char *, const char *);
static void kstrax_set_default_txt_filename(const char *, char *);
static void kstrax_set_trace_syscall(const char *);
static void kstrax_set_trace_pid(const char *);
static void kstrax_usage(void);
static void kstrax_version(void);
static void kstrax_get_status(struct kstrax_status *);
static void kstrax_print_status(const struct kstrax_status *);
static int kstrax_strtol(const char *);
static void kstrax_restore_status(struct kstrax_status *, struct kstrax_status *);
static void kstrax_set_snapshot(void);

int main(int argc, char *argv[])
{
	
	char c;
	char input_file[KSTRAX_FNAME_MAX] = INPUT_FILE;
	char output_file[KSTRAX_FNAME_MAX] = "\0";
	int nr_entry = DEFAULT_NR_ENTRY, limit_time = DEFAULT_LIMIT_TIME; /* sec */
	struct kstrax_status old_status = {0}, new_status={0};

	while ((c = getopt(argc, argv, "+bc:e:ghko:p:r:Rs:St:T:v")) != EOF) {
		switch (c) {
		case 'b': /* output binary */
			kstrax_get_status(&new_status);
			kstrax_bflag = 1;
			break;
		case 'c': /* count calls, times... print statistics information */
			if (strlen(optarg) > (KSTRAX_FNAME_MAX - 5)) {
				printf("Too long filename\n");
				exit(1);
			}
			strncpy(input_file, optarg, (KSTRAX_FNAME_MAX - 5));
			kstrax_cflag = 1;
			break;
		case 'e': /* system call specification in kernel space */
			if (old_status.nr_cpu == 0)
				kstrax_get_status(&old_status);
			kstrax_set_trace_syscall(optarg);
			kstrax_get_status(&new_status);
			kstrax_print_status(&new_status);
			kstrax_eflag = 1;
			break;
		case 'g': /* for Graph */
			kstrax_gflag = 1;
			break;
		case 'h': /* help */
			kstrax_usage();
			goto out;
		case 'k': /* keep read index in kernel space */
			kstrax_kflag = 1;
			break;
		case 'o': /* set print file name */
			kstrax_set_output_filename(output_file, optarg);
			kstrax_oflag = 1;
			break;
		case 'p': /* pid specification in kernel space */
			if (old_status.nr_cpu == 0)
				kstrax_get_status(&old_status);
			kstrax_set_trace_pid(optarg);
			kstrax_get_status(&new_status);
			kstrax_print_status(&new_status);
			kstrax_pflag = 1;
			break;
		case 'r': /* print text at RAW mode */
			if (strlen(optarg) > (KSTRAX_FNAME_MAX - 5)) {
				printf("Too long filename\n");
				exit(1);
			}
			strncpy(input_file, optarg, (KSTRAX_FNAME_MAX - 5));
			kstrax_rflag = 1;
			break;
		case 'R': /* get snapshot */
			kstrax_set_snapshot();
			kstrax_kflag = 1;
			kstrax_Rflag = 1;
			break;
		case 's': /* set buffer size in user space */
			nr_entry = kstrax_strtol(optarg);
			if (nr_entry < 0) {
				fprintf(stderr, 
					"-s has bad argument\n"
					"Invalid buffer size(%d)\n", nr_entry);
				exit(1);
			}
			kstrax_sflag = 1;
			break;
		case 'S': /* get kstrax status */
			kstrax_get_status(&new_status);
			kstrax_print_status(&new_status);
			goto out;
		case 't': /* print text (nomal mode)*/
			if (strlen(optarg) > (KSTRAX_FNAME_MAX - 5)) {
				printf("Too long filename\n");
				exit(1);
			}
			strncpy(input_file, optarg, (KSTRAX_FNAME_MAX - 5));
			kstrax_tflag = 1;
			break;
		case 'T': /* timer */
			limit_time = kstrax_strtol(optarg);
			if (limit_time < 0) {
				fprintf(stderr, 
					"-T has bad argument\n"
					"Invalid limit time(%d)\n", limit_time);
				exit(1);
			}
			kstrax_Tflag = 1;
			break;
		case 'v': /* print version information*/
			kstrax_version();
			goto out;
		case '?':
			kstrax_usage();
			goto out;
		}
	}

	/* exclusion check */
	if (((kstrax_bflag || kstrax_kflag || kstrax_sflag || kstrax_Tflag ||
	      kstrax_Rflag) &&
	     (kstrax_cflag || kstrax_rflag || kstrax_tflag || kstrax_gflag)) ||
	    (kstrax_cflag && kstrax_rflag) || (kstrax_rflag && kstrax_tflag) ||
	    (kstrax_tflag && kstrax_cflag) || (kstrax_rflag && kstrax_gflag)) {
		fprintf(stderr, "option confliction\n");
		exit(1);
	}
	
	if(kstrax_bflag == 1) {
		if (kstrax_oflag == 0)
			kstrax_set_default_bin_filename(output_file, optarg);
		kstrax_read(nr_entry, limit_time, input_file, output_file);
		if (kstrax_eflag == 1 || kstrax_pflag == 1)
			kstrax_restore_status(&old_status, &new_status);
	} else if (kstrax_cflag == 1 || kstrax_rflag == 1 || kstrax_tflag == 1) {
		if (kstrax_oflag == 0)
			kstrax_set_default_txt_filename(input_file, output_file);
		kstrax_print(input_file, output_file);
	} else if (kstrax_pflag != 1 && kstrax_eflag != 1) {
			kstrax_usage();
	}
 out:
	return 0;
}

static void kstrax_set_trace_pid(const char *optarg)
{
	int fd;
	pid_t pid;

	pid = kstrax_strtol(optarg);
	if ((fd = open(INPUT_FILE, O_WRONLY)) < 0) {
		perror("kstrax_set_trace_pid(open)");
		exit(1);
	}
	if (ioctl(fd, KSTRAX_IOC_PID_SPEC, &pid) < 0) {
		perror("kstrax_set_trace_pid(ioctl)");
		exit(1);		
	}
	close(fd);
}

static void kstrax_set_trace_syscall(const char *optarg)
{

	int i, fd, ioctl_arg;
	int flag_specified = 0;
	char type[TRACE_TYPE_LIMIT][8] = {"ALL", "FILE", "NETWORK", "IPC",
					  "PROCESS", "SIGNAL", "DEMO"};

	/* specified by syscall type */
	for (i = 0; i < TRACE_TYPE_LIMIT; i++) {
		if (strcmp(optarg, type[i]) == 0) {
			ioctl_arg = -(i + 1);
			flag_specified = 1;
		}
	}

	/* specified by syscall name */
	if (flag_specified == 0) {
		for (i = 0; i < ks_nr_syscalls; i++) {
			if (strcmp(optarg, syscall_name[i].name) == 0) {
				ioctl_arg = i;
				flag_specified = 1;
			}
		}
	}

	__kstrax_specify_socketcall_ipc(&flag_specified, &ioctl_arg, optarg);

	if (flag_specified == 0) {
		fprintf(stderr, "Unknown System Call : %s\n", optarg);
		exit(1);
	}

	if ((fd = open(INPUT_FILE, O_WRONLY)) < 0) {
		perror("kstrax_set_trace_syscall(open)");
		exit(1);
	}
	if (ioctl(fd, KSTRAX_IOC_SYSCALL_SPEC, &ioctl_arg) < 0) {
		perror("kstrax_set_trace_syscall(ioctl)");
		exit(1);
	}
	close(fd);
}

static void kstrax_get_status(struct kstrax_status *status)
{
	int fd;

	if ((fd = open(INPUT_FILE, O_WRONLY)) < 0) {
		perror("kstrax_get_status(open)");
		exit(1);
	}
	if (ioctl(fd, KSTRAX_IOC_STATUS, status) < 0) {
		perror("kstrax_get_status(ioctl)");
		exit(1);		
	}
	kstrax_set_nr_syscalls(status->nr_syscalls);
	close(fd);
}

static void kstrax_print_status(const struct kstrax_status *status)
{
	int i;

	if (status == NULL) {
		fprintf(stderr, "kstrax_print_status : invalid argument\n");
		exit(1);
	}

	printf("kernel buf entry :%d\n", status->nr_buf_entry);
	printf("tracing pid      :");
	for (i = 0; i < NR_TRACE_PROCESS_MAX; i++) {
		if (status->trace_pid[i] == -1) {
			if (i == 0) 
				printf("ALL");
			break;
		}
		printf("%d ", status->trace_pid[i]);
	}
	printf("\ntracing syscall  :");

	if (status->flag_spec_syscall == 0)
		printf("ALL\n");
	else
		__kstrax_status_print_syscall_name(status);
}

static void kstrax_restore_trace_pid_half(struct kstrax_status *former,
					  struct kstrax_status *latter,
					  int fd)
{
	int i, j;
	int flag_no_change;

	for (i = 0; i < NR_TRACE_PROCESS_MAX; i++) {
		flag_no_change = 0;
		if (former->trace_pid[i] == INIT_PID)
			break;
		for (j = 0; j < NR_TRACE_PROCESS_MAX; j++) {
			if (latter->trace_pid[j] == INIT_PID)
				break;
			if (former->trace_pid[i] == latter->trace_pid[j]) {
				flag_no_change = 1;
			}
		}
		if (flag_no_change == 0) {
			if (ioctl(fd, KSTRAX_IOC_PID_SPEC, 
				  &former->trace_pid[i]) < 0) {
				perror("kstrax_restore_trace_pid_half(ioctl)");
				exit(1);
			}
		}
	}
}

static void kstrax_restore_status(struct kstrax_status *old, 
				  struct kstrax_status *new)
{
	int fd;

	if ((fd = open(INPUT_FILE, O_WRONLY)) < 0) {
		perror("kstrax_restore_status(open)");
		exit(1);
	}	

	__kstrax_restore_trace_syscall(old, new, fd);
	kstrax_restore_trace_pid_half(old, new, fd);
	kstrax_restore_trace_pid_half(new, old, fd);

	close(fd);
}


static void kstrax_set_default_bin_filename(char *filename, const char *optarg)
{
	char host[HOST_NAME_MAX];
	time_t now;
	struct tm *my_time;
	
	gethostname(host, HOST_NAME_MAX);
	if (strlen(host) > (KSTRAX_FNAME_MAX - 13)) {
		printf("Too long filename\n");
		exit(1);
	}
	time(&now);
	my_time = localtime(&now);
	sprintf(filename, "%02d%02d%02d.%02d%02d_%s",
		my_time->tm_year-100, 
		my_time->tm_mon+1, 
		my_time->tm_mday, 
		my_time->tm_hour,
		my_time->tm_min, host);
}

static void kstrax_set_default_txt_filename(const char *ifile, char *ofile)
{
	if (kstrax_rflag == 1)
		sprintf(ofile, "%s_raw", ifile);
	else if (kstrax_cflag == 1)
		sprintf(ofile, "%s_stat", ifile);
	else
		sprintf(ofile, "%s_txt", ifile);
}

static void kstrax_set_output_filename(char *ofile, const char *optarg)
{
	if (strlen(optarg) >= KSTRAX_FNAME_MAX) {
		printf("Too long filename\n");
		exit(1);
	}		
	strncpy(ofile, optarg, KSTRAX_FNAME_MAX);
	
}

static void kstrax_usage(void)
{
	printf("\nThis controls status of KSTRAX.\n");
	printf("Usage:\n\t%s [option(s)]\n\n", NAME);
	printf("<OPTION>\n");
	printf("-b \t\t\tRetrieve trace data from kernel buffer.\n");
	printf("-c [filename]\t\tPrint statistics information"
	       " from binary file[name]\n");
	printf("-e [syscall(type)]\tSet tracing syscall or syscall type"
	       "[ALL/FILE/NETWORK/IPC/PROCESS/SIGNAL]\n");
	printf("-h\t\t\tPrint this message.\n");
	printf("-k\t\t\tNot clear kernel buffer\n");
	printf("-o [filename]\t\tSet output file name\n");
	printf("-p [pid]\t\tSet tracing pid[pid]\n");
	printf("-r [filename]\t\tPrint system call information on console"
	       " from binary file[name] at RAW mode\n");
	printf("-R \t\t\tOutput buffer data (snap shot)\n");
	printf("-s [number of entry]\tSet buffer size to [number of entry]\n");
	printf("-S\t\t\tPrint trace status\n");
	printf("-t [filename]\t\tPrint system call information"
	       " from binary file[name] to text\n");
	printf("-T [sec]\t\tPrint binary file for [sec]seconds\n");
	printf("-v\t\t\tPrint Version information\n\n");
	exit(0);
}

static void kstrax_version(void)
{
        printf("%s command version %s\n%s\n", NAME, KSTRAX_VERSION, COPYRIGHT);
	exit(0);
}

static int kstrax_strtol(const char *string)
{
	int retval; 
	char *ptr;
	
	retval = strtol(string, &ptr, 10);
	if (*ptr == 0 && *string != 0) 
		return retval; /* success */
	
	fprintf(stderr, "bad argument(%s)\n", string);  
	exit(1);
}

static void kstrax_set_snapshot(void)
{
	int fd;

	if ((fd = open(INPUT_FILE, O_WRONLY)) < 0) {
		perror("kstrax_set_snapshot(open)");
		exit(1);
	}
	if (ioctl(fd, KSTRAX_IOC_SET_CUR_INDEX) < 0) {
		perror("kstrax_set_snapshot(ioctl)");
		exit(1);		
	}
	close(fd);
}

void kstrax_set_nr_syscalls(int num)
{
	if (ks_nr_syscalls == 0)
		ks_nr_syscalls = num;
}

int kstrax_get_nr_syscalls(void)
{
	return ks_nr_syscalls;
}
