/* Output a convertion table from UCS character to multibyte character
 * Copyright (C) 2006  MIRACLE LINUX CORPORATION.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
#include <stdio.h>
#include <iconv.h>
#include <errno.h>

typedef unsigned long ucs4_t;

#define MAXMB   10
#define MAXUCS4 10

void
print_ucs2mb_one_char(ucs4_t *ucs4, size_t inlen, unsigned char *mb, size_t outlen)
{
    int i;

    for (i = 0; i < inlen; i++) {
        printf("<U%04X>", ucs4[i]);
    }
    putchar(' ');
    for (i = 0; i < outlen; i++) {
        printf("\\x%02X", mb[i]);
    }
    putchar('\n');
}

void
error_ucs2mb_one_char(ucs4_t *ucs4, size_t inlen, int pos)
{
    int i;

    for (i = 0; i < inlen; i++) {
        fprintf(stderr, "<U%04X>", ucs4[i]);
    }
    fputc(' ', stderr);
    switch (errno) {
    case E2BIG:
        fprintf(stderr, "iconv(): E2BIG\n");
        break;
    case EILSEQ:
        fprintf(stderr, "iconv(): EILSEQ(%d)\n", pos);
        break;
    case EINVAL:
        fprintf(stderr, "iconv(): EINVAL\n");
        break;
    default:
        perror("iconv()");
        break;
    }
}

void
dump_ucs2mb_plane(iconv_t cd, int plane)
{
    ucs4_t u;
    unsigned char mb[MAXMB];
    ucs4_t ucs4[MAXUCS4];
    char *inbuf, *outbuf;
    size_t inbytesleft, outbytesleft;
    size_t r;
    int i, outlen;

    for (u = 0; u < 0x10000; u++) {
        if (plane == 0 && u >= 0xD800 && u <= 0xDFFF)
            continue;
        ucs4[0] = plane << 16 | u;

        inbuf = (char *)ucs4;
        outbuf = (char *)mb;
        inbytesleft = (1 * sizeof(ucs4_t));
        outbytesleft = sizeof(mb);

        if (iconv(cd, &inbuf, &inbytesleft, &outbuf, &outbytesleft)
            != (size_t)(-1))
        {
            print_ucs2mb_one_char(ucs4, 1, mb, outbuf - (char *)mb);
        } else {
            error_ucs2mb_one_char(ucs4, 1, inbuf - (char *)ucs4);
        }
    }
}

void
dump_ucs2mb(const char *codeset)
{
    iconv_t cd;
    ucs4_t endian = 0x1234;
    char *ucs4_codeset;

    if (*((unsigned char *)&endian) == 0)
        ucs4_codeset = "UCS-4BE";
    else
        ucs4_codeset = "UCS-4LE";

    if ((cd = iconv_open(codeset, ucs4_codeset)) == (iconv_t)(-1)) {
        fprintf(stderr, "%s is not suppoted.\n", codeset);
        exit(1);
    }
    dump_ucs2mb_plane(cd, 0);
    dump_ucs2mb_plane(cd, 2);

    iconv_close(cd);
}

int
main(int argc, char *argv[])
{
    char *codeset;

    if (argc != 2) {
        fprintf(stderr, "Usage: ucs2mb codeset\n");
        return 1;
    }
    codeset = argv[1];

    dump_ucs2mb(codeset);

    return 0;
}
