require_dependency 'wiki_renderer'
require_dependency 'html_element_identifier'

class Tiddler < ActiveRecord::Base
  has_many :aliases, :dependent => true, :order => "name"
  has_many :reminders, :class_name => "LesserWiki::Reminder",
           :order => "date, priority_type, content",
           :dependent => true
  
  acts_as_taggable
  acts_as_versioned
  
  after_validation :force_disable_autolink
  
  # localize field names
  field_name :title, _("attribute_title")
  field_name :content, _("attribute_content")
  
  # declare HTML elements and define convenient methods
  # (see lib/html_element_indentifier.rb)
  # these methods are used in exchanges of arguments between Rails and
  # JavaScript
  # commonly, these elements are converted into JSON in helper methods
  include HTMLElementIdentifier
  html_element :element, "tiddler_"
  html_element :subtitle
  html_element :toolbar
  html_element :tagged
  html_element :viewer
  html_element :form
  html_element :loading
  html_element :editor
  html_element :editor_title
  html_element :editor_content
  html_element :editor_submit
  html_element :colorbox
  html_element :color_select
  html_element :toolbar_close
  html_element :toolbar_close_others
  html_element :toolbar_edit
  html_element :toolbar_permalink
  html_element :toolbar_done
  html_element :toolbar_cancel
  html_element :toolbar_delete
  html_element :listener
  
  cattr_reader :start_here_title, :main_menu_title, :not_found_title,
               :login_required_title
  
  @@start_here_title = "StartHere"
  @@main_menu_title = "MainMenu"
  @@not_found_title = "NotFound"
  @@login_required_title = "LoginRequired"
  
  cattr_reader :special_titles
  @@special_titles = [
    @@start_here_title, @@main_menu_title, @@not_found_title,
    @@login_required_title
  ]
  
  cattr_reader :cannot_save_titles
  @@cannot_save_titles = [
    @@not_found_title, @@login_required_title
  ]
  
  cattr_reader :colors
  @@colors = %w[red orange yellow green blue purple gray]
  
  cattr_reader :colors_with_none
  @@colors_with_none = ["none"] + @@colors
  
  # echo -n "lessserwiki" | sha1sum
  @@salt = "5458b6ab4cde235a4598d4b53a04f83c044f8932"
  cattr_accessor :salt
  
  # Active Record's default error messages
  # @@default_error_messages = {
  #   :inclusion => "is not included in the list",
  #   :exclusion => "is reserved",
  #   :invalid => "is invalid",
  #   :confirmation => "doesn't match confirmation",
  #   :accepted  => "must be accepted",
  #   :empty => "can't be empty",
  #   :blank => "can't be blank",
  #   :too_long => "is too long (max is %d characters)",
  #   :too_short => "is too short (min is %d characters)",
  #   :wrong_length => "is the wrong length (should be %d characters)",
  #   :taken => "has already been taken",
  #   :not_a_number => "is not a number"
  # }
  
  validates_uniqueness_of :title, :message => _("error_taken")
  validates_length_of :title, :in => 1..40,
                      :too_short => _("error_too_short"),
                      :too_long => _("error_too_long")
  validates_length_of :content, :in => 1..10000,
                      :too_short => _("error_too_short"),
                      :too_long => _("error_too_long")
  validates_exclusion_of :title,
                         :in => @@cannot_save_titles,
                         :message => _("error_exclusion")
  validates_inclusion_of :color,
                         :in => @@colors_with_none,
                         :message => _("error_inclusion")
  
  def self.total_number
    count_by_sql("select count(*) from tiddlers")
  end
    
  def self.start_here
    find_by_title(@@start_here_title) || create_start_here
  end
  
  def self.main_menu
    find_by_title(@@main_menu_title) || create_main_menu
  end
  
  def self.create_start_here
    tiddler = create(:title => @@start_here_title,
                     :content => "Welcome to #{config_values[:wiki_title]}!",
                     :autolink => true,
                     :color => "none")
  end
  
  def self.create_main_menu
    create(:title => @@main_menu_title,
           :content => "StartHere\nMainMenu",
           :autolink => true,
           :color => "none")
  end
  
  def self.new_not_found(title = nil)
    if title
      self.new(:title => @@not_found_title,
               :content => "*#{_('not_found_title', title)}*",
               :created_at => Time.now,
               :updated_at => Time.now)
    else
      self.new(:title => @@not_found_title,
               :content => "*#{_('not_found')}*",
               :created_at => Time.now,
               :updated_at => Time.now)
    end
  end
  
  def self.new_login_required
    self.new(:title => @@login_required_title,
             :content => "*#{_('login_required')}*",
             :created_at => Time.now,
             :updated_at => Time.now)
  end
  
  def rendered_content(renderer = WikiRenderer.instance)
    @tiddlers ||= Tiddler.find(:all, :order => "length(title) desc, title")
    @aliases ||= Alias.find(:all, :order => "length(name) desc, name")
    
    engine = nil
    case config_values["markup_engine"]
    when "textile"
      engine = Engines::Textile
    when "markdown"
      engine = Engines::Markdown
    end
    
    renderer.render(self, engine, @tiddlers, @aliases)
  end
  
  def alias_names
    self.aliases
  end
  
  def alias_names_with_brackets
    alias_names.map {|a| a.name.enclose_with_brackets }
  end
  
  def alias_names_str
    alias_names_with_brackets.join(" ")
  end
  
  def tag_names
    self.tags.sort_by {|t| t.name }
  end
  
  def tag_names_with_brackets
    tag_names.map {|t| t.name.enclose_with_brackets }
  end
  
  def tag_names_str
    tag_names_with_brackets.join(" ")
  end
  
  def color_name
    if @@colors_with_none.include?(self.color)
      self.color
    else
      "none"
    end
  end
  
  def author
    self.written_by ? self.written_by : _("anonymous")
  end
  
  def adjusted_updated_at(fmt = _("datetime_format"))
    adjust_time(self.updated_at, fmt)
  end
  
  def adjusted_created_at(fmt = _("datetime_format"))
    adjust_time(self.created_at, fmt)
  end
  
  def digest
    Digest::SHA1.hexdigest("#{salt}--#{self.attributes}--")
  end
  
  protected
  
  def adjust_time(time, fmt = _("datetime_format"))
    zone = TimeZone[config_values[:time_zone]] || TimeZone["London"]
    zone.adjust(time).strftime(fmt)
  end
  
  def render
    self.content
  end
  
  def force_disable_autolink
    if self.title.length < 3
      self.update_attribute(:autolink, false)
    end
  end
end
