require 'wiki_words'
require 'chunks/chunk'
require 'chunks/wiki'

# Contains all the methods for finding and replacing wiki related links.
module WikiChunk
  include Chunk

  # A wiki reference is the top-level class for anything that refers to
  # another wiki tiddler.
  class WikiReference < Chunk::Abstract
    # Name of the referenced tiddler
    attr_reader :tiddler_name
    
    # the referenced tiddler
    def reftiddler
      @content.web.tiddlers[@tiddler_name]
    end
  end

  # A wiki link is the top-level class for links that refers to
  # another wiki tiddler.
  class WikiLink < WikiReference
    attr_reader :link_text, :link_type

    def initialize(match_data, content)
      super(match_data, content)
      @link_type = :show
    end

    def self.apply_to(content, tiddlers, aliases)
      content.gsub!(self.pattern) do |matched_text|
        chunk = self.new($~, content)
        if chunk.textile_url?
          # do not substitute
          matched_text
        else
          content.add_chunk(chunk)
          chunk.mask
        end
      end
    end

    # the referenced tiddler
    def reftiddler
      @content.web.tiddlers[@tiddler_name]
    end

    def textile_url?
      not @textile_link_suffix.nil?
    end
  end

  # This chunk handles [[bracketted wiki words]] and 
  # [[AliasedWords|aliased wiki words]]. The first part of an
  # aliased wiki word must be a WikiWord. If the WikiWord
  # is aliased, the +link_text+ field will contain the
  # alias, otherwise +link_text+ will contain the entire
  # contents within the double brackets.
  #
  # NOTE: This chunk must be tested before WikiWord since
  #       a WikiWords can be a substring of a WikiLink. 
  class Link < WikiLink
    unless defined? WIKI_LINK
      WIKI_LINK = /(":)?\[\[([^\]]+)\]\]/
    end
    
    def self.pattern() WIKI_LINK end

    def initialize(match_data, content)
      super
      textile_link_suffix, tiddler_name = match_data[1..2]
      @unmask_text = @content.tiddler_link(tiddler_name, tiddler_name)
    end
  end
  
  class AutoLink < WikiLink
    class LinkTarget
      attr_reader :name, :tiddler, :priority

      def initialize(name, tiddler, priority)
        @name = name
        @tiddler = tiddler
        @priority = priority
      end
      
      def title
        @tiddler.title
      end
      
      def <=>(other)
        if (len = other.name.length <=> @name.length) == 0
          if (cmp = other.name <=> @name) == 0
            other.priority <=> @priority
          else
            cmp
          end
        else
          len
        end
      end
    end
    
    def self.apply_to(content, tiddlers, aliases)
      targets = []
      tiddlers.select {|t| t.autolink == 1 or t.autolink == true }.each do |t|
        targets << LinkTarget.new(t.title, t, 1)
      end
      #Alias.find(:all, :order => "length(name) desc, name").each do |a|
      aliases.each do |a|
        targets << LinkTarget.new(a.name, a.tiddler, 0)
      end
      targets.sort.each do |t|
        content.gsub!( /(#{Regexp.escape(t.name)})(?!chunk)/ ) do |match|
          new_chunk = self.new($~, content, t)
          content.add_chunk(new_chunk)
          new_chunk.mask
        end
      end
    end
    
    def initialize(match_data, content, target)
      super(match_data, content)
      #tiddler_name = target.title
      tiddler = target.tiddler
      link_text = match_data[0]
      @unmask_text = @content.tiddler_link(tiddler, link_text)
    end
  end
end
