module LesserWiki
  class Reminder < ActiveRecord::Base
    belongs_to :tiddler
    
    TYPE_NORMAL = "-".freeze
    TYPE_TODO = "+".freeze
    TYPE_DEFER = "~".freeze
    TYPE_DEADLINE = "!".freeze
    TYPE_SCHEDULE = "@".freeze
    TYPE_DONE = ".".freeze
    
    SCHEDULE_TYPES = [ "@", "!" ].freeze
    TODO_TYPES = [ "!", "+", "-", "~" ].freeze
    
    HOWM_HUGE_MINUS = 66666.freeze
    HOWM_HUGE = 77777.freeze
    HOWM_HUGE_PLUS = 88888.freeze
    
    def equals(object)
      self.tiddler.title == object.tiddler.title and
      self.date === object.date and
      self.priority_type == object.priority_type and
      self.priority_number == object.priority_number and
      self.content == object.content
    end
    
    def type_name
      case self.priority_type
      when TYPE_NORMAL
        "Normal"
      when TYPE_TODO
        "Todo"
      when TYPE_DEFER
        "Defer"
      when TYPE_DEADLINE
        "Deadline"
      when TYPE_SCHEDULE
        "Schedule"
      when TYPE_DONE
        "Done"
      end
    end
    
    def schedule?
      SCHEDULE_TYPES.include?(self.priority_type)
    end
    
    def todo?
      TODO_TYPES.includee?(self.priority_type)
    end
    
    def late
      (Time.now - Time.local(self.date.year, self.date.mon, self.date.day)) / 60.0 / 60.0 / 24.0
    end
    
    def relative_late
      late.to_f / (self.priority_number != nil ? self.priority_number : default_priority_number).to_f
    end
    
    def default_priority_number
      case self.priority_type
      when TYPE_NORMAL
        1
      when TYPE_TODO
        7
      when TYPE_DEFER
        30
      when TYPE_DEADLINE
        7
      when TYPE_SCHEDULE
        1
      else
        1
      end
    end

    def priority_init
      case self.priority_type
      when TYPE_TODO
        -7
      when TYPE_DEFER
        -14
      when TYPE_DEADLINE
        -2
      else
        0
      end
    end

    def priority_peek
      0
    end

    def priority_bottom
      case self.priority_type
      when TYPE_DONE
        - HOWM_HUGE_PLUS
      else
        - HOWM_HUGE
      end
    end

    def priority_top
      HOWM_HUGE
    end
    
    def priority
      case self.priority_type
      when TYPE_NORMAL
        r = relative_late
        if r < 0
          priority_bottom + r
        else
          - r
        end
      when TYPE_TODO
        r = relative_late
        if r < 0
          priority_bottom + r
        else
          (- priority_init) * (r - 1)
        end
      when TYPE_DEFER
        r = relative_late
        if r < 0
          priority_bottom + r
        else
          priority_peek - (priority_peek - priority_init) * ((r % 1) - 0.5).abs * 2
        end
      when TYPE_DEADLINE
        r = self.relative_late
        if r > 0
          priority_top + r
        elsif r < -1
          priority_bottom + r
        else
          (- priority_init) * r
        end
      when TYPE_SCHEDULE
        r = relative_late
        if r > 0
          priority_bottom + r
        else
          r
        end
      else
        priority_bottom + late
      end
    end
  end
end
