require_dependency 'html_element_identifier'

# Add :level and :count to the Tag class of acts_as_taggable.
class Tag < ActiveRecord::Base
  has_many :taggings
  
  # added by LesserWiki
  attr_accessor :level, :count
  
  include HTMLElementIdentifier
  html_element :element, "tag_"
  html_element :toolbar, "tag_toolbar_"
  html_element :toolbar_close, "tag_toolbar_close_"
  html_element :toolbar_close_others, "tag_toolbar_close_others_"
  html_element :toolbar_permalink, "tag_toolbar_permalink_"
  html_element :handler, "tag_handler_"
  html_element :tagging, "tag_tagging_"

  def self.parse(list)
    tag_names = []

    # first, pull out the quoted tags
    list.gsub!(/\"(.*?)\"\s*/ ) { tag_names << $1; "" }

    # then, replace all commas with a space
    list.gsub!(/,/, " ")

    # then, get whatever's left
    tag_names.concat list.split(/\s/)

    # strip whitespace from the names
    tag_names = tag_names.map { |t| t.strip }

    # delete any blank tag names
    tag_names = tag_names.delete_if { |t| t.empty? }
    
    return tag_names
  end

  def tagged
    @tagged ||= taggings.collect { |tagging| tagging.taggable }
  end
  
  def on(taggable)
    taggings.create :taggable => taggable
  end
  
  def ==(comparison_object)
    super || name == comparison_object.to_s
  end
  
  def to_s
    name
  end
end

class TagCloud
  def initialize
    @counts = {}
  end
  
  def add(tag, count)
    @counts[tag] = tag.count = count
  end
  
  def calculate(limit = nil)
    tags = @counts.sort_by {|a, b| b }.reverse.map {|a, b| a }
    tags = tags[0..limit-1] if limit
    if tags.empty?
      return ""
    elsif tags.size == 1
      tag = tags[0]
      tag.level = 24
      return [tag]
    end
    
    min = Math.sqrt(@counts[tags.last])
    max = Math.sqrt(@counts[tags.first])
    factor = 0
    
    # special case all tags having the same count
    if max - min == 0
      min = min - 24
      factor = 1
    else
      factor = 24 / (max - min)
    end
    
    tags.each do |tag|
      tag.level = ((Math.sqrt(tag.count) - min) * factor).to_i
    end
    
    tags.sort_by {|t| t.name }
  end
end