require File.dirname(__FILE__) + '/../../../../test/test_helper'

class TestValidationRules < Test::Unit::TestCase
  self.fixture_path = File.dirname(__FILE__) + '/fixtures/'
  
  fixtures :users

  def setup
    @params = {
      :attribute_confirmation => 'password',
      :password_confirm => 'password'
    }
    @validator = ActiveRecord::Validations::Validator.new(@params)
  end
  
  # validation options
  
  def test_perform_validation_strip
    @validator.send(:perform_validation, :attribute, ' dsdsds      w           ')
    assert_equal('dsdsds      w', @validator.params[:attribute])
    @validator.send(:perform_validation, :attribute, ' dsdsds      w           ', :strip => false)
    assert_equal(' dsdsds      w           ', @validator.params[:attribute])
    @validator.send(:perform_validation, :attribute, '        ', :filled => :optional)
    assert_equal('', @validator.params[:attribute])
    @validator.send(:perform_validation, :attribute, '        ', :strip => false)
    assert_equal('        ', @validator.params[:attribute])
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, '        ')
    assert_equal(ActiveRecord::Errors.default_error_messages[:blank], @validator.errors.on(:attribute))
  end
  
  def test_perform_validation_filled
    @validator.send(:perform_validation, :attribute, nil, :filled => :optional)
    @validator.send(:perform_validation, :attribute, '', :filled => :optional)
    @validator.send(:perform_validation, :attribute, 'josh', :filled => :optional)
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, nil)
    assert_equal(ActiveRecord::Errors.default_error_messages[:blank], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, '')
    assert_equal(ActiveRecord::Errors.default_error_messages[:blank], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 'josh')
    assert_nil(@validator.errors.on(:attribute))   
  end
  
  def test_perform_validation_number
    @validator.send(:perform_validation, :attribute, -22, :number => true)
    @validator.send(:perform_validation, :attribute, 5445, :number => true)
    @validator.send(:perform_validation, :attribute, -64.5546, :number => true)
    @validator.send(:perform_validation, :attribute, 4556.5454, :number => true)
    @validator.send(:perform_validation, :attribute, 2323, :number => :integer)
    @validator.send(:perform_validation, :attribute, -3434, :number => :integer)
    @validator.send(:perform_validation, :attribute, 5434, :number => :unsigned_integer)
    @validator.send(:perform_validation, :attribute, 14.43, :number => :unsigned)
    @validator.send(:perform_validation, :attribute, 1454, :number => :unsigned)
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, 'fd', :number => true)
    assert_equal(ActiveRecord::Errors.default_error_messages[:not_a_number], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 367.76, :number => :integer)
    assert_equal(ActiveRecord::Errors.default_error_messages[:not_a_integer], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 5434.54, :number => :unsigned_integer)
    assert_equal(ActiveRecord::Errors.default_error_messages[:not_a_integer], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, -43, :number => :unsigned_integer)
    assert_equal(ActiveRecord::Errors.default_error_messages[:not_a_unsigned], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, -43.12, :number => :unsigned)
    assert_equal(ActiveRecord::Errors.default_error_messages[:not_a_unsigned], @validator.errors.on(:attribute))
    @validator.errors.clear
  end
  
  def test_perform_validation_length
    @validator.send(:perform_validation, :attribute, 'asdfdsf', :length => 7)
    @validator.send(:perform_validation, :attribute, 'as', :length => 2..7)
    @validator.send(:perform_validation, :attribute, 'asdfdsf', :length => 2..7)
    @validator.send(:perform_validation, :attribute, 'asdfdsf', :length => {:min => 7})
    @validator.send(:perform_validation, :attribute, 'asdfdsf', :length => {:max => 7})
    assert_nil(@validator.errors.on(:attribute))

    @validator.send(:perform_validation, :attribute, 'asdfdsf', :length => 2)
    assert_equal(ActiveRecord::Errors.default_error_messages[:wrong_length] % 2, @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 'as', :length => 3..7)
    assert_equal(ActiveRecord::Errors.default_error_messages[:too_short] % 3, @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 'asdfdsf', :length => 2..6)
    assert_equal(ActiveRecord::Errors.default_error_messages[:too_long] % 6, @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 'asdfds', :length => {:min => 7})
    assert_equal(ActiveRecord::Errors.default_error_messages[:too_short] % 7, @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 'e', :length => {:max => 0})
    assert_equal(ActiveRecord::Errors.default_error_messages[:too_long] % 0, @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 'asdfdsfe', :length => {:max => 7})
    assert_equal(ActiveRecord::Errors.default_error_messages[:too_long] % 7, @validator.errors.on(:attribute))
  end
  
  def test_perform_validation_format
    @validator.send(:perform_validation, :attribute, 'asdfdsf', :format => /^[a-z]+$/)
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, 'e34r', :format => /^[a-z]+$/)
    assert_equal(ActiveRecord::Errors.default_error_messages[:invalid], @validator.errors.on(:attribute))
  end
  
  def test_perform_validation_include
    @validator.send(:perform_validation, :attribute, 'two', :include => ['one', 'two', 'three'])
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, 'four', :include => ['one', 'two', 'three'])
    assert_equal(ActiveRecord::Errors.default_error_messages[:inclusion], @validator.errors.on(:attribute))
  end
  
  def test_perform_validation_exclude
    @validator.send(:perform_validation, :attribute, 'four', :exclude => ['one', 'two', 'three'])
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, 'two', :exclude => ['one', 'two', 'three'])
    assert_equal(ActiveRecord::Errors.default_error_messages[:exclusion], @validator.errors.on(:attribute))
  end
  
  def test_perform_validation_confirm
    @validator.send(:perform_validation, :attribute, 'password', :confirm => true)
    @validator.send(:perform_validation, :attribute, 'password', :confirm => :password_confirm)
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, 'passwrd', :confirm => true)
    assert_equal(ActiveRecord::Errors.default_error_messages[:confirmation], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 'passwrd', :confirm => :password_confirm)
    assert_equal(ActiveRecord::Errors.default_error_messages[:confirmation], @validator.errors.on(:attribute))
  end
  
  def test_perform_validation_accept
    @validator.send(:perform_validation, :attribute, '1', :accept => true)
    @validator.send(:perform_validation, :attribute, 'accept', :accept => 'accept')
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, '2', :accept => true)
    assert_equal(ActiveRecord::Errors.default_error_messages[:accepted], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 'true', :accept => 'accept')
    assert_equal(ActiveRecord::Errors.default_error_messages[:accepted], @validator.errors.on(:attribute))
  end
  
  def test_perform_validation_unique
    @validator.send(:perform_validation, :attribute, 'max', :unique => 'User.nick')
    @validator.send(:perform_validation, :attribute, 23, :unique => 'User')
    @validator.send(:perform_validation, :attribute, 6, :unique => User)
    @validator.send(:perform_validation, :attribute, 'josh', :unique => ['User.nick', 1])
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, 'josh', :unique => 'User.nick')
    assert_equal(ActiveRecord::Errors.default_error_messages[:taken], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 1, :unique => 'User')
    assert_equal(ActiveRecord::Errors.default_error_messages[:taken], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 2, :unique => User)
    assert_equal(ActiveRecord::Errors.default_error_messages[:taken], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 'pere', :unique => ['User.nick', 1])
    assert_equal(ActiveRecord::Errors.default_error_messages[:taken], @validator.errors.on(:attribute))
  end
  
  def test_perform_validation_associated
    @validator.send(:perform_validation, :attribute, 'josh', :associated => 'User.nick')
    @validator.send(:perform_validation, :attribute, 1, :associated => 'User')
    @validator.send(:perform_validation, :attribute, 2, :associated => User)
    assert_nil(@validator.errors.on(:attribute))

    @validator.send(:perform_validation, :attribute, 'david', :associated => 'User.nick')
    assert_equal(ActiveRecord::Errors.default_error_messages[:associated], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 23, :associated => 'User')
    assert_equal(ActiveRecord::Errors.default_error_messages[:associated], @validator.errors.on(:attribute))
    @validator.errors.clear
    @validator.send(:perform_validation, :attribute, 7, :associated => User)
    assert_equal(ActiveRecord::Errors.default_error_messages[:associated], @validator.errors.on(:attribute))
  end
  
  def test_perform_validation_block
    @validator.send(:perform_validation, :attribute, 'josh') { |v, a, v| true }
    assert_nil(@validator.errors.on(:attribute))
    
    @validator.send(:perform_validation, :attribute, 'josh') do |validator, attribute, value|
      validator.errors.add(attribute, ActiveRecord::Errors.default_error_messages[:invalid])
      false
    end
    assert_equal(ActiveRecord::Errors.default_error_messages[:invalid], @validator.errors.on(:attribute))
  end
end