﻿/*
"Copyright (first published1 date)2 (most recent published date)3 Intel Corporation.

The source code, information and material ("Material") contained herein is owned by Intel Corporation or its 
suppliers or licensors, and title to such Material remains with Intel Corporation or its suppliers or licensors. 
The Material contains proprietary information of Intel or its suppliers and licensors. The Material is 
protected by worldwide copyright laws and treaty provisions. No part of the Material may be used, copied, 
reproduced, modified, published, uploaded, posted, transmitted, distributed or disclosed in any way without Intel's 
prior express written permission. No license under any patent, copyright or other intellectual property rights in 
the Material is granted to or conferred upon you, either expressly, by implication, inducement, estoppel or otherwise. 
Any license under such intellectual property rights must be express and approved by Intel in writing.

Include any supplier copyright notices as supplier requires Intel to use.

Include supplier trademarks or logos as supplier requires Intel to use, preceded by an asterisk. An asterisked footnote 
can be added as follows: *Third Party trademarks are the property of their respective owners.

Unless otherwise agreed by Intel in writing, you may not remove or alter this notice or any other notice 
embedded in Materials by Intel or Intel's suppliers or licensors in any way."

*/

using System;


/*
 * Author   : Rick Blacker
 * Date     : June / 2015
 * 
 * Abstract :   RSVoice is a wrapper class around the Voice Recognition of the RealSense SDK.
 *              It is used in command mode and only recognizes three words "Fire", "Bomb", "Laser" 
 *              This class was created to support speech in a Unity game that I am authoring. 
 */


namespace RSVoiceSample
{

    class RSVoice
    {
        // Name of the device which has the microphone. In this case the RealSense camera.
        string DEVICE_NAME              = "Microphone Array (Creative VF0800)";

        // The name of the module we are using.
        //string MODULE_FRIENDLY_NAME     = "Voice Recognition (Nuance*)";

        // The confidence level when detecting words.  Score. 
        int CONFIDENCE_LEVEL            = 50;


        // The session that ALL RealSense apps must use.
        private PXCMSession                     _session;
        
        // Holds the audio source (Microphone)
        private PXCMAudioSource                 _audioSource;

        // Information about the audio device (Microphone)
        private PXCMAudioSource.DeviceInfo      _audioDeviceInfo;

        // The speech recognition module
        private PXCMSpeechRecognition           _speechRecognition;

        // Speech event handler when it has detected a recognized word.
        private PXCMSpeechRecognition.Handler   _handler;


        // Event to let client know when a word has been detected
        public event EventHandler<WordEventArg>     OnWordDetected;


        // Lets the client app know that it's ok for the client to call the Destroy method.
        // The only reason this is needed is because the _speechRecognition.StopRec( ); takes
        // a bit long to fully stop. If you try to call "Destroy" befor _speechRecognition.StopRec( );
        // has completely finished execution, you get a runtime error. So, to avoid this, I created
        // an event handler to allow the client app to know when it's safe to call "Destroy"
        public event EventHandler<EventArgs>        OnStopProcessing;



        // The words we want to try to detect
        private string[] _commandWords = { "Fire", "Bomb", "Laser" };

        // Lets client know if the class was initialized properly.
        private bool _initialized   = false;

        // Flag to allow engine to run or stop
        private bool _run           = false;
        
        
        /// <summary>
        /// Constructor, initializes the voice recognition and gets it ready to run.
        /// </summary>
        public RSVoice( )
        {
            _initialized = false;
        
            try
            {
                // Create the session and audio source and the speech recognition interface.
                _session = PXCMSession.CreateInstance( );
                if( _session == null )
                    throw new Exception( "Failed to create RealSense Session" );

                // Initialize the main module
                if( !CreateSpeechRecognitionModule( ) )
                    throw new Exception( "Failed to create Speech Recognition module" );

                // Tell it what commands we want to make use of
                if( !CreateSpeechModuleGrammar( ) )
                    throw new Exception( "Failed to create Speech Recognition gramar" );

                // Tell it the audio source
                if( !CreateAudioSource( ) )
                    throw new Exception( "Failed to create audio source" );

                if( !CreateSpeechHandler( ) )
                    throw new Exception( "Failed to create Speech Handler" );

                _initialized = true;

                //pxcmStatus status = _speechRecognition.StartRec( _audioSource, _handler );

            }
            catch( Exception e )
            {
                Dispose( );
                throw e;
            }
        }



        /// <summary>
        /// Disposes and destroys all class global objects.
        /// </summary>
        public void Dispose( )
        {
            if( _audioSource != null )
                _audioSource.Dispose( );

            if( _speechRecognition != null )
                _speechRecognition.Dispose( );
            
            if( _session != null )
                _session.Dispose( );

            _session            = null;
            _audioSource        = null;
            _speechRecognition  = null;
            _initialized        = false;
        }



        /// <summary>
        /// Getter for the Initialized flag. Should be used to ensure that the class was properly initialized before attempting
        /// to run speech recognition.
        /// </summary>
        public bool Initialized
        {
            get
            {
                return _initialized;
            }
        }


        
        /// <summary>
        ///  Sets up the speech recognition module
        /// </summary>
        /// <returns>true if the module was succesfully created, false otherwise.</returns>
        private bool CreateSpeechRecognitionModule( )
        {
            pxcmStatus status;

            status = _session.CreateImpl<PXCMSpeechRecognition>( out _speechRecognition );
            
            if( status < pxcmStatus.PXCM_STATUS_NO_ERROR )
                return false;

            return true;
        }





        /// <summary>
        /// Creates the grammer data used by the speech recognition engine.
        /// </summary>
        /// <returns>True if the grammar was able to be created, false otherwise</returns>
        private bool CreateSpeechModuleGrammar( )
        {
            bool created = false;
            pxcmStatus status;

            status = _speechRecognition.BuildGrammarFromStringList( 1, _commandWords, null );
            
            _speechRecognition.SetGrammar( 1 );

            if( status == pxcmStatus.PXCM_STATUS_NO_ERROR )
                created = true;

            return created;
        }



        /// <summary>
        /// Creates the audio source information used by the Speech Recognition module.
        /// </summary>
        /// <returns>True if the audio source was sucesfully created, false otherwise</returns>
        private bool CreateAudioSource( )
        {
            bool created = false;
            // Create the audio source object and ensure it's not null, if ok, then load the commands
            _audioSource = _session.CreateAudioSource( );
            if( _audioSource != null )
            {
                // Create and get the device information that we want to use
                // IE plantronics headset.
                if( CreateDeviceInfo( ) )
                {
                    _audioSource.SetVolume( 0.2f );
                    _audioSource.SetDevice( _audioDeviceInfo );
                    created = true;
                }

            }
            return created;
        }


              
        /// <summary>
        /// Creates the device information for the audio device. In this case, the plantronics headset.
        /// </summary>
        /// <returns>True of the device info was created succesfully, false otherwise</returns>
        private bool CreateDeviceInfo( )
        {
            bool        created = false;
            pxcmStatus  status;
            _audioSource.ScanDevices( );

            // Now iterate over all the scanned devices looking for the specific plantronic headset.
            // Understand that there could be other devices that are available, but we don't want those
            // in this example, only interested in the RealSense Creative cameara specified in DEVICE_NAME!
            for( int i = 0; ; i++  )
            {
                status = _audioSource.QueryDeviceInfo( i, out _audioDeviceInfo );
                if( status == pxcmStatus.PXCM_STATUS_NO_ERROR )
                {
                    if( _audioDeviceInfo.name == DEVICE_NAME )
                    {
                        created = true;
                        break;
                    }
                }
                else
                {
                    // didn't find it
                    break;
                }
            }
            return created;
        }



        /// <summary>
        /// Creates the handler that processes the data when a word has been recognised.
        /// </summary>
        /// <returns>True if the handler was created successfully, false otherwise</returns>
        private bool CreateSpeechHandler( )
        {
            bool created = false;
            _handler = new PXCMSpeechRecognition.Handler( );
            if( _handler != null )
            {
                _handler.onRecognition = OnSpeechRecognition;
                created = true;
            }
            return created;
        }




        /// <summary>
        /// The event handler when the speech engine has encountered a word it recognizes.
        /// </summary>
        /// <param name="data">A fully initialized RecognitionData object.</param>
        private void OnSpeechRecognition( PXCMSpeechRecognition.RecognitionData data )
        {
            string s = "";
            for( int i = 0; i < data.scores.Length; i++ )
            {
                if( data.scores[ i ].confidence > CONFIDENCE_LEVEL )
                {
                    s = data.scores[ i ].sentence;
                    OnWordDetected( this, new WordEventArg( s ) );
                }
            }
        }



        /// <summary>
        /// Starts the speech engine. Assumptions are that the class was already initialized properly.
        /// Client should use the Initialized property to check.
        /// </summary>
        public void StartSpeechRecognition( )
        {
            if( !_initialized )
                return;

            _run = true;
            pxcmStatus status = _speechRecognition.StartRec( _audioSource, _handler );

            /*
            if( status == pxcmStatus.PXCM_STATUS_NO_ERROR )
            {
                // Keep processing the speech recognition 
                while( _run )
                {
                    System.Threading.Thread.Sleep( 5 );
                }

                if( _speechRecognition != null )
                {
                    _speechRecognition.StopRec( );
                }

                // Fire the event that tells the client app it's ok to clean up the resourses by calling the Dispose() method
                OnStopProcessing( this, new EventArgs( ) );
            }
             */
        }


        


        /// <summary>
        /// Triggers the speech recognition engine to stop.
        /// </summary>
        public void StopSpeechRecognition( )
        {
            _speechRecognition.StopRec( );
            OnStopProcessing( this, new EventArgs( ) );
            _run = false;
        }
    }
}
