// 簡単！実践！ロボットシミュレーション
// Open Dynamics Engineによるロボットプログラミング
// 出村公成著, 森北出版 (2007) http://demura.net/
// このプログラムは上本のサンプルプログラムです．
// プログラム 6.4:  ３自由度ロボットアーム(逆運動学実装）
// arm3.cpp by Kosei Demura (2007-5-18)
// 2008-7-7: dInitODE(),dCloseODE()の追加
#include <stdio.h>
#include <stdlib.h>
#include <ode/ode.h>
//#include <drawstuff/drawstuff.h>
#include <irrdrawstuff.h>

#ifdef _MSC_VER
#pragma warning(disable:4244 4305)  // for VC++, no precision loss complaints
#endif

#ifdef dDOUBLE
#define dsDrawCapsule  dsDrawCapsuleD
#define dsDrawCylinder dsDrawCylinderD
#define dsDrawBox      dsDrawBoxD
#define dsDrawSphere   dsDrawSphereD
#endif
#define NUM 4                          // リンク数

dWorldID      world;                   // 動力学計算用のワールド
dSpaceID      space;                   // 衝突検出用のスペース
dGeomID       ground;                  // 地面のジオメトリID番号
dJointGroupID contactgroup;            // 接触点グループ
dJointID      joint[NUM];              // ジョイントのID番号
dsFunctions   fn;                      // ドロースタッフの描画関数

typedef struct {
	dBodyID body;                        // ボディのID番号
	dGeomID geom;                        // ジオメトリのID番号
} MyObject;                            // MyObject構造体

MyObject rlink[NUM];                   // リンク

dBodyID       sensor;                  // センサ用のボディID
dJointID      sensor_joint;            // センサ固定用の関節
int           ANSWER = 1;              // 逆運動学の解

dReal P[3] = {1.0, 1.0, 1.5};             // 先端の位置
dReal a[3], b[3];                         // 有顔ベクトル(a,b)
dReal THETA[NUM] = {0.0, 0.0, 0.0, 0.0};  // 関節の目標角度[rad]
dReal l[4] = { 0.10, 0.90, 1.00, 1.00};   // リンクの長さ[m]

// センサの作成
void makeSensor()
{
	dMass mass;
	double sx = 0.0, sy = 0.0, sz = 3.0;  // センサの初期座標[m]
	double size = 0.01, weight = 0.00001; // センサのサイズ[m]と重量[kg]

	sensor = dBodyCreate(world);          // センサの生成
	dBodySetPosition(sensor,sx,sy,sz);
	dMassSetZero(&mass);
	dMassSetBoxTotal(&mass,weight,size,size,size);
	dBodySetMass(sensor, &mass);

	sensor_joint = dJointCreateFixed(world, 0); // 固定ジョイントの生成
	dJointAttach(sensor_joint, rlink[NUM-1].body, sensor); // 先端リンクと結合
	dJointSetFixed(sensor_joint);
}

// センサ位置の表示
void printSensorPosition()
{
	double *pos = (double *) dBodyGetPosition(sensor);
	printf("Current Position: x=%6.2f y=%6.2f z=%6.2f \n",pos[0],pos[1],pos[2]);
	// printf("P : x=%5.2f y=%5.2f z=%5.2f \n",P[0],P[1],P[2]);
}

/*** ロボットアームの生成 ***/
void  makeArm()
{
	dMass mass;                                    // 質量パラメータ
	dReal x[NUM]      = {0.00, 0.00, 0.00, 0.00};  // 重心 x
	dReal y[NUM]      = {0.00, 0.00, 0.00, 0.00};  // 重心 y
	dReal z[NUM]      = {0.05, 0.50, 1.50, 2.50};  // 重心 z
	dReal length[NUM] = {0.10, 0.90, 1.00, 1.00};  // 長さ
	dReal weight[NUM] = {9.00, 2.00, 2.00, 2.00};  // 質量
	dReal r[NUM]      = {0.20, 0.04, 0.04, 0.04};  // 半径
	dReal c_x[NUM]    = {0.00, 0.00, 0.00, 0.00};  // 関節中心点 x
	dReal c_y[NUM]    = {0.00, 0.00, 0.00, 0.00};  // 関節中心点 y
	dReal c_z[NUM]    = {0.00, 0.10, 1.00, 2.00};  // 関節中心点 z
	dReal axis_x[NUM] = {0, 0, 0, 0};              // 関節回転軸 x
	dReal axis_y[NUM] = {0, 0, 1, 1};              // 関節回転軸 y
	dReal axis_z[NUM] = {1, 1, 0, 0};              // 関節回転軸 z

	// リンクの生成
	for (int i = 0; i < NUM; i++) {
		rlink[i].body = dBodyCreate(world);
		dBodySetPosition(rlink[i].body, x[i], y[i], z[i]);
		dMassSetZero(&mass);
		dMassSetCapsuleTotal(&mass,weight[i],3,r[i],length[i]);
		dBodySetMass(rlink[i].body, &mass);
		rlink[i].geom = dCreateCapsule(space,r[i],length[i]);
		dGeomSetBody(rlink[i].geom,rlink[i].body);
	}

	// ジョイントの生成とリンクへの取り付け
	joint[0] = dJointCreateFixed(world, 0);  // 固定ジョイント
	dJointAttach(joint[0], rlink[0].body, 0);
	dJointSetFixed(joint[0]);
	for (int j = 1; j < NUM; j++) {
		joint[j] = dJointCreateHinge(world, 0); // ヒンジジョイント
		dJointAttach(joint[j], rlink[j].body, rlink[j-1].body);
		dJointSetHingeAnchor(joint[j], c_x[j], c_y[j], c_z[j]);
		dJointSetHingeAxis(joint[j], axis_x[j], axis_y[j],axis_z[j]);
	}
}

/*** ロボットアームの描画 ***/
void drawArm()
{
	dReal r,length;

	dsSetColor(1.0, 1.0, 1.0);
	for (int i = 0; i < NUM; i++ ) {       // カプセルの描画
		dGeomCapsuleGetParams(rlink[i].geom, &r,&length);
		if (i != NUM -1) {
			dsDrawCapsule(dBodyGetPosition(rlink[i].body),dBodyGetRotation(rlink[i].body),length,r);
			//dsDrawCylinder(dBodyGetPosition(rlink[i].body), dBodyGetRotation(rlink[i].body),length,r)
		}
		else {
			dsDrawCylinder(dBodyGetPosition(rlink[i].body), dBodyGetRotation(rlink[i].body),length,r);
			//dsDrawCapsule(dBodyGetPosition(rlink[i].body),dBodyGetRotation(rlink[i].body),length,r);

		}
	}
}

// 位置センサの描画
void drawSensor()
{
	dReal sides[] = {0.01,0.01,0.01};

	dsSetColor(1,0,0);
	dBodyGetRotation(sensor);
	dsDrawBox(dBodyGetPosition(sensor),dBodyGetRotation(sensor),sides);
}

// 目標位置の描画
void drawP()
{
	dReal tmpP[3];
	dMatrix3 tmpR;

	tmpP[0] = P[0];
	tmpP[1] = P[1];
	tmpP[2] = P[2];

	dsSetColor(1,0,0);

	dRSetIdentity(tmpR);
	dsDrawSphere(tmpP, tmpR, 0.02);
	//printf("P= %f %f %f \n",tmpP[0],tmpP[1],tmpP[2]);
}

/*** 制御 ***/
void Pcontrol()
{
	dReal k =  10.0, fMax = 100.0;                   // 比例ゲイン，最大トルク

	for (int j = 1; j < NUM; j++) {
		dReal tmp = dJointGetHingeAngle(joint[j]);     // 関節角の取得
		dReal z = THETA[j] - tmp;                      // 残差
		if (z >=   M_PI) z -= 2.0 * M_PI;
		if (z <= - M_PI) z += 2.0 * M_PI;
		dJointSetHingeParam(joint[j],dParamVel, k*z);  // 角速度の設定
		dJointSetHingeParam(joint[j],dParamFMax,fMax); // トルクの設定
	}
}

/*** 視点と視線の設定 ***/
void start()
{
	float xyz[3] = {    3.0f, 1.3f, 0.8f};          // 視点[m]
	float hpr[3] = { -160.0f, 4.5f, 0.0f};          // 視線[°]
	dsSetViewpoint(xyz, hpr);                       // 視点と視線の設定
}

// 逆運動学
// 目標位置がアームの到達範囲以外のときは，逆運動学を計算できないので
// その処理を組み込む必要があります．
void  inverseKinematics()
{
	double Px, Py, Pz;

	Px = P[0], Py = P[1], Pz = P[2]; // アーム先端の目標座標P(Px,Py,Pz)
	printf("Target  Position: x=%6.2f y=%6.2f z=%6.2f \n", Px, Py, Pz);

	double tmpL  = sqrt(Px * Px + Py * Py);
	double P1P   = sqrt(Px * Px + Py * Py
		+ (Pz - (l[0] + l[1])) * (Pz - (l[0] + l[1])));
	double Ca    = (l[2] * l[2] + P1P * P1P -l[3] * l[3])/(2 * l[2] * P1P);  // cosα

	double phi   = atan2(Pz - (l[0] + l[1]), tmpL);                      //φ
	double alpha = atan2(sqrt(1 - Ca * Ca), Ca);                         //α

	double Cb    = (l[2]*l[2] + l[3]*l[3] - P1P*P1P)/(2 * l[2] * l[3]);  //cosβ
	double beta  = atan2(sqrt(1- Cb * Cb), Cb);                          //β

	switch (ANSWER) { // ANSWERはキーボードからの入力で変更
  case 1: // 姿勢１
	  THETA[1] = atan2(Py, Px);
	  THETA[2] = M_PI/2 - phi - alpha;
	  THETA[3] = M_PI - beta; break;
  case 2: // 姿勢２
	  THETA[1] = atan2(Py, Px);
	  THETA[2] = M_PI/2 - phi + alpha;
	  THETA[3] = M_PI + beta; break;
  case 3:  // 姿勢３  海老反り姿勢
	  THETA[1] = atan2(Py, Px) + M_PI;
	  THETA[2] = -(M_PI/2 - phi - alpha);
	  THETA[3] = M_PI + beta; break;
  case 4:  // 姿勢４  海老反り姿勢
	  THETA[1] = atan2(Py, Px) + M_PI;
	  THETA[2] = -(M_PI/2 - phi + alpha);
	  THETA[3] = M_PI - beta; break;
	}
}

/*** シミュレーションループ ***/
void simLoop(int pause)
{
	inverseKinematics();
	printSensorPosition();
	Pcontrol();                                  // P制御
	dWorldStep(world, 0.01);                     // 動力学計算
	drawArm();                                   // ロボットの描画
	drawP();                                     // 目標位置の描画
	drawSensor();                                // 先端位置の描画
}

/*** キー入力関数 ***/
void command2(int cmd)
{
	switch (cmd) {
		case '1':  ANSWER = 1; break;    // 1キーを押すと姿勢１
		case '2':  ANSWER = 2; break;    // 2キーを押すと姿勢２
		case '3':  ANSWER = 3; break;    // 3キーを押すと姿勢３
		case '4':  ANSWER = 4; break;    // 4キーを押すと姿勢４
		case 'j':  P[0] += 0.1; break;   // jキーを押すと先端のx座標が増加
		case 'f':  P[0] -= 0.1; break;   // fキーを押すと先端のx座標が減少
		case 'k':  P[1] += 0.1; break;   // kキーを押すと先端のy座標が増加
		case 'd':  P[1] -= 0.1; break;   // dキーを押すと先端のy座標が減少
		case 'l':  P[2] += 0.1; break;   // lキーを押すと先端のz座標が増加
		case 's':  P[2] -= 0.1; break;   // sキーを押すと先端のz座標が減少
	}
}

/*** ドロースタッフの設定 ***/
void setDrawStuff()
{
	fn.version = DS_VERSION;                     // バージョン番号
	fn.start   = &start;                         // start関数
	fn.step    = &simLoop;                       // simLoop関数
	fn.command = &command2;                       // command関数
	fn.path_to_textures = "../../drawstuff/textures";
}

int main(int argc, char *argv[])
{
	dInitODE(); //ＯＤＥの初期化
	setDrawStuff();
	world        = dWorldCreate();                  // ワールドの生成
	space        = dHashSpaceCreate(0);             // スペースの生成
	contactgroup = dJointGroupCreate(0);            // 接触グループの生成
	ground       = dCreatePlane(space, 0, 0, 1, 0); // 地面の生成
	dWorldSetGravity(world, 0, 0, -9.8);            // 重力の設定
	makeArm();                                      // アームの生成
	makeSensor();                                   // センサの生成
	dsSimulationLoop(argc, argv, 640, 480, &fn);    // シミュレーションループ
	dSpaceDestroy(space);                           // スペースの破壊
	dWorldDestroy(world);                           // ワールドの破壊
	dCloseODE(); //ＯＤＥの終了
	return 0;
}
