// A Petit Waru Robot Simulator
// Copyright (c) 2007-2010 Kosei Demura
// My web site is http://demura.net

// ちょいワル２足歩行ロボットプログラム
// 著作権者：出村公成
// 私のウェブサイトはhttp://demura.netです．

#include <stdio.h>
#include <stdlib.h>
#include <ode/ode.h>
// #include <drawstuff/drawstuff.h>

#include "irrdrawstuff.h"


#ifdef dDOUBLE
#define dsDrawCapsule  dsDrawCapsuleD
#define dsDrawCylinder dsDrawCylinderD
#define dsDrawBox      dsDrawBoxD
#define dsDrawLine     dsDrawLineD
#define dsDrawSphere   dsDrawSphereD
#endif

#define LINK_NUM 5  // 片脚のリンク数
#define JT_NUM   5  // 片脚のジョイント数 
#define LEG_NUM  2  // 脚数
#define GAIT_NUM 6 // 1歩容を何ステップで行うか

enum link_no {
	HIP_DUMMY,   // 0 腰関節取付用(2自由度直交のため必要）
	UPPER_LEG,   // 1 上脚
	LOWER_LEG,   // 2 下脚
	ANKLE_DUMMY, // 3 足関節取付用(2自由度直交のため必要）
	FOOT         // 4 足
};

enum joint_no {
	HIP_PITCH,   // 0 腰関節ピッチ
	HIP_ROLL,    // 1 腰関節ロール
	KNEE_PITCH,  // 2 膝関節ピッチ
	ANKLE_PITCH, // 3 足関節ピッチ
	ANKLE_ROLL   // 4 足関節ピッチ
};

static dWorldID      world;        // 動力学計算用ワールド
static dSpaceID      space;        // 衝突検出計算用スペース
static dGeomID       ground;       // 地面
static dJointGroupID contactgroup; // 接触点グループ
static dsFunctions   fn;           // 描画関数

// マイオブジェクト構造体
typedef struct {
  dBodyID  body;
  dGeomID  geom;
  dJointID joint;
  dReal    m,r,h,x,y,z; // 質量、半径、高さ、位置(x,y,z)
  dReal    lx,ly,lz;    // サイズ
} MyObject;

static MyObject leg[LEG_NUM][LINK_NUM],base,bullet,cannon,wall[200];

static dReal  THETA[LEG_NUM][LINK_NUM] = {{0},{0}}; // ステップ毎の目標関節
static dReal  SX = 0, SY = 0, SZ = 0.65;     // 初期位置(x,y,z)sz = 0.61
static dReal  gait[GAIT_NUM][LEG_NUM][JT_NUM] ;  // 歩行の目標関節角
// 足関節だけは足裏が地面と平行になるように制御しているのでgait配列を使わない

static dReal cannon_angle = 0.0, cannon_elevation = 0.0;

static int wb = 0;
static int WALKING_ON = 0;         

const dReal HIP_DUMMY_LENGTH   = 0.05;    // 腰関節取付用リンク
const dReal UPPER_LEG_LENGTH   = 0.3;     // 上脚の長さ
const dReal LOWER_LEG_LENGTH   = 0.3;     // 下脚の長さ
const dReal ANKLE_DUMMY_LENGTH = 0.00001; // 足関節取付用リンク 0.0001
const dReal FOOT_LENGTH        = 0.02;    // 足の厚さ
const dReal FOOT_RADIUS        = 0.15;    // 足の半径
const dReal LEG_RADIUS         = 0.02;    // 脚の半径
const dReal BODY_LENGTH        = 0.15;    // 胴体の長さ
const dReal BODY_WIDTH         = 0.25;    // 胴体の幅
const dReal BODY_HEIGHT        = 0.40;    // 胴体の高さ


// 砲弾の生成
static void makeBullet()
{
    dMass mass;

    bullet.r    = 0.01;
    bullet.m    = 0.1;
    bullet.body = dBodyCreate(world);
    bullet.geom = dCreateSphere(space,bullet.r);

    dMassSetSphereTotal(&mass,bullet.m,bullet.r);
    dBodySetMass(bullet.body,&mass);
    dGeomSetBody(bullet.geom,bullet.body);

    dVector3 anchor;
    dJointGetHingeAnchor(cannon.joint,anchor);
    dBodySetPosition(bullet.body,anchor[0],anchor[1],anchor[2]);
}

// 胴体の生成
static void makeTorso()
{
    dReal L1 = BODY_LENGTH, W1 = BODY_WIDTH, H1 = BODY_HEIGHT;
    dMass mass;
    dReal b_m = 10.0;    // mass
    dReal b_s[3] = {L1,W1/2.0,H1/3.5}; // sides
    dReal b_p[3] = {SX,SY,SZ}; // pos

    dMatrix3 R;
    dRFromAxisAndAngle(R,0,1,0,M_PI/2);

    base.m  = b_m;
    base.x  = b_p[0];
    base.y  = b_p[1];
    base.z  = b_p[2];
    base.lx = b_s[0];
    base.ly = b_s[1];
    base.lz = b_s[2];

    base.body  = dBodyCreate(world);
    dMassSetZero(&mass);
    dMassSetCapsuleTotal(&mass,base.m,3,base.ly, base.lz); // ly:半径、lz:長さ
    dBodySetMass(base.body, &mass);

    base.geom = dCreateCapsule(space,base.ly, base.lz);
    dGeomSetBody(base.geom, base.body);
    dBodySetPosition(base.body, base.x, base.y, base.z);
    dBodySetRotation(base.body,R);

    // キャノンの生成
    dRFromAxisAndAngle(R,0,1,0,M_PI/2);
    cannon.body = dBodyCreate(world);
    dBodySetRotation(cannon.body,R);
    dBodySetPosition(cannon.body, base.x + H1/2.0, base.y, base.z);
    dMassSetZero(&mass);
    dMassSetCylinderTotal(&mass,0.1,3,0.01, 0.1); // ly:半径、lz:長さ
    dBodySetMass(cannon.body, &mass);
    cannon.geom  = dCreateCylinder(space,0.01, 0.1);
    cannon.joint = dJointCreateHinge(world, 0);
    dJointAttach(cannon.joint, base.body, cannon.body);
    dJointSetHingeAnchor(cannon.joint, base.x + H1/2.0, base.y, base.z);
    dJointSetHingeAxis(cannon.joint, 0, 1, 0);
}

// ロボットの生成
void  makeRobot()
{
    dReal l0 = HIP_DUMMY_LENGTH;
    dReal l1 = UPPER_LEG_LENGTH;
    dReal l2 = LOWER_LEG_LENGTH;
    dReal l3 = ANKLE_DUMMY_LENGTH;
    dReal l4 = FOOT_LENGTH;
    dReal bw = BODY_WIDTH;  // 胴体の幅
    dReal lr = LEG_RADIUS;  // 脚の半径
    dReal fr = FOOT_RADIUS; // 足の半径
    // link0: 胴体から股関節, link1: 上脚, link2:下脚, link3:足関節ダミー, link4:足
    dReal l0m = 0.001,l1m = 0.5, l2m = 0.5, l3m = 0.001, l4m = 0.5; // リンクの質量

    dMass mass;

    // c_x,c_y,c_z: 関節中心の座標
    dReal c_x[LEG_NUM][LINK_NUM] = {{ 0, 0, 0, 0, 0},{ 0, 0,0,0,0}};
    dReal c_y[LEG_NUM][LINK_NUM] = {{ bw/2, (bw+l0)/2, (bw+l0)/2, (bw+l0)/2, (bw+l0)/2},
        {-bw/2,-(bw+l0)/2,-(bw+l0)/2,-(bw+l0)/2,-(bw+l0)/2}
    };
    dReal c_z[LEG_NUM][LINK_NUM] =  {{0,0,-l1,-l1-l2,-l1-l2-l3},
        {0,0,-l1,-l1-l2,-l1-l2-l3}
    };
    dReal axis_x[LEG_NUM][LINK_NUM] = {{ 0,1,0,0,1},{ 0,1,0,0,1}}; //回転軸ベクトルx座標
    dReal axis_y[LEG_NUM][LINK_NUM] = {{ 1,0,1,1,0},{ 1,0,1,1,0}}; //回転軸ベクトルy座標
    dReal axis_z[LEG_NUM][LINK_NUM] = {{ 0,0,0,0,0},{ 0,0,0,0,0}}; //回転軸ベクトルz座標

    // リンクの座標
    dReal x[LEG_NUM][LINK_NUM], y[LEG_NUM][LINK_NUM];
    dReal z[LEG_NUM][LINK_NUM] =
    {
        {c_z[0][0],c_z[0][1]-l1/2,c_z[0][2]-l2/2,c_z[0][3]-l3/2,c_z[0][4]-l4/2},
        {c_z[0][0],c_z[0][1]-l1/2,c_z[0][2]-l2/2,c_z[0][3]-l3/2,c_z[0][4]-l4/2}
    };

    dReal lx[LINK_NUM] = {2*lr, lr, lr, lr, 2.0*fr}; // 脚の奥行き
    dReal ly[LINK_NUM] = {2*lr, lr, lr, lr, fr};     // 脚の半径
    dReal lz[LINK_NUM] = {  l0, l1, l2, l3, l4};     // 脚の長さ(高さ）

    dReal weight[LINK_NUM]     =  {l0m,l1m,l2m, l3m,l4m}; // 脚の質量

    for (int i = 0; i < LEG_NUM; i++)
    {
        for (int j = 0; j < LINK_NUM; j++)
        {
            x[i][j] = c_x[i][j];
            if (j == 0)
            {
                y[i][j] = (c_y[i][0]+c_y[i][1])/2;
            }
            else
            {
                y[i][j] = c_y[i][j];
            }
        }
    }

    // 胴体の生成
    makeTorso();

    // 脚の生成
    dMatrix3 R;
    dRFromAxisAndAngle(R,1,0,0,M_PI/2);
    for (int i = 0; i < LEG_NUM; i++)
    {
        for (int j = 0; j < LINK_NUM-1; j++)
        {
            leg[i][j].body = dBodyCreate(world);
            if (j == 0) dBodySetRotation(leg[i][j].body,R);
            dBodySetPosition(leg[i][j].body, SX+x[i][j], SY+y[i][j], SZ+z[i][j]);
            dMassSetZero(&mass);
            dMassSetCylinderTotal(&mass,weight[j],3,ly[j],lz[j]); // ly:半径、lz:長さ
            dBodySetMass(leg[i][j].body, &mass);
            leg[i][j].geom = dCreateCylinder(space,ly[j],lz[j]);
            dGeomSetBody(leg[i][j].geom,leg[i][j].body);
        }
        int j = FOOT;
        leg[i][j].body = dBodyCreate(world);
        dBodySetPosition(leg[i][j].body, SX+x[i][j], SY+y[i][j], SZ+z[i][j]);
        dMassSetZero(&mass);
        dMassSetBoxTotal(&mass,weight[j],lx[j],ly[j],lz[j]); // ly:半径、lz:長さ
        dBodySetMass(leg[i][j].body, &mass);
        leg[i][j].geom = dCreateBox(space,lx[j],ly[j],lz[j]);
        dGeomSetBody(leg[i][j].geom,leg[i][j].body);
    }

    // ジョイントの生成と取り付け
    for (int i = 0; i < LEG_NUM; i++)
    {
        for (int j = 0; j < LINK_NUM; j++)
        {
            leg[i][j].joint = dJointCreateHinge(world, 0);
            if (j == 0)
                dJointAttach(leg[i][j].joint, base.body, leg[i][j].body);
            else
                dJointAttach(leg[i][j].joint, leg[i][j-1].body, leg[i][j].body);
            dJointSetHingeAnchor(leg[i][j].joint, SX+c_x[i][j], SY+c_y[i][j],SZ+c_z[i][j]);
            dJointSetHingeAxis(leg[i][j].joint, axis_x[i][j], axis_y[i][j],axis_z[i][j]);
        }
    }
}

// ジョイントの描画
void drawJoint(int i,int j, float length, float radius)
{
    dVector3 anchor,axis;
    dMatrix3 R,R1,R2;
    dJointGetHingeAnchor(leg[i][j].joint,anchor);
    dJointGetHingeAxis(leg[i][j].joint,axis);
    dRFromAxisAndAngle(R,axis[0],axis[1],axis[2],0.5 * M_PI);
    dRFromAxisAndAngle(R1,0,0,1,0.5 * M_PI);
    dMultiply0(R2,R1,R,3,3,3);
    dsDrawCylinder(anchor,R2,length,radius);
}

// 目の描画
void drawEye(int i,int j, float length, float radius)
{
    dVector3 axis;
    dMatrix3 R,R1,R2;
    dsSetColorAlpha(0, 0,1.3f,0.6f);
    dJointGetHingeAxis(leg[i][j].joint,axis);
    dRFromAxisAndAngle(R,axis[0],axis[1],axis[2],0.5 * M_PI);
    dRFromAxisAndAngle(R1,0,0,1,0.5 * M_PI);
    dMultiply0(R2,R1,R,3,3,3);
    dsDrawCapsule(dBodyGetPosition(base.body),
                  dBodyGetRotation(base.body),length,radius);
}

void drawJoint2(int i,int j, float radius)
{
    dVector3 anchor;
    dMatrix3 R;
    dRSetIdentity(R);
    dJointGetHingeAnchor(leg[i][j].joint,anchor);
    dsDrawSphere(anchor,R,radius);
}

void drawCannon()
{
    dReal r, length;
    // draw the cannon
    dsSetColor (1,1,0);
    dMatrix3 R1,R2,R3,Rtmp,R4;

    dRFromAxisAndAngle(R1,0,1,0,M_PI/2.0);
    dRFromAxisAndAngle(R2,0,0,1,cannon_angle);
    dRFromAxisAndAngle(R3,0,1,0,cannon_elevation);
    dMultiply0(Rtmp,R2,R1,3,3,3);
    dMultiply0(R4,R3,Rtmp,3,3,3);

    dVector3 anchor;
    dJointGetHingeAnchor(cannon.joint,anchor);
    dGeomCylinderGetParams(cannon.geom, &r,&length);
    dsDrawCylinder(anchor,R4, 0.1, 0.01);
}

void drawRobot()
{
    dReal r,length;
    dReal kr = 0.04; // knee radius
    dsSetTexture(DS_WOOD);

    // 目の描画
    dsSetColor(1.0,1.0,1.0);
    drawEye(1, LINK_NUM-2, 0.3, 0.062);

    // キャノンの描画
    drawCannon();

    // 胴体の描画
    dsSetColor(1.0,1.0,1.0);
    dGeomCapsuleGetParams(base.geom, &r,&length);
    dsDrawCapsule(dBodyGetPosition(base.body),
                  dBodyGetRotation(base.body),length,r);

    // 脚の描画
    for (int i = 0; i < LEG_NUM; i++)
    {
        for (int j = 0; j < LINK_NUM; j++ )
        {
            dsSetColor(1.0,1.0,1.0);
            if (j== 0)
            {
                dGeomCylinderGetParams(leg[i][j].geom, &r,&length);
                dsDrawSphere(dBodyGetPosition(leg[i][j].body),
                             dBodyGetRotation(leg[i][j].body),kr);
            }
            else
            {
                if (j != FOOT)
                {
                    dGeomCylinderGetParams(leg[i][j].geom, &r,&length);
                    dsDrawCylinder(dBodyGetPosition(leg[i][j].body),
                                   dBodyGetRotation(leg[i][j].body),length,r);
                }
                else
                {
                    dReal ss[3]={2.0*FOOT_RADIUS, FOOT_RADIUS, FOOT_LENGTH};
                    dsDrawBox(dBodyGetPosition(leg[i][j].body),
                              dBodyGetRotation(leg[i][j].body),ss);
                }
            }
        }

        // 膝関節の描画
        drawJoint2(0, KNEE_PITCH, kr);  // left leg
        drawJoint2(1, KNEE_PITCH, kr);  // right leg
    }

}

// スタート関数
void start()
{
    float xyz[3] = {  1.85, -1.94, 0.53};
    float hpr[3] = {  106.5, -3.5, 0.0};
    dsSetViewpoint(xyz,hpr);
    dsSetSphereQuality(3);
    dsSetCapsuleQuality(6);
}

// ロボットの破壊
static void destroyRobot()
{
    dBodyDestroy(base.body);
    dGeomDestroy(base.geom);

    for (int i = 0; i < LEG_NUM; i++)
    {
        for (int j = 0; j < LINK_NUM; j++)
        {
            dBodyDestroy(leg[i][j].body);
            dGeomDestroy(leg[i][j].geom);
        }
    }

    for (int i = 0; i < LEG_NUM; i++)
    {
        for (int j = 0; j < LINK_NUM; j++)
        {
            dJointDestroy(leg[i][j].joint);
        }
    }
}

// 再スタート
static void restart()
{
    destroyRobot();
    dJointGroupDestroy(contactgroup);
    contactgroup = dJointGroupCreate(0);
    makeRobot();
}

// f:前進, r:再スタート, １：砲身の上昇、２：砲身の下降、
// [:砲身の左旋回, ]:砲身の右旋回、 x:発射
void command(int cmd)
{
    switch (cmd)
    {
    case 'r':
        restart();
        break;
    case 'f':
        WALKING_ON = 1;
        break;
    case 's':
        WALKING_ON = 0;
        break;
    case 'q':
    {
        exit(1);
        break;
    }
    case '9':
        float xyz1[3],hpr1[3];
        dsGetViewpoint(xyz1,hpr1);
        printf("xyz=%4.2f %4.2f %4.2f ",xyz1[0],xyz1[1],xyz1[2]);
        printf("hpr=%6.2f %6.2f %5.2f \r",hpr1[0],hpr1[1],hpr1[2]);
        while (1) ;
        break;
    case '[':
        cannon_angle += 0.1;
        break;
    case ']':
        cannon_angle -= 0.1;
        break;
    case '1':
        cannon_elevation += 0.1;
        break;
    case '2':
        cannon_elevation -= 0.1;
        break;
    case 'x':
    {
        dMatrix3 R2,R3,R4;
        dRFromAxisAndAngle(R2,0,0,1,cannon_angle);
        dRFromAxisAndAngle(R3,0,1,0,cannon_elevation);
        dMultiply0 (R4,R2,R3,3,3,3);

        // 砲弾の発射
        dVector3 anchor;
        dJointGetHingeAnchor(cannon.joint,anchor);
        dReal cpos[3] = {anchor[0], anchor[1], anchor[2]};
        dBodySetPosition(bullet.body,cpos[0],cpos[1],cpos[2]);
        dReal force = 50;
        dBodySetLinearVel(bullet.body,force*R4[0],force*R4[4],force*R4[8]);
        printf("x=%f y=%f z=%f \n",R4[2], R4[6], R4[10]);
        dBodySetAngularVel(bullet.body,0,0,0);
        break;
    }
    }
}


// 衝突検出関数
static void nearCallback(void *data, dGeomID o1, dGeomID o2)
{
    dBodyID b1 = dGeomGetBody(o1);
    dBodyID b2 = dGeomGetBody(o2);
    if (b1 && b2 && dAreConnectedExcluding(b1,b2,dJointTypeContact)) return;

    bool cond1 = ((o1 == ground) || (o2 == ground));
    bool cond2 = ((dGeomGetClass(o1) == dBoxClass) && (dGeomGetClass(o2) == dSphereClass)) ||
                 ((dGeomGetClass(o2) == dBoxClass) && (dGeomGetClass(o1) == dSphereClass));
    bool cond3 = ((dGeomGetClass(o1) == dBoxClass) && (dGeomGetClass(o2) == dBoxClass));

    if ( cond1 || cond2)
    {
        static const int N = 10;
        dContact contact[N];
        int n = dCollide(o1,o2,N,&contact[0].geom,sizeof(dContact));
        if (n > 0)
        {
            for (int i=0; i<n; i++)
            {
                contact[i].surface.mode = dContactSoftERP | dContactSoftCFM;
                contact[i].surface.mu   = dInfinity; //dInfinity; //2.0;
                contact[i].surface.soft_erp = 0.2;
                contact[i].surface.soft_cfm =1e-4;//1e-5;
                dJointID c = dJointCreateContact(world,contactgroup,&contact[i]);
                dJointAttach(c,b1,b2);
            }
        }
    }
    else
    {
        if (cond3)
        {
            static const int N = 10;
            dContact contact[N];
            int n = dCollide(o1,o2,N,&contact[0].geom,sizeof(dContact));
            if (n > 0)
            {
                for (int i=0; i<n; i++)
                {
                    contact[i].surface.mu = dInfinity;
                    contact[i].surface.mode = dContactSoftERP | dContactSoftCFM;
                    contact[i].surface.soft_erp = 0.2;
                    contact[i].surface.soft_cfm = 1e-4;
                    dJointID c = dJointCreateContact (world,contactgroup,contact+i);
                    dJointAttach (c,dGeomGetBody(o1),dGeomGetBody(o2));
                }
            }
        }
    }
}


// 3関節の逆運動学
void  inverseKinematics(dReal x, dReal y, dReal z,
                        dReal *ang1, dReal *ang2, dReal *ang3,int posture)
{
    dReal l1 = HIP_DUMMY_LENGTH;
    dReal l2 = UPPER_LEG_LENGTH;
    dReal l3 = LOWER_LEG_LENGTH;

    dReal l1a = 0;                // 肩関節を中心としているので0
    dReal l3a = l3 + FOOT_LENGTH; // 足底の厚さを考慮

    if (fabs(l2 * l3a) < 1e-6)
    {
        printf("Error:l1 * l2a is zero \n");
        exit(1);
    }
    double c3 = (x*x + z*z + (y-l1a)*(y-l1a) - (l2*l2+l3a*l3a))/(2*l2*l3a);

    if (fabs(l2 + l3a*c3) < 1e-6)
    {
        printf("Error:l1+l2a*c2 is zero \n");
        exit(1);
    }
    double s2 = (y-l1a) / (l2 + l3a*c3);
    double c2 = sqrt(1 - s2 * s2);

    if (x*x + z*z  < 1e-6)
    {
        printf("Error x*x + z*z \n");
        exit(1);
    }
    double c1 = (l2 + l3a*c3)*c2/sqrt(x*x+z*z);

    switch (posture)
    {
    case 1: // elbow up
        *ang1 = atan2(x,-z) - atan2(sqrt(1 - c1*c1),c1);
        *ang2 = -atan2(s2,c2);
        *ang3 = atan2(sqrt(1-c3*c3),c3);
        break;
    case 2: // elbow down
        *ang1= atan2(x,-z) + atan2(sqrt(1 - c1*c1),c1);
        *ang2= -atan2(s2,c2);
        *ang3= -atan2(sqrt(1-c3*c3),c3);
        break;
    case 3:  // elbow Up
        *ang1 = M_PI +(atan2(x,-z) - atan2(sqrt(1 - c1*c1),c1));
        *ang2 = - M_PI + atan2(s2,c2);
        *ang3 = -atan2(sqrt(1-c3*c3),c3);
        break;
    case 4:  // elbow down
        *ang1 = M_PI + atan2(x,-z) + atan2(sqrt(1 - c1*c1),c1);
        *ang2 = -M_PI + atan2(s2,c2);
        *ang3 = atan2(sqrt(1-c3*c3),c3);
        break;
    }
}


// P制御
void Pcontrol()
{
    static dReal kp = 5.0, fMax = 400.0;

    for (int i = 0; i < LEG_NUM; i++)
    {
        for (int j = 0; j < LINK_NUM; j++)
        {
            dReal tmp = dJointGetHingeAngle(leg[i][j].joint);
            dReal diff = THETA[i][j] - tmp;
            if (diff >= M_PI)  diff -= 2.0 * M_PI;
            if (diff <= -M_PI) diff += 2.0 * M_PI;
            dReal u = kp * diff;
            if (u >   M_PI) u =   M_PI;
            if (u < - M_PI) u = - M_PI;
            dJointSetHingeParam(leg[i][j].joint,  dParamVel, u);
            dJointSetHingeParam(leg[i][j].joint, dParamFMax, fMax);
        }
    }
}

// 歩行
void walk()
{
    static int t = 0, steps = 0, counter=0;
    int interval = 50;
    int PREP_STEPS = 150; // 歩行するための準備期間
    int START_STEP = 150; // 歩行開始のステップ
    dReal g_tmp,g_next;

    if (steps < PREP_STEPS)
    {
        for (int leg_no = 0; leg_no < LEG_NUM; leg_no++)
        {
            for (int joint_no = 0; joint_no < JT_NUM; joint_no++)
            {
                THETA[leg_no][joint_no]
                = (dReal) steps * gait[0][leg_no][joint_no]/PREP_STEPS;
            }
        }
    }
    else
    {
        if (steps >= START_STEP )
        {
            if (steps == START_STEP) t = 0;
            for (int leg_no = 0; leg_no < LEG_NUM; leg_no++)
            {
                for (int joint_no = 0; joint_no < JT_NUM; joint_no++)
                {
                    g_tmp  = gait[t%GAIT_NUM][leg_no][joint_no];
                    if (GAIT_NUM != (t%GAIT_NUM) + 1)
                        g_next = gait[(t%GAIT_NUM)+1][leg_no][joint_no];
                    else
                        g_next = gait[0][leg_no][joint_no];
                    THETA[leg_no][joint_no]
                    = g_tmp + (g_next - g_tmp) * counter / interval;
                }
            }
            counter++;
        }
        if ((steps % interval)==0)
        {
            t++;
            counter = 0;
        }
    }
    Pcontrol();
    steps++;
}

// シミュレーションループ
void simLoop(int pause)
{
    if (!pause)
    {
        for (int i = 0; i < 5; i++)
        {
            if (WALKING_ON)
            {
                walk();
            }
            Pcontrol();
            dSpaceCollide(space,0,&nearCallback);
            dWorldStep(world, 0.01);
            dJointGroupEmpty(contactgroup);
        }
    }
    drawRobot();
    dsSetTexture(DS_NONE);
    dsSetColor(1.0,1.0,1.0);
    dsDrawSphere(dBodyGetPosition(bullet.body),
                 dBodyGetRotation(bullet.body),1 * bullet.r);

}

// 描画関数の設定
void setDrawStuff()
{
    fn.version = DS_VERSION;
    fn.start   = &start;
    fn.step    = &simLoop;
    fn.command = &command;
    fn.path_to_textures = "../../irrdrawstuff/demo/textures/warusap";
}

int main(int argc, char **argv)
{
    dInitODE();
	setDrawStuff();    // 描画関数の設定
  
    world        = dWorldCreate();              // ワールドの生成
    space        = dHashSpaceCreate(0);         // スペースの生成
    contactgroup = dJointGroupCreate(0);        // 接触点グループの生成
    ground       = dCreatePlane(space,0,0,1,0); // 地面の生成
    dWorldSetGravity(world, 0, 0, -9.8);        // 重力加速度の設定
    dWorldSetERP(world,0.95);  // ERPの設定 0.95
    dWorldSetCFM(world,1e-5); // CFMの設定 1e-5
    dWorldSetContactSurfaceLayer(world, 0.001);

    makeRobot();   // ロボットの生成
    makeBullet();  // 砲弾の生成

    //dReal z0 = -0.4,z1 = -0.37;
    dReal z0 = -0.45;   // 足の着地点のz座標
    dReal z1 = -0.42;   // 足の最高点のz座標
    dReal y1 =  0.07;   // 重心の左右への移動量 4legged 0.05
    dReal fs =  0.15;   // footstep 歩幅 4legged 0.1

    dReal dy = -0.00;  // 脚を内側への移動量


    dReal  traj[GAIT_NUM][LEG_NUM][3] =   // trajectory 目標軌道
    {
        {{     0,  y1+dy, z0},{     0,  y1-dy, z0}},  //  0 初期姿勢
        {{  fs/2,  y1+dy, z1},{     0,  y1-dy, z0}},  //  1 左脚を上げ前方へ移動
        {{    fs,  y1+dy, z0},{     0,  y1-dy, z0}},  //  2 左脚の着地
        {{  fs/2,    +dy, z0},{ -fs/2,    -dy, z0}},  //  3 重心を中央へ移動
        {{     0, -y1+dy, z0},{   -fs, -y1-dy, z0}},  //  4 胴体を前方へ移動
        {{     0, -y1+dy, z0},{ -fs/2, -y1-dy, z1}},  //  5 右脚を上げ前方へ移動
    };

	// 逆運動学を使い歩行軌道から関節角度を生成
	// 歩行の関節角時系列データはgait配列に格納される
    dReal angle0, angle1, angle2;
    int   posture = 2;
    for (int i = 0; i < GAIT_NUM; i++)
    {
        for (int k = 0; k < LEG_NUM; k++)
        {
            inverseKinematics(traj[i][k][0], traj[i][k][1],traj[i][k][2],&angle0, &angle1, &angle2,posture);
            gait[i][k][HIP_PITCH]   =   angle0;
            gait[i][k][HIP_ROLL]    =   angle1;
            gait[i][k][KNEE_PITCH]  =   angle2;
            // 足関節
            gait[i][k][ANKLE_PITCH] = - (angle0 + angle2);
            gait[i][k][ANKLE_ROLL]  =  - angle1;
            printf(" angle0 = %f \n",angle0 * 180.0/M_PI);
            printf(" angle1 = %f \n",angle1 * 180.0/M_PI);
            printf(" angle2 = %f \n",angle2 * 180.0/M_PI);
            printf(" angle3 = %f \n", - (angle0 + angle2) * 180.0/M_PI);
            //exit(1);
        }
    }

	// シミュレーションループ
    dsSimulationLoop(argc,argv,640, 400,&fn);
    dSpaceDestroy(space); // スペースの破壊
    dWorldDestroy(world); // ワールドの破壊
    dCloseODE();
    return 0;
}
