// Copyright (C) 2010 Kosei Demura
// This file is part of the irrDrawStuff library.
// Please read copyright notice in irrdrawstuff.h for conditions of distribution and usage.

#include "vision.h"
#include "misc.h"
#include "drawPrimitives.h"

namespace {
	SimLoop *sim = SimLoop::getInstance();
}

Ground::Ground()
{
	red   = 0.5f;           // ground color for when there's no texture
	green = 0.5f;
	blue  = 0.3f;
	scale  = 1.0f/1.0f;	    // ground texture scale (1/size)
	offsetX = 0.5f;		    // offset of ground texture (x axis)
	offsetY = 0.5f;         // offset of ground texture (y axis)
}

Ground::~Ground()
{
}


DrawPrimitives::DrawPrimitives(irr::video::IVideoDriver *driver, irr::scene::ISceneManager *smgr,
							   irr::gui::IGUIEnvironment *env)
{
	ground = new Ground;

	ismgr = smgr;
	ienv = env;

	red   = 1.0f;           // ground color for when there's no texture
	green = 1.0f;
	blue  = 1.0f;
	alpha = 1.0f;
	sphere_quality = 1;          // 球の品質
	capped_cylinder_quality = 3; // カプセルの品質
	shadowIntensity = 0.65f;     // 影の濃さ
	last_draw = NONE;            // 最後に描画された基本図形

	tnum = 0; // テクスチャの番号

	idriver = driver;
}

DrawPrimitives::~DrawPrimitives()
{
	safe_delete(ground);
}

void DrawPrimitives::resetLastDraw()
{
	last_draw = NONE;
}

irr::gui::IGUIEnvironment* DrawPrimitives::getEnv()
{
	return ienv;
}


void DrawPrimitives::setTextureNum(int a)
{
	tnum = a;
}

void DrawPrimitives::setTexturePath(char *p)
{
	iprefix = p;

	char *s1 = new char[100];
	char *s2 = new char[100];
	char *s3 = new char[100];
	char *s4 = new char[100];

	strcpy(s1, iprefix);
	strcat(s1,"/sky.ppm");
	sky_texture = idriver->getTexture(s1);

	strcpy(s2, iprefix);
	strcat(s2,"/ground.ppm");
	ground_texture = idriver->getTexture(s2);

	strcpy(s3,iprefix);
    strcat(s3,"/wood.ppm");
    wood_texture = idriver->getTexture(s3);

 	strcpy(s4,iprefix);
    strcat(s4,"/checkered.ppm");
    checkered_texture = idriver->getTexture(s4);

	safe_delete_array(s1);
	safe_delete_array(s2);
	safe_delete_array(s3);
	safe_delete_array(s4);
}

float DrawPrimitives::getRed()
{
	return red;
}

float DrawPrimitives::getGreen()
{
	return green;
}

float DrawPrimitives::getBlue()
{
	return blue;
}

float DrawPrimitives::getAlpha()
{
	return alpha;
}


void DrawPrimitives::setColor(float r, float g, float b, float alp)
{
	red   = r;
	green = g;
	blue  = b;
	alpha = alp;

	if (r > 1.0)   red = 1.0f;
    if (r < 0.0)   red = 0.0f;
    if (g > 1.0)   green = 1.0f;
    if (g < 0.0)   green = 0.0f;
    if (b > 1.0)   blue = 1.0f;
    if (b < 0.0)   blue = 0.0f;
    if (alp > 1.0) alpha = 1.0f;
    if (alp < 0.0) alpha = 0.0f;

    color = SColor((int) (alpha * 255.0f), (int) (red  * 255.0f), (int) (green * 255.0f),
		(int) (blue * 255.0f));
}

// sets the material color, not the light color
void DrawPrimitives::setColor(SMaterial *material)
{
    int ambient[4],diffuse[4],specular[4],raw[4];
    //float k1 = 0.3f, k2 = 0.7f, k3 = 0.2f;
    float k1 = 0.8f, k2 = 0.8f, k3 = 0.2f;

	float r = red;
	float g = green;
	float b = blue;

    raw[0]  = (int) (r*255);
    raw[1]  = (int) (g*255);
    raw[2]  = (int) (b*255);
    raw[3]  = (int) (alpha*255);

    ambient[0]  = (int) (r*k1*255);
    ambient[1]  = (int) (g*k1*255);
    ambient[2]  = (int) (b*k1*255);
    ambient[3]  = (int) (alpha*255);
    diffuse[0]  = (int) (r*k2*255);
    diffuse[1]  = (int) (g*k2*255);
    diffuse[2]  = (int) (b*k2*255);
    diffuse[3]  = (int) (alpha*255);
    specular[0] = (int) (r*k3*255);
    specular[1] = (int) (g*k3*255);
    specular[2] = (int) (b*k3*255);
	specular[3] = (int) (alpha*255);

    // 環境光
    material->AmbientColor.set(ambient[3],ambient[0],ambient[1],ambient[2]);
    // 拡散光
    material->DiffuseColor.set(diffuse[3],diffuse[0],diffuse[1],diffuse[2]);
    // 鏡面光
    material->SpecularColor.set(specular[3],specular[0],specular[1],specular[2]);

    material->Shininess = 20.0f;  // 0 to 128, 20 is common
}


// sets the material color, not the light color
void DrawPrimitives::setColor(ISceneNode *node, float r, float g, float b, float alpha)
{
    int ambient[4],diffuse[4],specular[4],raw[4];
    //float k1 = 0.3f, k2 = 0.7f, k3 = 0.2f;
    float k1 = 0.8f, k2 = 0.8f, k3 = 0.2f;

    raw[0]  = (int) (r*255);
    raw[1]  = (int) (g*255);
    raw[2]  = (int) (b*255);
    raw[3]  = (int) (alpha*255);

    ambient[0]  = (int) (r*k1*255);
    ambient[1]  = (int) (g*k1*255);
    ambient[2]  = (int) (b*k1*255);
    ambient[3]  = (int) (alpha*255);
    diffuse[0]  = (int) (r*k2*255);
    diffuse[1]  = (int) (g*k2*255);
    diffuse[2]  = (int) (b*k2*255);
    diffuse[3]  = (int) (alpha*255);
    specular[0] = (int) (r*k3*255);
    specular[1] = (int) (g*k3*255);
    specular[2] = (int) (b*k3*255);
	specular[3] = (int) (alpha*255);

    // 環境光
    node->getMaterial(0).AmbientColor.set(ambient[3],ambient[0],ambient[1],ambient[2]);
    // 拡散光
    node->getMaterial(0).DiffuseColor.set(diffuse[3],diffuse[0],diffuse[1],diffuse[2]);
    // 鏡面光
    node->getMaterial(0).SpecularColor.set(specular[3],specular[0],specular[1],specular[2]);

    node->getMaterial(0).Shininess = 20.0f;  // 0 to 128, 20 is common
}

// Create and set a texture file: テクスチャーファイルの生成と設定
// 今は使っていない
void DrawPrimitives::setTexture(Primitive pri_num, SMaterial *material)
{
	char *s  = new char[strlen(iprefix)+20];
	char *s2 = new char[10];

	strcpy(s, iprefix);

	switch (pri_num)
	{
	case BOX: strcat(s,"/box"); break;
	case SPHERE: strcat(s,"/sphere"); break;
	case CAPSULE: strcat(s,"/capsule"); break;
	case CYLINDER: strcat(s,"/cylinder"); break;
	}
	int num = 1;
	sprintf(s2,"%d",num);
	strcat(s2,".bmp");
	strcat(s,s2);

	sim->misc->makeTexture(s, red, green, blue);
	material->TextureLayer[0].Texture = idriver->getTexture(s);
}


void DrawPrimitives::setTexture(Primitive pri_num, int num,
									scene::ISceneNode *node, video::ITexture *nodeTexture)
{
	char *s  = new char[strlen(iprefix)+20];
	char *s2 = new char[10];

	strcpy(s, iprefix);

	switch (pri_num)
	{
	case BOX: strcat(s,"/box"); break;
	case SPHERE: strcat(s,"/sphere"); break;
	case CAPSULE: strcat(s,"/capsule"); break;
	case CYLINDER: strcat(s,"/cylinder"); break;
	}
	sprintf(s2,"%d",num);
	strcat(s2,".bmp");
	strcat(s,s2);

	sim->misc->makeTexture(s, red, green, blue);
	nodeTexture = idriver->getTexture(s);

	safe_delete_array(s);
	safe_delete_array(s2);

	node->setMaterialTexture(0, nodeTexture);
}

void DrawPrimitives::drawBox(const double pos[3], const double R[12],const double sides[3])
{
	static int num = 0; // Count number of this function per a frame; この関数が1描画フレームに何回呼ばれたか数える
    static int NUM = 0; // Total number of boxes; 全ボックス数
    static double last_frame = 0; // Number of the last frame; 最近のフレーム数

    // FRAME_COUNT: 現在のフレーム数

    if (sim->getFrameCount() != last_frame)
    {
        // Set zero when a new frame;
        // 現在のフレーム数が，このAPIを呼び出されたときのフレーム数と違う時
        // つまり，新しいフレームになったとき
        num = 0;
    }

    if (num >= NUM)
    {
        // Create a new node when number of calls is more than number of boxes;
        // 箱の数よりも，呼び出し回数が多いときは新しいノードを生成する
        boxNode[num] = ismgr->addCubeSceneNode(1.0f, 0);
        //BoxNode[num]->setAutomaticCulling(scene::EAC_OFF);
        boxNode[num]->setAutomaticCulling(scene::EAC_BOX);
        boxNode[num]->setMaterialType(video::EMT_SOLID);
        boxNode[num]->setScale(vector3df(sides[1],sides[2],sides[0]));
        sim->misc->updateForDraw(boxNode[num],pos, R);

        if (tnum == DS_WOOD)  // if texture number equals DS_WOODS
        {
            boxNode[num]->setMaterialTexture(0,wood_texture);
            boxNode[num]->setMaterialFlag(video::EMF_LIGHTING, false);
        }
        else
        {
            setTexture(BOX, num, boxNode[num], boxTexture[num]);
            setColor(boxNode[num], red, green, blue, alpha);
            boxNode[num]->setMaterialFlag(video::EMF_LIGHTING, true); // true
        }
        //BoxNode[num]->setMaterialFlag(video::EMF_NORMALIZE_NORMALS, true);
        boxNode[num]->setMaterialFlag(video::EMF_NORMALIZE_NORMALS, false);

        NUM++;
    }
    else
    {
        // draw a box which is not a new node
        // 新しいノードを作らない場合の描画処理
        sim->misc->updateForDraw(boxNode[num],pos, R);
    }

	if (sim->getUseShadows())
	{
		//double new_pos[3];
		vector3df new_Rot;

		//dsToiDS(pos, R, new_pos, &new_Rot);
		matrix4 matrix;

		drawBoxShadow(pos,R,sides);
	}
    last_frame = sim->getFrameCount();
    num++;
}

void DrawPrimitives::drawSphere(const double pos[3], const double R[12], double radius)
{
	static int num = 0; // この関数が1描画フレームに何回呼ばれたか数える
    static int NUM = 0; // 全球数
    static double last_frame = 0; // 最近のフレーム数

    // FRAME_COUNT: 現在のフレーム数
    if (sim->getFrameCount() != last_frame)
    {
        // 現在のフレーム数が，このAPIを呼び出されたときのフレーム数と違う時
        // つまり，新しいフレームになったとき
        num = 0;
    }

    if (num >= NUM)   // 球の数よりも，呼び出し回数が多いときは新しい球ノードを生成する
    {
        // addSphereSceneNode(radius, polyCount, parent);  // org
	    sphereNode[num] = ismgr->addSphereSceneNode(radius,16,NULL); // good
    	sphereNode[num]->setMaterialType(video::EMT_SOLID);

		sim->misc->updateForDraw(sphereNode[num],pos, R);
        if (tnum == DS_WOOD)  // if texture number equals DS_WOODS
        {
            sphereNode[num]->setMaterialTexture(0,wood_texture);
            sphereNode[num]->setMaterialFlag(video::EMF_LIGHTING, false);
        }
        else
        {
           	setTexture(SPHERE, num, sphereNode[num], sphereTexture[num]);
            setColor(sphereNode[num], red, green, blue, alpha);
            sphereNode[num]->setMaterialFlag(video::EMF_LIGHTING,true);// true
        }

        NUM++;
    }
    else
    {
        // 球の数が前のフレームと同じならそのまま球を描く
        sim->misc->updateForDraw(sphereNode[num],pos, R);
    }

	if (sim->getUseShadows())
	{
		drawSphereShadow(pos[0], pos[1], pos[2], radius);
	}

    last_frame = sim->getFrameCount();
    num++;
}

void DrawPrimitives::drawSphere2(const double pos[3], const double R[12], double radius)
{
	if (sim->getUseShadows())
	{
		drawSphereShadow(pos[0], pos[1], pos[2], radius);
	}

	IMesh* smesh = ismgr->getGeometryCreator()->createSphereMesh(radius,16,16);
	sim->misc->odeToIrrTransform(pos, R);

	SMaterial material;
	//material.BackfaceCulling = true;  // カリング設定
	//material.Lighting        = true;  // ライト反映
  	material.ColorMaterial = ECM_NONE;

    if (tnum == DS_WOOD)  // if texture number equals DS_WOODS
    {
        material.TextureLayer[0].Texture = wood_texture;
	}
	else {
		setColor(&material);
	}

	idriver->setMaterial(material);
	idriver->drawMeshBuffer(smesh->getMeshBuffer(smesh->getMeshBufferCount()-1));

	last_draw = SPHERE;
}


void DrawPrimitives::drawBox2(const double pos[3], const double R[12], const double sides[3])
{
	static int count = 0;

	IMesh* smesh = ismgr->getGeometryCreator()->createCubeMesh(
		core::vector3df(sides[1], sides[2], sides[0]));
	sim->misc->odeToIrrTransform(pos, R);

	SMaterial material;
	//material.BackfaceCulling = true;  // カリング設定
	//material.Lighting        = true;  // ライト反映
  	material.ColorMaterial = ECM_NONE;

    if (tnum == DS_WOOD)  // if texture number equals DS_WOODS
    {
        material.TextureLayer[0].Texture = wood_texture;
	}
	else {
		setColor(&material);
	}

	idriver->setMaterial(material);
	idriver->drawMeshBuffer(smesh->getMeshBuffer(smesh->getMeshBufferCount()-1));

	if (sim->getUseShadows())
	{
		// drawSphereShadow(pos[0], pos[1], pos[2], radius);
	}

	last_draw = BOX;
}

/* A cylinder with proper normals and texture coords from CGeometryCreator.cpp in Irrlicht1.7 */
// IrrlichtのcreateCylinderMeshの位置はシリンダの下端．ODEの位置はシリンダの重心なので
// 高さ方向を直径の半分だけ下げている．後はirrlichtのコードと同じ
IMesh* DrawPrimitives::createCylinderMesh2(f32 radius, f32 length, u32 tesselation, const video::SColor& color,
			bool closeTop, f32 oblique) const
{
	SMeshBuffer* buffer = new SMeshBuffer();

	const f32 recTesselation = core::reciprocal((f32)tesselation);
	const f32 recTesselationHalf = recTesselation * 0.5f;
	//const f32 angleStep = (core::PI * 2.f ) * recTesselation;  // original
	const f32 angleStep = (M_PI * 2.f ) * recTesselation;
	const f32 angleStepHalf = angleStep*0.5f;

	u32 i;
	video::S3DVertex v;
	v.Color = color;
	buffer->Vertices.reallocate(tesselation*4+4+(closeTop?2:1));
	buffer->Indices.reallocate((tesselation*2+1)*(closeTop?12:9));
	f32 tcx = 0.f;
	for ( i = 0; i <= tesselation; ++i )
	{
		const f32 angle = angleStep * i;
		v.Pos.X = radius * cosf(angle);
		// v.Pos.Y = 0.f;
		v.Pos.Y = - 0.5 * length;
		v.Pos.Z = radius * sinf(angle);
		v.Normal = v.Pos;
		v.Normal.normalize();
		v.TCoords.X=tcx;
		v.TCoords.Y=0.f;
		buffer->Vertices.push_back(v);

		v.Pos.X += oblique;
		// v.Pos.Y = length;
		v.Pos.Y = 0.5 * length;
		v.Normal = v.Pos;
		v.Normal.normalize();
		v.TCoords.Y=1.f;
		buffer->Vertices.push_back(v);

		v.Pos.X = radius * cosf(angle + angleStepHalf);
		// v.Pos.Y = 0.f;
		v.Pos.Y = -0.5 * length;
		v.Pos.Z = radius * sinf(angle + angleStepHalf);
		v.Normal = v.Pos;
		v.Normal.normalize();
		v.TCoords.X=tcx+recTesselationHalf;
		v.TCoords.Y=0.f;
		buffer->Vertices.push_back(v);

		v.Pos.X += oblique;
		// v.Pos.Y = length;
		v.Pos.Y = 0.5 * length;
		v.Normal = v.Pos;
		v.Normal.normalize();
		v.TCoords.Y=1.f;
		buffer->Vertices.push_back(v);
		tcx += recTesselation;
	}

	// indices for the main hull part
	const u32 nonWrappedSize = tesselation* 4;
	for (i=0; i != nonWrappedSize; i += 2)
	{
		buffer->Indices.push_back(i + 2);
		buffer->Indices.push_back(i + 0);
		buffer->Indices.push_back(i + 1);

		buffer->Indices.push_back(i + 2);
		buffer->Indices.push_back(i + 1);
		buffer->Indices.push_back(i + 3);
	}

	// two closing quads between end and start
	buffer->Indices.push_back(0);
	buffer->Indices.push_back(i + 0);
	buffer->Indices.push_back(i + 1);

	buffer->Indices.push_back(0);
	buffer->Indices.push_back(i + 1);
	buffer->Indices.push_back(1);

	// close down
	v.Pos.X = 0.f;
	// v.Pos.Y = 0.f;
	v.Pos.Y = -0.5 * length;
	v.Pos.Z = 0.f;
	v.Normal.X = 0.f;
	v.Normal.Y = -1.f;
	v.Normal.Z = 0.f;
	v.TCoords.X = 1.f;
	v.TCoords.Y = 1.f;
	buffer->Vertices.push_back(v);

	u32 index = buffer->Vertices.size() - 1;

	for ( i = 0; i != nonWrappedSize; i += 2 )
	{
		buffer->Indices.push_back(index);
		buffer->Indices.push_back(i + 0);
		buffer->Indices.push_back(i + 2);
	}

	buffer->Indices.push_back(index);
	buffer->Indices.push_back(i + 0);
	buffer->Indices.push_back(0);

	if (closeTop)
	{
		// close top
		v.Pos.X = oblique;
		// v.Pos.Y = length;
		v.Pos.Y = 0.5 * length;
		v.Pos.Z = 0.f;
		v.Normal.X = 0.f;
		v.Normal.Y = 1.f;
		v.Normal.Z = 0.f;
		v.TCoords.X = 0.f;
		v.TCoords.Y = 0.f;
		buffer->Vertices.push_back(v);

		index = buffer->Vertices.size() - 1;

		for ( i = 0; i != nonWrappedSize; i += 2 )
		{
			buffer->Indices.push_back(i + 1);
			buffer->Indices.push_back(index);
			buffer->Indices.push_back(i + 3);
		}

		buffer->Indices.push_back(i + 1);
		buffer->Indices.push_back(index);
		buffer->Indices.push_back(1);
	}

	buffer->recalculateBoundingBox();
	SMesh* mesh = new SMesh();
	mesh->addMeshBuffer(buffer);
	mesh->setHardwareMappingHint(EHM_STATIC);
	mesh->recalculateBoundingBox();
	buffer->drop();
	return mesh;
}


void DrawPrimitives::drawCylinder2(const double pos[3], const double R[12],
								   float length, float radius)
{
	if (sim->getUseShadows())
	{
		// drawCylinderShadow(const double pos[3], const double R[12],  float length, float radius);
	}

	// Create a cylinder (半径，長さ，分割数，色，トップの閉塞)
	//IMesh* smesh = ismgr->getGeometryCreator()->createCylinderMesh(f32(radius), f32(length),
	//	u32(16));
	IMesh* smesh = createCylinderMesh2(f32(radius), f32(length), u32(16), 0xFFFFFFFF, true, 0.0f);

	// irrlichtのcreateCylinderMeshの位置はcylinderの下端，ODEの位置はcylinderの重心なので
	// 直径の
	sim->misc->odeToIrrTransform(pos, R);

	SMaterial material;
	//material.BackfaceCulling = true;  // カリング設定
	//material.Lighting        = true;  // ライト反映
  	material.ColorMaterial = ECM_NONE;

    if (tnum == DS_WOOD)  // if texture number equals DS_WOODS
    {
        material.TextureLayer[0].Texture = wood_texture;
	}
	else {
		setColor(&material);
	}

	idriver->setMaterial(material);
	idriver->drawMeshBuffer(smesh->getMeshBuffer(smesh->getMeshBufferCount()-1));

	last_draw = CYLINDER;
}

void DrawPrimitives::drawCapsuleMesh(float length, float radius)
{
	int i,j;
	float tmp,nx,ny,nz,start_nx,start_ny,a,ca,sa;
	// number of sides to the cylinder (divisible by 4):
	int capped_cylinder_quality = 3;
	const int n = capped_cylinder_quality*4;

	float l = length;
	float r = radius;

	// l = 1; r = 1;

	l *= 0.5;
	a = float(M_PI*2.0)/float(n);
	sa = (float) sin(a);
	ca = (float) cos(a);

	// draw cylinder body
	ny=1;
	nz=0;		  // normal vector = (0,ny,nz)

	SColor capsule_color = SColor((int) (getAlpha() * 255), (int) (getRed() * 255),
		(int) (getGreen() * 255), (int) (getBlue() * 255));

	for (i=0; i<=n; i++)
	{
		//四角形作成
		u16 rectList[] = {0,2,1,1,2,3};
		S3DVertex rectVer[4];

		rectVer[0] = S3DVertex( nz*r, l,-ny*r, nz,0,-ny,capsule_color, 0,0);// 上
		rectVer[1] = S3DVertex( nz*r,-l,-ny*r, nz,0,-ny,capsule_color, 0,0);// 右下

		// rotate ny,nz
		tmp = ca*ny - sa*nz;
		nz = sa*ny + ca*nz;
		ny = tmp;

		rectVer[2] = S3DVertex( nz*r, l, -ny*r, nz,0, -ny,capsule_color, 0,0);// 上
		rectVer[3] = S3DVertex( nz*r,-l, -ny*r, nz,0, -ny,capsule_color, 0,0);// 右下

		//idriver->setTransform(video::ETS_WORLD, AbsoluteTransformation);
		idriver->drawIndexedTriangleList(rectVer, 4, rectList, 2);
	}

	// draw first cylinder cap
	start_nx = 0;
	start_ny = 1;

	u16 triList1[] = {0,2,1},triList2[] = {0,2,1};

	S3DVertex triVer1[3],triVer2[3];

	for (j=0; j<(n/4); j++)
	{
		// get start_n2 = rotated start_n
		float start_nx2 =  ca*start_nx + sa*start_ny;
		float start_ny2 = -sa*start_nx + ca*start_ny;
		// get n=start_n and n2=start_n2
		nx = start_nx;
		ny = start_ny;
		nz = 0;
		float nx2 = start_nx2, ny2 = start_ny2, nz2 = 0;
		//glBegin (GL_TRIANGLE_STRIP);
		for (i=0; i<=n; i++)
		{
			triVer1[0] = S3DVertex(nz2*r, l+nx2*r, -ny2*r, nz2,nx2,-ny2, capsule_color, 0,0);// 上
			triVer1[1] = S3DVertex(nz *r, l+nx *r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上
			triVer2[0] = S3DVertex(nz *r, l+nx *r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上

			tmp = ca*ny2- sa*nz2;
			nz2 = sa*ny2 + ca*nz2;
			ny2 = tmp;
			tmp = ca*ny - sa*nz;
			nz  = sa*ny + ca*nz;
			ny  = tmp;

			triVer1[2] = S3DVertex(nz2*r, l+nx2*r, -ny2*r,  nz2, nx2,-ny2,  capsule_color, 0,0);// 上
			triVer2[1] = S3DVertex(nz *r, l+nx*r,  -ny *r,   nz,  nx, -ny,  capsule_color, 0,0);// 上
			triVer2[2] = S3DVertex(nz2*r, l+nx2*r, -ny2*r,  nz2, nx2,-ny2,  capsule_color, 0,0);// 上

			//idriver->setTransform(video::ETS_WORLD, AbsoluteTransformation);
			idriver->drawIndexedTriangleList(triVer1, 3, triList1, 1);

			idriver->drawIndexedTriangleList(triVer2, 3, triList2, 1);
		}

		start_nx = start_nx2;
		start_ny = start_ny2;
	}

	// draw second cylinder cap
	start_nx = 0;
	start_ny = 1;

	u16 triList3[] = {0,1,2},triList4[] = {0,1,2};

	S3DVertex triVer3[3],triVer4[3];

	for (j=0; j<(n/4); j++)
	{
		// get start_n2 = rotated start_n
		float start_nx2 =  ca*start_nx - sa*start_ny;
		float start_ny2 =  sa*start_nx + ca*start_ny;
		// get n=start_n and n2=start_n2
		nx = start_nx;
		ny = start_ny;
		nz = 0;
		float nx2 = start_nx2, ny2 = start_ny2, nz2 = 0;
		//glBegin (GL_TRIANGLE_STRIP);
		for (i=0; i<=n; i++)
		{
			triVer3[0] = S3DVertex(nz2*r, -l+nx2*r,-ny2*r, nz2,nx2, -ny2, capsule_color, 0,0);// 上
			triVer3[1] = S3DVertex(nz *r, -l+nx*r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上
			triVer4[0] = S3DVertex(nz *r, -l+nx*r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上

			tmp = ca*ny2- sa*nz2;
			nz2 = sa*ny2 + ca*nz2;
			ny2 = tmp;
			tmp = ca*ny - sa*nz;
			nz = sa*ny + ca*nz;
			ny = tmp;

			triVer3[2] = S3DVertex(nz2*r, -l+nx2*r,-ny2*r,  nz2, nx2, -ny2, capsule_color, 0,0);// 上
			triVer4[1] = S3DVertex(nz *r, -l+nx *r,-ny *r,   nz,  nx,  -ny, capsule_color, 0,0);// 上
			triVer4[2] = S3DVertex(nz2*r, -l+nx2*r,-ny2*r,  nz2, nx2, -ny2, capsule_color, 0,0);// 上


			//idriver->setTransform(video::ETS_WORLD, AbsoluteTransformation);
			idriver->drawIndexedTriangleList(triVer3, 3, triList3, 1);
			idriver->drawIndexedTriangleList(triVer4, 3, triList4, 1);
		}
		//glEnd();
		start_nx = start_nx2;
		start_ny = start_ny2;
	}
}

void DrawPrimitives::drawCapsule2(const double pos[3], const double R[12],  float length, float radius)
{
	static int count = 0;

	sim->misc->odeToIrrTransform(pos, R);

	SMaterial material;
	//material.BackfaceCulling = true;  // カリング設定
	//material.Lighting        = true;  // ライト反映
  	material.ColorMaterial = ECM_NONE;

    if (tnum == DS_WOOD)  // if texture number equals DS_WOODS
    {
        material.TextureLayer[0].Texture = wood_texture;
	}
	else {
		setColor(&material);
	}

	idriver->setMaterial(material);
	drawCapsuleMesh(length, radius);

	if (sim->getUseShadows())
	{
		// drawCylinderShadow(const double pos[3], const double R[12],  float length, float radius);
	}

	last_draw = CAPSULE;
}
void DrawPrimitives::drawCapsule(const double pos[3], const double R[12],
                    float length, float radius)
{
	static int num = 0; // この関数が1描画フレームに何回呼ばれたか数える
    static int NUM = 0; // 全ボックス数
    static double last_frame = 0; // 最近のフレーム数

    // FRAME_COUNT: 現在のフレーム数
    if (sim->getFrameCount() != last_frame) num = 0;

    if (num >= NUM)   // 箱の数よりも，呼び出し回数が多いときは新しいノードを生成する
    {
        myCapsuleNode[num] =
            new idsCapsuleSceneNode(length, radius, ismgr->getRootSceneNode(), ismgr, 666);

        if (tnum == DS_WOOD)
        {
            myCapsuleNode[num]->setMaterialTexture(0,wood_texture);
            myCapsuleNode[num]->setMaterialFlag(video::EMF_LIGHTING, false);
        }
        else
        {
            setTexture(CAPSULE, num, myCapsuleNode[num], capsuleTexture[num]);
			setColor(myCapsuleNode[num], red, green, blue, alpha);
            myCapsuleNode[num]->setMaterialFlag(video::EMF_LIGHTING, true);
        }
        NUM++;
    }
    else
    {
        sim->misc->updateForDraw(myCapsuleNode[num],pos, R);
    }

	if (sim->getUseShadows())
	{
		drawCapsuleShadow(pos,R,length, radius);
	}

    last_frame = sim->getFrameCount();
    num++;
}

void DrawPrimitives::drawCylinder(const double pos[3], const double R[12],
                    float length, float radius)
{
	static int num = 0; // この関数が1描画フレームに何回呼ばれたか数える
    static int NUM = 0; // 全ボックス数
    static double last_frame = 0; // 最近のフレーム数

    // frameCount: 現在のフレーム数
    if (sim->getFrameCount() != last_frame) num = 0;

    if (num >= NUM)   // 箱の数よりも，呼び出し回数が多いときは新しいノードを生成する
    {
        myCylinderNode[num] =
            new idsCylinderSceneNode(length, radius, ismgr->getRootSceneNode(), ismgr, 666);

        sim->misc->updateForDraw(myCylinderNode[num],pos, R);

        if (tnum == DS_WOOD)
        {
            myCylinderNode[num]->setMaterialTexture(0,wood_texture);
            myCylinderNode[num]->setMaterialFlag(video::EMF_LIGHTING, false);
        }
        else
        {
            setTexture(CYLINDER, num, myCylinderNode[num], cylinderTexture[num]);
			setColor(myCylinderNode[num], red, green, blue, alpha);
            myCylinderNode[num]->setMaterialFlag(video::EMF_LIGHTING, true);
        }

        NUM++;
    }
    else
    {
        sim->misc->updateForDraw(myCylinderNode[num],pos, R);
    }

	if (sim->getUseShadows())
	{
		//drawCylinderShadow(pos,R,length,radius);
	}

    last_frame = sim->getFrameCount();
    num++;
}

// draw a cylinder of length l and radius r, aligned along the z axis
void DrawPrimitives::drawCylinderShadow(const double pos[3], const double R[12],float l, float r)
{
    int i;
    float tmp,ny,nz,a,ca,sa;
    const int n = 24;	// number of sides to the cylinder (divisible by 4)
	float zoffset = 0;

	SColor color;
	setShadowDrawingMode(&color);

	sim->misc->odeToIrrTransform(pos, R);
	sim->misc->setShadowTransform();

    l *= 0.5;
    a = float(M_PI*2.0)/float(n);
    sa = (float) sin(a);
    ca = (float) cos(a);

    // draw cylinder body
    ny=1;
    nz=0;   // normal vector = (0,ny,nz)
    for (i=0; i<=n; i++)
    {
        // 四角形作成 make square
        u16 rectList[] = {0,1,2,1,3,2};
        S3DVertex rectVer[4];
        rectVer[0] = S3DVertex( ny*r,nz*r, l+zoffset, ny,nz,0, color, 0,0);// 上 upper
        rectVer[1] = S3DVertex( ny*r,nz*r,-l+zoffset,ny,nz,0, color, 0,0);// 右下 bottom right

        // rotate ny,nz
        tmp = ca*ny - sa*nz;
        nz = sa*ny + ca*nz;
        ny = tmp;

        rectVer[2] = S3DVertex( ny*r,nz*r, l+zoffset, ny,nz,0, color, 0,0);// 上 upper
        rectVer[3] = S3DVertex( ny*r,nz*r,-l+zoffset,ny,nz,0, color, 0,0);// 右下 bottom right

        //matrix4 matrix;
        //driver->setTransform(ETS_WORLD, matrix);// ワールドに反映
        idriver->drawIndexedTriangleList(rectVer, 4, rectList, 2);
    }

    ny=1;
    nz=0;

    u16 triList[] = {0,1,2};
    S3DVertex triVer[3];
    triVer[0] = S3DVertex(0, 0, l+zoffset,0,0,1, color, 0,0);// 上 upper

    for (i=0; i<=n; i++)
    {
        triVer[1] = S3DVertex(ny*r,nz*r,l+zoffset,0,0,1, color, 0,0);// 上 upper
        tmp = ca*ny - sa*nz;
        nz = sa*ny + ca*nz;
        ny = tmp;
        triVer[2] = S3DVertex(ny*r,nz*r,l+zoffset,0,0,1, color, 0,0);// 上 upper

        //matrix4 matrix2;
        //driver->setTransform(ETS_WORLD, matrix2);// ワールドに反映
        idriver->drawIndexedTriangleList(triVer, 3, triList, 1);
    }

    ny=1;
    nz=0;

    u16 triList2[] = {0,1,2};
    S3DVertex triVer2[3];
    triVer2[0] = S3DVertex(0, 0, -l+zoffset,0,0,-1, color, 0,0);// 上 upper

    for (i=0; i<=n; i++)
    {
        triVer2[1] = S3DVertex(ny*r,nz*r,-l+zoffset,0,0,-1, color, 0,0);// 上 upper

        tmp = ca*ny + sa*nz;
        nz = -sa*ny + ca*nz;
        ny = tmp;

        triVer2[2] = S3DVertex(ny*r,nz*r,-l+zoffset,0,0,-1, color, 0,0);// 上 upper

        //matrix4 matrix3;
        //driver->setTransform(ETS_WORLD, matrix3);// ワールドに反映
        idriver->drawIndexedTriangleList(triVer2, 3, triList2, 1);
    }
}


void DrawPrimitives::drawCapsuleShadow(const double pos[3], const double R[12], double length, double radius)
{
    int i,j;
    float tmp,nx,ny,nz,start_nx,start_ny,a,ca,sa;
    // number of sides to the cylinder (divisible by 4):
    const int n = capped_cylinder_quality*4;

	SColor capsule_color;
	setShadowDrawingMode(&capsule_color);

	float l = length;
    float r = radius;

    l *= 0.5;
    a = float(M_PI*2.0)/float(n);
    sa = (float) sin(a);
	ca = (float) cos(a);

	// draw cylinder body
	ny=1;
	nz=0;		  // normal vector = (0,ny,nz)

	sim->misc->odeToIrrTransform(pos, R);
	sim->misc->setShadowTransform();

	for (i=0; i<=n; i++)
	{
		//四角形作成
		u16 rectList[] = {0,2,1,1,2,3};
		S3DVertex rectVer[4];

		rectVer[0] = S3DVertex( nz*r, l,-ny*r, nz,0,-ny,capsule_color, 0,0);// 上
		rectVer[1] = S3DVertex( nz*r,-l,-ny*r, nz,0,-ny,capsule_color, 0,0);// 右下

		// rotate ny,nz
		tmp = ca*ny - sa*nz;
		nz = sa*ny + ca*nz;
		ny = tmp;

		rectVer[2] = S3DVertex( nz*r, l, -ny*r, nz,0, -ny,capsule_color, 0,0);// 上
		rectVer[3] = S3DVertex( nz*r,-l, -ny*r, nz,0, -ny,capsule_color, 0,0);// 右下

		//matrix4 matrix;
		//driver->setTransform(ETS_WORLD, core::matrix4());
		//driver->setTransform(ETS_WORLD, matrix);
		idriver->drawIndexedTriangleList(rectVer, 4, rectList, 2);
	}

	// draw first cylinder cap
	start_nx = 0;
	start_ny = 1;

	u16 triList1[] = {0,2,1},triList2[] = {0,2,1};

	S3DVertex triVer1[3],triVer2[3];

	for (j=0; j<(n/4); j++)
	{
		// get start_n2 = rotated start_n
		float start_nx2 =  ca*start_nx + sa*start_ny;
		float start_ny2 = -sa*start_nx + ca*start_ny;
		// get n=start_n and n2=start_n2
		nx = start_nx;
		ny = start_ny;
		nz = 0;
		float nx2 = start_nx2, ny2 = start_ny2, nz2 = 0;
		//glBegin (GL_TRIANGLE_STRIP);
		for (i=0; i<=n; i++)
		{
			triVer1[0] = S3DVertex(nz2*r, l+nx2*r, -ny2*r, nz2,nx2,-ny2, capsule_color, 0,0);// 上
			triVer1[1] = S3DVertex(nz *r, l+nx *r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上
			triVer2[0] = S3DVertex(nz *r, l+nx *r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上

			tmp = ca*ny2- sa*nz2;
			nz2 = sa*ny2 + ca*nz2;
			ny2 = tmp;
			tmp = ca*ny - sa*nz;
			nz  = sa*ny + ca*nz;
			ny  = tmp;

			triVer1[2] = S3DVertex(nz2*r, l+nx2*r, -ny2*r,  nz2, nx2,-ny2,  capsule_color, 0,0);// 上
			triVer2[1] = S3DVertex(nz *r, l+nx*r,  -ny *r,   nz,  nx, -ny,  capsule_color, 0,0);// 上
			triVer2[2] = S3DVertex(nz2*r, l+nx2*r, -ny2*r,  nz2, nx2,-ny2,  capsule_color, 0,0);// 上

			//matrix4 matrix;
			//driver->setTransform(ETS_WORLD, core::matrix4());
			//driver->setTransform(ETS_WORLD, matrix);
			idriver->drawIndexedTriangleList(triVer1, 3, triList1, 1);
			idriver->drawIndexedTriangleList(triVer2, 3, triList2, 1);
		}

		start_nx = start_nx2;
		start_ny = start_ny2;
	}

	// draw second cylinder cap
	start_nx = 0;
	start_ny = 1;

	u16 triList3[] = {0,1,2},triList4[] = {0,1,2};

	S3DVertex triVer3[3],triVer4[3];

	for (j=0; j<(n/4); j++)
	{
		// get start_n2 = rotated start_n
		float start_nx2 =  ca*start_nx - sa*start_ny;
		float start_ny2 =  sa*start_nx + ca*start_ny;
		// get n=start_n and n2=start_n2
		nx = start_nx;
		ny = start_ny;
		nz = 0;
		float nx2 = start_nx2, ny2 = start_ny2, nz2 = 0;
		//glBegin (GL_TRIANGLE_STRIP);
		for (i=0; i<=n; i++)
		{
			triVer3[0] = S3DVertex(nz2*r, -l+nx2*r,-ny2*r, nz2,nx2, -ny2, capsule_color, 0,0);// 上
			triVer3[1] = S3DVertex(nz *r, -l+nx*r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上
			triVer4[0] = S3DVertex(nz *r, -l+nx*r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上

			tmp = ca*ny2- sa*nz2;
			nz2 = sa*ny2 + ca*nz2;
			ny2 = tmp;
			tmp = ca*ny - sa*nz;
			nz = sa*ny + ca*nz;
			ny = tmp;

			triVer3[2] = S3DVertex(nz2*r, -l+nx2*r,-ny2*r,  nz2, nx2, -ny2, capsule_color, 0,0);// 上
			triVer4[1] = S3DVertex(nz *r, -l+nx *r,-ny *r,   nz,  nx,  -ny, capsule_color, 0,0);// 上
			triVer4[2] = S3DVertex(nz2*r, -l+nx2*r,-ny2*r,  nz2, nx2, -ny2, capsule_color, 0,0);// 上

			//matrix4 matrix;
			//driver->setTransform(ETS_WORLD, core::matrix4());
			//driver->setTransform(ETS_WORLD, matrix);
			idriver->drawIndexedTriangleList(triVer3, 3, triList3, 1);
			idriver->drawIndexedTriangleList(triVer4, 3, triList4, 1);
		}
		//glEnd();
		start_nx = start_nx2;
		start_ny = start_ny2;
	}
}


void DrawPrimitives::drawPatch(float pos[3],float R[12],float radius,float p1[3], float p2[3], float p3[3], int level)
{
    int i;
    if (level > 0)
    {
        float q1[3],q2[3],q3[3];    // sub-vertices
        for (i=0; i<3; i++)
        {
            q1[i] = 0.5f*(p1[i]+p2[i]);
            q2[i] = 0.5f*(p2[i]+p3[i]);
            q3[i] = 0.5f*(p3[i]+p1[i]);
        }
        float length1 = (float)(1.0/sqrt(q1[0]*q1[0]+q1[1]*q1[1]+q1[2]*q1[2]));
        float length2 = (float)(1.0/sqrt(q2[0]*q2[0]+q2[1]*q2[1]+q2[2]*q2[2]));
        float length3 = (float)(1.0/sqrt(q3[0]*q3[0]+q3[1]*q3[1]+q3[2]*q3[2]));
        for (i=0; i<3; i++)
        {
            q1[i] *= length1;
            q2[i] *= length2;
            q3[i] *= length3;
        }
        drawPatch(pos,R,radius,p1,q1,q3,level-1);
        drawPatch(pos,R,radius,q1,p2,q2,level-1);
        drawPatch(pos,R,radius,q1,q2,q3,level-1);
        drawPatch(pos,R,radius,q3,q2,p3,level-1);
    }
    else
    {
        // Create a triangle; 三角形作成
        matrix4 matrix;
        u16 triList[] = {0,1,2};
        S3DVertex triVer[3];

        SColor tmp_color = SColor((int) (alpha * 255), (int) (red * 255),
			(int) (green * 255), (int) (blue * 255));
        SColor tmp_color2 = 0xFFFF0000;
        triVer[0] = S3DVertex(p1[0],p1[1],p1[2], p1[0],p1[1],p1[2], tmp_color, 0,0);// 上 upper
        triVer[1] = S3DVertex(p2[0],p2[1],p2[2], p2[0],p2[1],p2[2], tmp_color, 0,0);// 右下 bottom right
        triVer[2] = S3DVertex(p3[0],p3[1],p3[2], p3[0],p3[1],p3[2], tmp_color, 0,0);// 左下 bottom left

        matrix.setTranslation(vector3df(pos[0],pos[2],pos[1]));
        matrix.setScale(vector3df(radius,radius,radius));
        idriver->setTransform(ETS_WORLD, matrix);// ワールドに反映
        idriver->drawIndexedTriangleList(triVer, 3, triList, 1);
    }
}

void DrawPrimitives::drawPatch2(float p1[3], float p2[3], float p3[3], int level)
{
    //cout << "*** pos=" << pos[0] << ", " << pos[1] << ", " << pos[2] << endl;
	if (level > 0)
    {
        float q1[3], q2[3], q3[3];    // sub-vertices
        for (int i=0; i<3; i++)
        {
            q1[i] = 0.5f*(p1[i]+p2[i]);
            q2[i] = 0.5f*(p2[i]+p3[i]);
            q3[i] = 0.5f*(p3[i]+p1[i]);
        }
        float length1 = (float)(1.0/sqrt(q1[0]*q1[0]+q1[1]*q1[1]+q1[2]*q1[2]));
        float length2 = (float)(1.0/sqrt(q2[0]*q2[0]+q2[1]*q2[1]+q2[2]*q2[2]));
        float length3 = (float)(1.0/sqrt(q3[0]*q3[0]+q3[1]*q3[1]+q3[2]*q3[2]));
        for (int i=0; i<3; i++)
        {
            q1[i] *= length1;
            q2[i] *= length2;
            q3[i] *= length3;
        }
		//cout << "*** pos=" << pos[0] << ", " << pos[1] << ", " << pos[2] << endl;
        drawPatch2(p1,q1,q3,level-1);
        drawPatch2(q1,p2,q2,level-1);
        drawPatch2(q1,q2,q3,level-1);
        drawPatch2(q3,q2,p3,level-1);
    }
    else
    {
        // Create a triangle; 三角形作成
        matrix4 matrix;
        u16 triList[] = {0,1,2};
        S3DVertex triVer[3];

        SColor tmp_color = SColor((int) (alpha * 255), (int) (red * 255),
			(int) (green * 255), (int) (blue * 255));
        //SColor tmp_color2 = 0xFFFF0000;
        triVer[0] = S3DVertex(p1[0],p1[1],p1[2], p1[0],p1[1],p1[2], tmp_color, 0,0); // 上 upper
        triVer[1] = S3DVertex(p2[0],p2[1],p2[2], p2[0],p2[1],p2[2], tmp_color, 0,0); // 右下 bottom right
        triVer[2] = S3DVertex(p3[0],p3[1],p3[2], p3[0],p3[1],p3[2], tmp_color, 0,0); // 左下 bottom left
        idriver->drawIndexedTriangleList(triVer, 3, triList, 1);
    }
}


void DrawPrimitives::setShadowDrawingMode(SColor *color)
{
	SMaterial material;
	//material.Wireframe = false;
	material.BackfaceCulling = true;  // カリング設定
	material.Lighting     = false;  // ライト反映
    material.ZBuffer      = false;  // Zバッファ
    material.ZWriteEnable = false;

	if (sim->getUseTextures())
	{
		material.TextureLayer[0].Texture = ground_texture;
		*color = SColor(0, (int) 255*shadowIntensity,
			(int) 255*shadowIntensity, (int) 255* shadowIntensity);
		//SColor tmp_color = 0xFFFFFFFF;
	}
	else
	{
		*color = SColor((int) 255, (int) (ground->red * 255*shadowIntensity),
			(int) (ground->green * 255*shadowIntensity),(int) (ground->blue * 255*shadowIntensity));
	}
	idriver->setMaterial(material);
}



// 影描画用
void DrawPrimitives::drawBoxShadow(const double pos[3], const double R[12], const double sides[3])
{
	const float gsize = sim->vision->getLightX();
	SColor box_color;

	setShadowDrawingMode(&box_color);

	// Irrlesson No.5 参照
	u16 list1[] = { 0, 1, 2,  2, 1, 3};
	u16 list2[] = { 4, 5, 6,  6, 5, 7};
	u16 list3[] = { 8, 9,10, 10, 9,11};
	u16 list4[] = {12,13,14, 14,13,15};
	u16 list5[] = {16,17,18, 18,17,19};
	u16 list6[] = {20,21,22, 22,21,23};

	f32 lx = sides[1]*0.5f;
	f32 ly = sides[2]*0.5f;
	f32 lz = sides[0]*0.5f;

	// 直方体
	S3DVertex vertex[24];
	//lx += ly * LIGHTY;


	//lz += ly * LIGHTX;
	f32 tx = 0, ty = 0, tz = 0;
    tx = pos[1];
	ty = pos[2];
	tz = -pos[0];

	// lx, lzが負にならないようにする

	tx +=  sim->vision->getLightX() * (2 * ly + ty);
	tz +=  sim->vision->getLightZ() * (2 * ly + ty);

	ty = 0;
	lx = 0;


	vertex[ 0] = S3DVertex(tx-lx, ty+ly,tz-lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//前左上
	vertex[ 1] = S3DVertex(tx+lx, ty+ly,tz-lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//前右上
	vertex[ 2] = S3DVertex(tx-lx,ty-ly,tz-lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//前左下
	vertex[ 3] = S3DVertex(tx+lx,ty-ly,tz-lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//前右下

	vertex[ 4] = S3DVertex(tx+lx,ty+ly,tz+lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//奥左上
	vertex[ 5] = S3DVertex(tx-lx,ty+ly,tz+lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//奥右上
	vertex[ 6] = S3DVertex(tx+lx,ty-ly,tz+lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//奥左下
	vertex[ 7] = S3DVertex(tx-lx,ty-ly,tz+lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//奥右下

	vertex[ 8] = S3DVertex(tx-lx, ty+ly,tz+lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//左左上
	vertex[ 9] = S3DVertex(tx-lx, ty+ly,tz-lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//左右上
	vertex[10] = S3DVertex(tx-lx,ty-ly, tz+lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//左左下
	vertex[11] = S3DVertex(tx-lx,ty-ly,tz-lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//左右下

	vertex[12] = S3DVertex(tx+lx,ty+ly,tz-lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//右左上
	vertex[13] = S3DVertex(tx+lx,ty+ly,tz+lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//右右上
	vertex[14] = S3DVertex(tx+lx,ty-ly,tz-lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//右左下
	vertex[15] = S3DVertex(tx+lx,ty-ly,tz+lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//右右下

	vertex[16] = S3DVertex(tx+lx,ty+ly,tz-lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//上左上
	vertex[17] = S3DVertex(tx-lx,ty+ly,tz-lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//上右上
	vertex[18] = S3DVertex(tx+lx,ty+ly,tz+lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//上左下
	vertex[19] = S3DVertex(tx-lx,ty+ly,tz+lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//上右下

	vertex[20] = S3DVertex(tx-lx,ty-ly,tz-lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//下左上
	vertex[21] = S3DVertex(tx+lx,ty-ly,tz-lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, gsize*ground->scale+ground->offsetY);//下右上
	vertex[22] = S3DVertex(tx-lx,ty-ly,tz+lz,  0,0,0, box_color,
		gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//下左下
	vertex[23] = S3DVertex(tx+lx,ty-ly,tz+lz,  0,0,0, box_color,
		-gsize*ground->scale+ground->offsetX, -gsize*ground->scale+ground->offsetY);//下右下

	sim->misc->odeToIrrRotation(R);
	//setShadowTransform();


	//matrix4 matrix2;
    //matrix2 = driver->getTransform(ETS_WORLD);
	//matrix2 *= setShadowTransform();

	//driver->setTransform(ETS_WORLD, matrix2);

	// 影用の射影
	//f32 fov = 30 * DEGTORAD;
	//f32 farPlane = 1000.0f;
	//vector3df light_pos = vector3df(LIGHTY, 1, -LIGHTX);
	//vector3df light_vec = vector3df(-LIGHTY, -1, LIGHTX);
	vector3df light_pos = sim->vision->getLight()->getLightData().Position;
	vector3df light_vec = sim->vision->getLight()->getLightData().Direction;

	// view and projection matrix;


	/* matrix4 projMat, viewMat;
	projMat.buildProjectionMatrixPerspectiveFovLH(fov,1.0f,1.0f,farPlane);
	viewMat.buildCameraLookAtMatrixLH(light_pos, light_vec, (light_pos-light_vec).dotProduct(vector3df(1.0f,0.0f,1.0f)) == 0.0f ? vector3df(0.0f,0.0f,1.0f) : vector3df(0.0f,1.0f,0.0f)); */

	//driver->setTransform(ETS_PROJECTION,projMat);
	//driver->setTransform(ETS_VIEW,viewMat);


	///matrix4 worldViewProj = projMat;
	//worldViewProj *= viewMat;
	//worldViewProj *= driver->getTransform(ETS_WORLD);
	//driver->setTransform(ETS_WORLD, worldViewProj);// ワールドに反映


	idriver->drawIndexedTriangleList(&vertex[0], 24, &list1[0], 2);
	idriver->drawIndexedTriangleList(&vertex[0], 24, &list2[0], 2);
	idriver->drawIndexedTriangleList(&vertex[0], 24, &list3[0], 2);
	idriver->drawIndexedTriangleList(&vertex[0], 24, &list4[0], 2);
	idriver->drawIndexedTriangleList(&vertex[0], 24, &list5[0], 2);
	idriver->drawIndexedTriangleList(&vertex[0], 24, &list6[0], 2);
}

void DrawPrimitives::drawSphereShadow(float px, float py, float pz, float radius)
{
	SMaterial material;
	SColor tmp_color;
    material.Lighting     = false;	// ライト反映
    material.ZBuffer      = false;  // Zバッファ
    material.ZWriteEnable = false;

	if (sim->getUseTextures())
	{
		//ITexture* Texture;
		material.TextureLayer[0].Texture = ground_texture;
		//tmp_color = SColor(20, (int) 255*shadowIntensity,
		//	(int) 255*shadowIntensity, (int) 255* shadowIntensity);
	}
	else
	{
		tmp_color = SColor((int) 255, (int) (ground->red * 255*shadowIntensity),
			(int) (ground->green * 255 * shadowIntensity),
			(int) (ground->blue * 255 * shadowIntensity));
	}
	idriver->setMaterial(material);

	static int init=0;
	static float len2,len1,scale;
	if (!init) {
		len2 = sim->vision->getLightX() * sim->vision->getLightX()
			 + sim->vision->getLightZ() * sim->vision->getLightZ();
		len1 = 1.0f/(float)sqrt(len2);
		scale = (float) sqrt(len2 + 1);
		init = 1;
	}

	// ode-> irr: x -> -z,  y->x, z->y
	// map sphere center to ground plane based on light vector
	px -= sim->vision->getLightZ() * pz;
	py -= sim->vision->getLightX() * pz;

	const float kx = 0.96592582628907f;
	const float ky = 0.25881904510252f;
	float x = radius, y=0;

	u16 shadowList[] = {0,2,1, 0,3,2, 0,4,3, 0,5,4, 0,6,5, 0,7,6, 0,8,7, 0,9,8, 0,10,9,
		0,11,10, 0,12,11, 0,13,12, 0,14,13, 0,15,14, 0,16,15, 0,17,16, 0,18,17,
		0,19,18, 0,20,19, 0,21,20, 0,22,21, 0,23,22};

	S3DVertex shadowVer[24];

	for (int i=0; i<24; i++) {
		// for all points on circle, scale to elongated rotated shadow and draw
		float x2 = (sim->vision->getLightX() * x * scale - sim->vision->getLightZ() * y) * len1 + px;
		float y2 = (sim->vision->getLightZ() * x * scale + sim->vision->getLightX() * y) * len1 + py;

		shadowVer[i] = S3DVertex(y2, 0, -x2, 0,0,0, tmp_color,
			y2 * ground->scale - ground->offsetX, - x2 * ground->scale + ground->offsetY);
		// rotate [z,x] vector
		float xtmp = kx*x - ky*y;
		y = ky*x + kx*y;
		x = xtmp;
	}

	matrix4 matrix;
	idriver->setTransform(ETS_WORLD, core::matrix4());
	idriver->setTransform(ETS_WORLD, matrix);
	idriver->drawIndexedTriangleList(shadowVer, 24, shadowList, 22);
}

void DrawPrimitives::drawSky(float view_xyz3[3])
{
    static float offset = 0.0f;
	const float ssize = 1000.0f;
	const float sky_scale   = 1.0f/4.0f;	// sky texture scale (1/size)
	const float sky_height  = 1.0f;		// sky height above viewpoint

    float x = ssize * sky_scale;
    float z = view_xyz3[1] + sky_height;

    SColor color;

    matrix4 matrix;
    SMaterial material;

    material.Lighting     = false;	// ライト反映
    material.ZBuffer      = false;  // Zバッファ
    material.ZWriteEnable = false;

    if (sim->getUseTextures())
    {
  		material.TextureLayer[0].Texture = sky_texture;  //1.4対応
        color = 0xFFFFFFFF; // alpha, red, green, blue
    }
    else
    {
        color = 0xFF0077FF; // alpha, red, green, blue
    }

    idriver->setMaterial(material);
    //driver->setFog(color, video::EFT_FOG_LINEAR, 250, 1000, .003f, true, false);
    //四角形作成 頂点の順番が時計周りが表
    u16 indices[] = {0,2,1, 1,2,3};
    S3DVertex vertices[4];

    vertices[0] = S3DVertex(-ssize+view_xyz3[0],  z,  ssize+view_xyz3[1],
                            0,-1,0, color,	  -x-offset, x+offset);
    vertices[1] = S3DVertex( ssize+view_xyz3[0],  z,  ssize+view_xyz3[1],
                             0,-1,0, color,		  x-offset, x+offset);
    vertices[2] = S3DVertex(-ssize+view_xyz3[0],  z, -ssize+view_xyz3[1],
                            0,-1,0, color,    -x-offset, -x+offset);
    vertices[3] = S3DVertex( ssize+view_xyz3[0],  z, -ssize+view_xyz3[1],
                             0,-1,0, color,     x-offset,-x+offset);

    offset -= 0.002f;    // moving cloud 雲を動かしている
    if (offset > 1) offset -= 1;

    idriver->setTransform(ETS_WORLD, matrix); // ワールドに反映
    idriver->drawIndexedTriangleList(vertices, 4, indices, 2);
}

void DrawPrimitives::drawGround()
{
    const float gsize = 100.0f;
    const float offset = 0;

    // 属性設定; set material
    matrix4 matrix;
    SMaterial material;
    SColor tmp_color;

    material.Lighting = false;	// ライト反映なし; no lighting

    if (sim->getUseTextures())
    {
      	material.TextureLayer[0].Texture = ground_texture;  //1.4対応
        tmp_color = 0xFFFFFFFF; // alpha, red, green, blue
    }
    else
    {
        tmp_color = SColor((int) 255, (int) (ground->red * 255),(int) (ground->green * 255),(int) (ground->blue * 255));
    }

    idriver->setMaterial(material);
    // idriver->setFog(color, video::EFT_FOG_LINEAR, 250, 1000, .003f, true, false); // fog

    //四角形作成
    u16 indices[] = {0,1,2, 1,3,2};
    S3DVertex vertices[4];

    float z1 = 1;
    vertices[0] = S3DVertex(-gsize,offset,gsize,   0,1,0, tmp_color,
		-gsize*ground->scale + ground->offsetX,	gsize*ground->scale + ground->offsetY);// 左上 upper left
    vertices[1] = S3DVertex( gsize,offset,gsize,   0,1,0, tmp_color,
         gsize*ground->scale + ground->offsetX,	gsize*ground->scale + ground->offsetY);// 右上 upper right
    vertices[2] = S3DVertex(-gsize,offset, -gsize , 0,1,0, tmp_color,
        -gsize*ground->scale + ground->offsetX,	-gsize*ground->scale + ground->offsetY);// 左下 bottom left
    vertices[3] = S3DVertex(gsize,offset, -gsize,  0,1,0, tmp_color,
         gsize*ground->scale + ground->offsetX,	-gsize*ground->scale + ground->offsetY);// 右下 bottom left

	matrix.setTranslation(vector3df(0,0,0));
    idriver->setTransform(ETS_WORLD, matrix);
    idriver->drawIndexedTriangleList(vertices, 4, indices, 2);
}

// Call this to update the current camera position. the bits in `mode' say
// if the left (1), middle (2) or right (4) mouse button is pressed, and
// (deltax,deltay) is the amount by which the mouse pointer has moved.
void diMotion(int mode, int deltax, int deltay)
{
    float side = 0.01f * float(deltax);
    float fwd = (mode==4) ? (0.01f * float(deltay)) : 0.0f;
    float s = (float) sin (degToRad(-sim->vision->view_hpr[2]));
    float c = (float) cos (degToRad(-sim->vision->view_hpr[2]));

    if (mode==1)
    {
        sim->vision->view_hpr[2] -= float (deltax) * 0.5f;
        sim->vision->view_hpr[1] += float (deltay) * 0.5f;
    }
    else
    {
        sim->vision->view_xyz[2] -= -s*side + c*fwd;
        sim->vision->view_xyz[0] +=  c*side + s*fwd;
        if (mode==2 || mode==5) sim->vision->view_xyz[1] += 0.01f * float(deltay);

    }
    sim->vision->wrapCameraAngles(sim->vision->view_hpr);
}

void DrawPrimitives::drawPyramidGrid()
{
    matrix4 matrix;
    SMaterial material;

	material.Lighting = false; //true;		// ライト反映なし; no lighting

	idriver->setMaterial(material);

    // 四面体作成; create a pyramid
    u16 list[] = {0,2,1, 0,3,2, 0,4,3, 0,1,4};
    static S3DVertex ver[5];
    static float side = 0.03f;

    SColor pcolor; // alpha, red, green, blue

    for (int i=-1; i<=1; i++)
    {
        for (int j=-1; j<=1; j++)
        {
            if ((i == 0) && (j == -1)) pcolor = 0xFFFF0000; // red
            else
            {
                if ((i ==  1) && (j == 0)) pcolor = 0xFF0000FF; // blue
                else                       pcolor = 0xFFFFFF00; // yellow
            }

            //ver[0] = S3DVertex( 0, side, 0,  0,1,-1, pcolor, 0,0);
            //ver[1] = S3DVertex(-k, 0, -k,  0,1,-1, pcolor, 0,0);
            //ver[2] = S3DVertex( k, 0, -k,  0,1,-1, pcolor, 0,0);
            //ver[3] = S3DVertex( k, 0,  k,  1,1, 0, pcolor, 0,0);
            //ver[4] = S3DVertex(-k, 0,  k, -1,1, 0, pcolor, 0,0);
            ver[0] = S3DVertex( 0, side,  0,     0,1,-1, pcolor, 0, 0);
            ver[1] = S3DVertex(-side, 0, -side,  0,1,-1, pcolor, 0, 0);
            ver[2] = S3DVertex( side, 0, -side,  0,1,-1, pcolor, 0, 0);
            ver[3] = S3DVertex( side, 0,  side,  1,1, 0, pcolor, 0 ,0);
            ver[4] = S3DVertex(-side, 0,  side, -1,1, 0, pcolor, 0, 0);

            matrix.setTranslation(vector3df(i,0,j));
            idriver->setTransform(ETS_WORLD, matrix);//ワールドに反映
            idriver->drawIndexedTriangleList(&ver[0], 5, &list[0], 4);
        }
    }
}

// 多角形を３角形に分割して表示
void DrawPrimitives::drawConvex(const double pos[3], const double R[12],  double *_planes,
				unsigned int _planecount, double *_points,   unsigned int _pointcount,
				unsigned int *_polygons)
{
	// 属性設定
    SMaterial material;

    material.Lighting = false;        // ライト反映なし
    material.BackfaceCulling = true;  // カリング設定
    matrix4 matrix;

	sim->misc->odeToIrrTransform(pos, R);

	u16 triList[] = {0,1,2};
    S3DVertex triVer[3];

    unsigned int polyindex=0;

    SColor tmp_color = SColor((int) (alpha * 255), (int) (0.75f * red * 255),
                              (int) (0.75f * green * 255), (int) (0.75f * blue * 255));

    for (unsigned int i=0; i<_planecount; i++)
    {
        unsigned int pointcount=_polygons[polyindex];
        polyindex++; // forループ先頭のpolyindexは多角形の頂点数が格納されている配列の添え字

        triVer[0] = S3DVertex(_points[(_polygons[polyindex]*3)+1],  _points[(_polygons[polyindex]*3)+2],
                - _points[_polygons[polyindex]*3], _planes[(i*4)+1], _planes[(i*4)+2],
				- _planes[(i*4)+0],tmp_color, 0,0);
        polyindex++;

        for (unsigned int j=0; j<pointcount-2; j++)
        {
            triVer[1] = S3DVertex(_points[(_polygons[polyindex]*3)+1], _points[(_polygons[polyindex]*3)+2],
                           - _points[_polygons[polyindex]*3],_planes[(i*4)+1],_planes[(i*4)+2],
						   - _planes[(i*4)+0], tmp_color, 0,0);
            polyindex++;

            triVer[2] = S3DVertex(_points[(_polygons[polyindex]*3)+1], _points[(_polygons[polyindex]*3)+2],
				         - _points[_polygons[polyindex]*3], _planes[(i*4)+1], _planes[(i*4)+2],
						 - _planes[(i*4)+0], tmp_color, 0,0);
            idriver->drawIndexedTriangleList(triVer, 3, triList, 1);
        }
        polyindex++;
    }

    idriver->setMaterial(material);

	last_draw = CONVEX;
}

void DrawPrimitives::drawTriangle(const double pos[3], const double R[12], const double *v0,
								const double *v1, const double *v2, int solid)
{
	// 属性設定
    SMaterial material;

    material.Lighting = true; // false;          // ライト反映なし
    material.BackfaceCulling = true;    // カリング設定

    if (solid == 0)   // wireframe mode
    {
        material.setFlag(EMF_WIREFRAME, true);
    }

    float u[3],v[3],normal[3];

    u[0] = float( v1[1] - v0[1] );
    u[1] = float( v1[2] - v0[2] );
    u[2] = float(-v1[0] + v0[0] );
    v[0] = float( v2[1] - v0[1] );
    v[1] = float( v2[2] - v0[2] );
    v[2] = float(-v2[0] + v0[0] );

    dCROSS(normal,=,u,v);

    sim->misc->normalizeVector3(normal);

    u16 triList[] = {0,1,2};
    S3DVertex triVer[3];

	float k1 = 1.0f;

	static double count = 0;

	static ITexture *texture;
	if (count == 0) {
		char *s =  (char*) calloc(strlen(iprefix)+20, sizeof(char));
		char *s2 = (char*) calloc(10, sizeof(char));
		strcpy(s, iprefix);
		strcat(s,"/mesh.bmp");
		sim->misc->makeTexture(s, red, green, blue);
		texture = idriver->getTexture(s);
		free(s);
		free(s2);
		count++;
	}
	material.TextureLayer[0].Texture = texture;

    //SColor tmp_color = SColor((int) (alpah * 255), (int) (k1 * red * 255),
    //                          (int) (k1 * green * 255), (int) (k1 * blue * 255));
	SColor tmp_color = 0xFFFFFFFF; // 0xFFFFFFFF;

    triVer[0] = S3DVertex(v0[1],v0[2],-v0[0],  normal[0],normal[1],normal[2], tmp_color, 0,0);//上
    triVer[1] = S3DVertex(v1[1],v1[2],-v1[0],  normal[0],normal[1],normal[2], tmp_color, 0,1);//上
    triVer[2] = S3DVertex(v2[1],v2[2],-v2[0],  normal[0],normal[1],normal[2], tmp_color, 1,0);//上

    sim->misc->odeToIrrTransform(pos, R);
	idriver->drawIndexedTriangleList(triVer, 3, triList, 1);
    idriver->setMaterial(material);
}

void DrawPrimitives::drawTriangle2(const double pos[3], const double R[12], const double *v0,
								const double *v1, const double *v2, int solid)
{
	// 属性設定
    SMaterial material;
	static int last_solid = 1;

	// solidとwireframeの２種類の描画があるため
	if ((last_draw != TRIANGLE) || (last_solid != solid)) {
		material.Lighting = true; // false;          // ライト反映なし
		material.BackfaceCulling = true;    // カリング設定

		if (solid == 0)   // wireframe mode
		{
			material.setFlag(EMF_WIREFRAME, true);
		}

		material.ColorMaterial = ECM_NONE;
		if (tnum == DS_WOOD)  // if texture number equals DS_WOODS
		{
			material.TextureLayer[0].Texture = wood_texture;
		}
		else {
			setColor(&material);
		}
	}


    float u[3], v[3], normal[3];

    u[0] = float( v1[1] - v0[1] );
    u[1] = float( v1[2] - v0[2] );
    u[2] = float(-v1[0] + v0[0] );
    v[0] = float( v2[1] - v0[1] );
    v[1] = float( v2[2] - v0[2] );
    v[2] = float(-v2[0] + v0[0] );

    dCROSS(normal,=,u,v);

    sim->misc->normalizeVector3(normal);

    u16 triList[] = {0,1,2};
    S3DVertex triVer[3];

	float k1 = 1.0f;

    triVer[0] = S3DVertex(v0[1],v0[2],-v0[0],  normal[0],normal[1],normal[2], 0xFFFFFFFF, 0,0);//上
    triVer[1] = S3DVertex(v1[1],v1[2],-v1[0],  normal[0],normal[1],normal[2], 0xFFFFFFFF, 0,1);//上
    triVer[2] = S3DVertex(v2[1],v2[2],-v2[0],  normal[0],normal[1],normal[2], 0xFFFFFFFF, 1,0);//上

	if ((last_draw != TRIANGLE) || (last_solid != solid)) {
		sim->misc->odeToIrrTransform(pos, R);
	}

	idriver->drawIndexedTriangleList(triVer, 3, triList, 1);

	if ((last_draw != TRIANGLE) || (last_solid != solid)) {
		idriver->setMaterial(material);
	}
	last_draw = TRIANGLE;
	last_solid = solid;
}

void DrawPrimitives::drawLine(const double pos1[3], const double pos2[3])
{
	if (last_draw != LINE) {
		SMaterial material;
		material.setFlag(video::EMF_LIGHTING, false); // true
		material.Thickness = 2;
		//material.AntiAliasing =  EAAM_OFF; //EAAM_FULL_BASIC; // EAAM_OFF;
		//material.ColorMaterial =  ECM_NONE;
		//material.GouraudShading = false;
		//material.Wireframe = true;

		idriver->setMaterial(material);
  		//idriver->setTransform(video::ETS_WORLD, core::matrix4());
		idriver->setTransform(video::ETS_WORLD, core::IdentityMatrix);
	}
    idriver->draw3DLine(vector3df(pos1[1],pos1[2],-pos1[0]),vector3df(pos2[1],pos2[2],-pos2[0]),color);
	last_draw = LINE;
}

void DrawPrimitives::setSphereQuality(int n)
{
	sphere_quality = n;
}

void DrawPrimitives::setCapsuleQuality(int n)
{
	capped_cylinder_quality = n;
}


extern "C" void diDrawBoxD(const double pos[3], const double R[12],const double sides[3])
{
    sim->pri->drawBox2(pos, R, sides);
}

extern "C" void diDrawBox2D(const double pos[3], const double R[12],const double sides[3])
{
    sim->pri->drawBox2(pos, R, sides);
}

extern "C" void diDrawSphereD(const double pos[3], const double R[12], double radius)
{
    sim->pri->drawSphere2(pos, R, radius);
}

extern "C" void diDrawSphere2D(const double pos[3], const double R[12], double radius)
{
    sim->pri->drawSphere2(pos, R, radius);
}


// カプセルの描画．カスタムシーンノードで実装．
// Draw a capsule by a custom scene node
extern "C" void diDrawCapsuleD(const double pos[3], const double R[12],
                    float length, float radius)
{
    sim->pri->drawCapsule2(pos, R, length, radius);
}

// Custom Scene Node
extern "C" void diDrawCylinderD(const double pos[3], const double R[12],
                     float length, float radius)
{
     sim->pri->drawCylinder2(pos, R, length, radius);
}


extern "C" void diDrawConvexD (const double pos[3], const double R[12],  double *_planes,
				unsigned int _planecount, double *_points,   unsigned int _pointcount,  unsigned int *_polygons)
{
    sim->pri->drawConvex(pos, R, _planes, _planecount, _points, _pointcount, _polygons);
}


extern "C" void diDrawLineD(const double pos1[3], const double pos2[3])
{
	sim->pri->drawLine(pos1, pos2);
}

// 球の描画品質; quality of sphere
extern "C" void diSetSphereQuality(int n)
{
    sim->pri->setSphereQuality(n);
}


// 円柱の描画品質；　quality of cylinder
extern "C" void diSetCapsuleQuality(int n)
{
    sim->pri->setCapsuleQuality(n);
}


extern "C" void diPrintf(int y,int x, const char *text, ...)
{
	char tmp[1024];
    va_list list;

	irr::gui::IGUIFont *defaultFont = sim->pri->getEnv()->getFont("../../media/font.bmp");
    va_start(list, text);
    vsnprintf(tmp, 1024, text, list);
    defaultFont->draw(tmp,rect<s32>(x,y,0,0),0xFF000000); // 赤色; red
	va_end(list);
}

extern "C" void diPrintw(int y,int x, const char *font,int color, const char*text, ...)
{
	char tmp[1024];
    va_list list;

    va_start(list, text);
    vsnprintf(tmp, 1024, text, list);

	irr::gui::IGUIFont* tmp_font = sim->pri->getEnv()->getFont(font);
    tmp_font->draw(tmp,rect<s32>(x,y,0,0),color);
    va_end(list);
}


extern "C" void diSetColorAlpha(float red, float green, float blue, float alpha)
{
    sim->pri->setColor(red, green, blue, alpha);
}


extern "C" void diSetColor(float red, float green, float blue)
{
    sim->pri->setColor(red, green, blue, 1.0);
}


extern "C" void diDrawTriangleD(const double pos[3], const double R[12], const double *v0,
								const double *v1, const double *v2, int solid)
{
    sim->pri->drawTriangle2(pos, R, v0, v1, v2, solid);
}


