// Copyright (C) 2010 Kosei Demura
// This file is part of the irrDrawStuff library.
// Please read copyright notice in irrdrawstuff.h for conditions of distribution and usage. 


#include "sceneNode.h"
#include "drawPrimitives.h"

namespace {
	SimLoop *sim = SimLoop::getInstance();
}

idsBoxSceneNode::idsBoxSceneNode(f32 sides[3], scene::ISceneNode* parent, scene::ISceneManager*mgr, s32 id)
: scene::ISceneNode(parent, mgr,id)
{
	material.Wireframe = false; // Draw as wireframe or filled triangles? Default: false
	material.Lighting  = true;  // Will this material be lighted? Default: true
	//        material.Lighting  = false;
	material.BackfaceCulling = false; //I s backface culling enabled? Default: true.

	f32 lx = sides[0]*0.5f;
	f32 ly = sides[2]*0.5f;
	f32 lz = sides[1]*0.5f;

	// rectangular solid
	SColor box_color = SColor((int) (sim->pri->getAlpha() * 255), (int) (sim->pri->getRed() * 255),
		(int) (sim->pri->getGreen() * 255), (int) (sim->pri->getBlue() * 255));
	
	// near side
	vertices[0] = S3DVertex(-lx, ly,-lz, 0,0,0, box_color, 0,0);// upper left
	vertices[1] = S3DVertex( lx, ly,-lz, 0,0,0, box_color, 0,0);// upper right
	vertices[2] = S3DVertex(-lx,-ly,-lz, 0,0,0, box_color, 0,0);// bottom left
	vertices[3] = S3DVertex( lx,-ly,-lz, 0,0,0, box_color, 0,0);// bottom right
	// back side
	vertices[4] = S3DVertex( lx, ly, lz, 0,0,0, box_color, 0,0);// upper left
	vertices[5] = S3DVertex(-lx, ly, lz, 0,0,0, box_color, 0,0);// upper right
	vertices[6] = S3DVertex( lx,-ly, lz, 0,0,0, box_color, 0,0);// bottom left
	vertices[7] = S3DVertex(-lx,-ly, lz, 0,0,0, box_color, 0,0);// bottom right

	boundingBox.reset(vertices[0].Pos);
	for (s32 i= 1; i<4;  ++i) boundingBox.addInternalPoint(vertices[i].Pos);
	for (s32 i=20; i<24; ++i) boundingBox.addInternalPoint(vertices[i].Pos);
}

idsBoxSceneNode::~idsBoxSceneNode()
{
}


void idsBoxSceneNode::OnRegisterSceneNode()
{
	if (IsVisible)
		SceneManager->registerNodeForRendering(this);

	ISceneNode::OnRegisterSceneNode();
}

void idsBoxSceneNode::render()
{
	u16 list1[] = { 0, 1, 2,  2, 1, 3, 4, 5, 6,  6, 5, 7};

	f32 lx = sides[0]*0.5f;
	f32 ly = sides[2]*0.5f;
	f32 lz = sides[1]*0.5f;

	// rectangular solid
	SColor box_color = SColor((int) (sim->pri->getAlpha() * 255), (int) (sim->pri->getRed() * 255), 
		(int) (sim->pri->getGreen() * 255), (int) (sim->pri->getBlue() * 255));

	// near side
	vertices[ 0] = S3DVertex(-lx, ly,-lz, 0,0,0, box_color, 0,0);// upper left
	vertices[ 1] = S3DVertex( lx, ly,-lz, 0,0,0, box_color, 0,0);// upper right
	vertices[ 2] = S3DVertex(-lx,-ly,-lz, 0,0,0, box_color, 0,0);// bottom left
	vertices[ 3] = S3DVertex( lx,-ly,-lz, 0,0,0, box_color, 0,0);// bottom right
	// back side
	vertices[4] = S3DVertex( lx, ly, lz, 0,0,0, box_color, 0,0);// upper left
	vertices[5] = S3DVertex(-lx, ly, lz, 0,0,0, box_color, 0,0);// upper right
	vertices[6] = S3DVertex( lx,-ly, lz, 0,0,0, box_color, 0,0);// bottom left
	vertices[7] = S3DVertex(-lx,-ly, lz, 0,0,0, box_color, 0,0);// bottom right

	video::IVideoDriver* driver = SceneManager->getVideoDriver();

	driver->setMaterial(material);
	driver->setTransform(video::ETS_WORLD, AbsoluteTransformation);

	driver->drawIndexedTriangleList(&vertices[0], 12, &list1[0], 4);
}

const core::aabbox3d<f32>& idsBoxSceneNode::getBoundingBox() const
{
	return boundingBox;
}

u32 idsBoxSceneNode::getMaterialCount() const
{
	return 1;
}

video::SMaterial& idsBoxSceneNode::getMaterial(u32 i)
{
	return material;
}


// Capsule Scene Node
idsCapsuleSceneNode:: idsCapsuleSceneNode(f32 l, f32 r, ISceneNode* parent, ISceneManager*mgr, s32 id)
	: ISceneNode(parent, mgr,id)
{
	material.Wireframe = false;
	material.Lighting = true; // false; true;

	length = l;
	radius = r;

	SColor capsule_color = SColor((int) (sim->pri->getAlpha() * 255), (int) (sim->pri->getRed() * 255), 
		(int) (sim->pri->getGreen() * 255), (int) (sim->pri->getBlue() * 255));

	vertices[0] = video::S3DVertex(-radius, 0.5*length, -radius,0,0,0, capsule_color, 0, 0);
	vertices[1] = video::S3DVertex(-radius, 0.5*length,  radius,0,0,0, capsule_color, 0, 0);
	vertices[2] = video::S3DVertex( radius, 0.5*length, -radius,0,0,0, capsule_color, 0, 0);
	vertices[3] = video::S3DVertex( radius, 0.5*length,  radius,0,0,0, capsule_color, 0, 0);
	vertices[4] = video::S3DVertex(-radius,-0.5*length, -radius,0,0,0, capsule_color, 0, 0);
	vertices[5] = video::S3DVertex(-radius,-0.5*length,  radius,0,0,0, capsule_color, 0, 0);
	vertices[6] = video::S3DVertex( radius,-0.5*length, -radius,0,0,0, capsule_color, 0, 0);
	vertices[7] = video::S3DVertex( radius,-0.5*length,  radius,0,0,0, capsule_color, 0, 0);

	boundingBox.reset(vertices[0].Pos);
	for (s32 i=1; i<8; ++i)
		boundingBox.addInternalPoint(vertices[i].Pos);
}

idsCapsuleSceneNode::~idsCapsuleSceneNode()
{
}


void idsCapsuleSceneNode::OnRegisterSceneNode()
{
	if (IsVisible) SceneManager->registerNodeForRendering(this);

	ISceneNode::OnRegisterSceneNode();
}

void idsCapsuleSceneNode::render()
{
	video::IVideoDriver* driver = SceneManager->getVideoDriver();

	driver->setMaterial(material);

	int i,j;
	float tmp,nx,ny,nz,start_nx,start_ny,a,ca,sa;
	// number of sides to the cylinder (divisible by 4):
	int capped_cylinder_quality = 3;
	const int n = capped_cylinder_quality*4;

	float l = length;
	float r = radius;

	// l = 1; r = 1;

	l *= 0.5;
	a = float(M_PI*2.0)/float(n);
	sa = (float) sin(a);
	ca = (float) cos(a);

	// draw cylinder body
	ny=1;
	nz=0;		  // normal vector = (0,ny,nz)

	SColor capsule_color = SColor((int) (sim->pri->getAlpha() * 255), (int) (sim->pri->getRed() * 255), 
		(int) (sim->pri->getGreen() * 255), (int) (sim->pri->getBlue() * 255));

	for (i=0; i<=n; i++)
	{
		//四角形作成
		u16 rectList[] = {0,2,1,1,2,3};
		S3DVertex rectVer[4];

		rectVer[0] = S3DVertex( nz*r, l,-ny*r, nz,0,-ny,capsule_color, 0,0);// 上
		rectVer[1] = S3DVertex( nz*r,-l,-ny*r, nz,0,-ny,capsule_color, 0,0);// 右下

		// rotate ny,nz
		tmp = ca*ny - sa*nz;
		nz = sa*ny + ca*nz;
		ny = tmp;

		rectVer[2] = S3DVertex( nz*r, l, -ny*r, nz,0, -ny,capsule_color, 0,0);// 上
		rectVer[3] = S3DVertex( nz*r,-l, -ny*r, nz,0, -ny,capsule_color, 0,0);// 右下

		driver->setTransform(video::ETS_WORLD, AbsoluteTransformation);
		driver->drawIndexedTriangleList(rectVer, 4, rectList, 2);
	}

	// draw first cylinder cap
	start_nx = 0;
	start_ny = 1;

	u16 triList1[] = {0,2,1},triList2[] = {0,2,1};

	S3DVertex triVer1[3],triVer2[3];

	for (j=0; j<(n/4); j++)
	{
		// get start_n2 = rotated start_n
		float start_nx2 =  ca*start_nx + sa*start_ny;
		float start_ny2 = -sa*start_nx + ca*start_ny;
		// get n=start_n and n2=start_n2
		nx = start_nx;
		ny = start_ny;
		nz = 0;
		float nx2 = start_nx2, ny2 = start_ny2, nz2 = 0;
		//glBegin (GL_TRIANGLE_STRIP);
		for (i=0; i<=n; i++)
		{
			triVer1[0] = S3DVertex(nz2*r, l+nx2*r, -ny2*r, nz2,nx2,-ny2, capsule_color, 0,0);// 上
			triVer1[1] = S3DVertex(nz *r, l+nx *r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上
			triVer2[0] = S3DVertex(nz *r, l+nx *r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上

			tmp = ca*ny2- sa*nz2;
			nz2 = sa*ny2 + ca*nz2;
			ny2 = tmp;
			tmp = ca*ny - sa*nz;
			nz  = sa*ny + ca*nz;
			ny  = tmp;

			triVer1[2] = S3DVertex(nz2*r, l+nx2*r, -ny2*r,  nz2, nx2,-ny2,  capsule_color, 0,0);// 上
			triVer2[1] = S3DVertex(nz *r, l+nx*r,  -ny *r,   nz,  nx, -ny,  capsule_color, 0,0);// 上
			triVer2[2] = S3DVertex(nz2*r, l+nx2*r, -ny2*r,  nz2, nx2,-ny2,  capsule_color, 0,0);// 上

			driver->setTransform(video::ETS_WORLD, AbsoluteTransformation);
			driver->drawIndexedTriangleList(triVer1, 3, triList1, 1);

			driver->drawIndexedTriangleList(triVer2, 3, triList2, 1);
		}

		start_nx = start_nx2;
		start_ny = start_ny2;
	}

	// draw second cylinder cap
	start_nx = 0;
	start_ny = 1;

	u16 triList3[] = {0,1,2},triList4[] = {0,1,2};

	S3DVertex triVer3[3],triVer4[3];

	for (j=0; j<(n/4); j++)
	{
		// get start_n2 = rotated start_n
		float start_nx2 =  ca*start_nx - sa*start_ny;
		float start_ny2 =  sa*start_nx + ca*start_ny;
		// get n=start_n and n2=start_n2
		nx = start_nx;
		ny = start_ny;
		nz = 0;
		float nx2 = start_nx2, ny2 = start_ny2, nz2 = 0;
		//glBegin (GL_TRIANGLE_STRIP);
		for (i=0; i<=n; i++)
		{
			triVer3[0] = S3DVertex(nz2*r, -l+nx2*r,-ny2*r, nz2,nx2, -ny2, capsule_color, 0,0);// 上
			triVer3[1] = S3DVertex(nz *r, -l+nx*r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上
			triVer4[0] = S3DVertex(nz *r, -l+nx*r, -ny *r,  nz, nx, -ny, capsule_color, 0,0);// 上

			tmp = ca*ny2- sa*nz2;
			nz2 = sa*ny2 + ca*nz2;
			ny2 = tmp;
			tmp = ca*ny - sa*nz;
			nz = sa*ny + ca*nz;
			ny = tmp;

			triVer3[2] = S3DVertex(nz2*r, -l+nx2*r,-ny2*r,  nz2, nx2, -ny2, capsule_color, 0,0);// 上
			triVer4[1] = S3DVertex(nz *r, -l+nx *r,-ny *r,   nz,  nx,  -ny, capsule_color, 0,0);// 上
			triVer4[2] = S3DVertex(nz2*r, -l+nx2*r,-ny2*r,  nz2, nx2, -ny2, capsule_color, 0,0);// 上


			driver->setTransform(video::ETS_WORLD, AbsoluteTransformation);
			driver->drawIndexedTriangleList(triVer3, 3, triList3, 1);
			driver->drawIndexedTriangleList(triVer4, 3, triList4, 1);
		}
		//glEnd();
		start_nx = start_nx2;
		start_ny = start_ny2;
	}
}

const core::aabbox3d<f32>& idsCapsuleSceneNode::getBoundingBox() const
{
	return boundingBox;
}

u32 idsCapsuleSceneNode::getMaterialCount() const
{
	return 1;
}

video::SMaterial& idsCapsuleSceneNode::getMaterial(u32 i)
{
	return material;
}


idsCylinderSceneNode::idsCylinderSceneNode(f32 l, f32 r, ISceneNode* parent, ISceneManager*mgr, s32 id)
: ISceneNode(parent, mgr,id)
{
	material.Wireframe = false;
	material.Lighting  = true;

	length = l;
	radius = r;

	SColor cylinder_color = SColor((int) (sim->pri->getAlpha() * 255), (int) (sim->pri->getRed() * 255),
		(int) (sim->pri->getGreen() * 255), (int) (sim->pri->getBlue() * 255));

	vertices[0] = video::S3DVertex(-radius, 0.5*length, radius, 0,0,0, cylinder_color, 0, 0);
	vertices[1] = video::S3DVertex( radius, 0.5*length, radius, 0,0,0, cylinder_color, 0, 0);
	vertices[2] = video::S3DVertex(-radius, 0.5*length,-radius, 0,0,0, cylinder_color, 0, 0);
	vertices[3] = video::S3DVertex( radius, 0.5*length,-radius, 0,0,0, cylinder_color, 0, 0);
	vertices[4] = video::S3DVertex(-radius,-0.5*length, radius, 0,0,0, cylinder_color, 0, 0);
	vertices[5] = video::S3DVertex( radius,-0.5*length, radius, 0,0,0, cylinder_color, 0, 0);
	vertices[6] = video::S3DVertex(-radius,-0.5*length,-radius, 0,0,0, cylinder_color, 0, 0);
	vertices[7] = video::S3DVertex( radius,-0.5*length,-radius, 0,0,0, cylinder_color, 0, 0);

	boundingBox.reset(vertices[0].Pos);
	for (s32 i=1; i<8; ++i)
		boundingBox.addInternalPoint(vertices[i].Pos);
}

idsCylinderSceneNode::~idsCylinderSceneNode()
{
}


void idsCylinderSceneNode::OnRegisterSceneNode()
{
	if (IsVisible)
		SceneManager->registerNodeForRendering(this);

	ISceneNode::OnRegisterSceneNode();
}

void idsCylinderSceneNode::render()
{
	video::IVideoDriver* driver = SceneManager->getVideoDriver();

	driver->setMaterial(material);

	int i;

	float l = length;
	float r = radius;

	float tmp,ny,nz,a,ca,sa;
	float zoffset = 0;
	const int n = 24;	// number of sides to the cylinder (divisible by 4)

	l *= 0.5;
	a = float(M_PI*2.0)/float(n);
	sa = (float) sin(a);
	ca = (float) cos(a);

	// draw cylinder body
	ny=1;
	nz=0;   // normal vector = (0,ny,nz)


	SColor cylinder_color = SColor((int) (sim->pri->getAlpha() * 255), (int) (sim->pri->getRed() * 255), 
		(int) (sim->pri->getGreen() * 255), (int) (sim->pri->getBlue() * 255));

	for (i=0; i<=n; i++)
	{
		// make square
		u16 rectList[] = {0,2,1,1,2,3};

		S3DVertex rectVer[4];
		rectVer[0] = S3DVertex( nz*r, l+zoffset,-ny*r, nz,0,-ny, cylinder_color, 0,0); //上
		rectVer[1] = S3DVertex( nz*r,-l+zoffset,-ny*r, nz,0,-ny, cylinder_color, 0,0); //右下

		// rotate ny,nz
		tmp = ca*ny - sa*nz;
		nz = sa*ny + ca*nz;
		ny = tmp;

		rectVer[2] = S3DVertex( nz*r, l+zoffset,-ny*r, nz, 0, -ny, cylinder_color, 0,0); //上
		rectVer[3] = S3DVertex( nz*r,-l+zoffset,-ny*r, nz, 0, -ny, cylinder_color, 0,0); //右下

		driver->setTransform(video::ETS_WORLD, AbsoluteTransformation);
		driver->drawIndexedTriangleList(rectVer, 4, rectList, 2);
	}

	ny=1;
	nz=0;		  // normal vector = (0,ny,nz)

	u16 triList[] = {0,2,1};
	S3DVertex triVer[3];
	triVer[0] = S3DVertex(0, l+zoffset, 0, 0,1,0, cylinder_color, 0,0);//上

	for (i=0; i<=n; i++)
	{
		triVer[1] = S3DVertex(nz*r,l+zoffset,-ny*r,0,1,0, cylinder_color, 0,0);//上

		tmp = ca*ny - sa*nz;
		nz  = sa*ny + ca*nz;
		ny  = tmp;

		triVer[2] = S3DVertex(nz*r,l+zoffset,-ny*r,0,1,0, cylinder_color, 0,0);//上

		driver->setTransform(video::ETS_WORLD, AbsoluteTransformation);
		driver->drawIndexedTriangleList(triVer, 3, triList, 1);
	}

	ny=1;
	nz=0;		  // normal vector = (0,ny,nz)

	u16 triList2[] = {0,2,1};
	S3DVertex triVer2[3];
	triVer2[0] = S3DVertex(0, -l+zoffset,0, 0, -1, 0, cylinder_color, 0,0);//上

	for (i=0; i<=n; i++)
	{
		triVer2[1] = S3DVertex(nz*r,-l+zoffset,-ny*r,0,-1,0, cylinder_color, 0,0);//上

		tmp = ca*ny + sa*nz;
		nz = -sa*ny + ca*nz;
		ny = tmp;

		triVer2[2] = S3DVertex(nz*r,-l+zoffset, -ny*r,0,-1,0, cylinder_color, 0,0);//上

		driver->setTransform(video::ETS_WORLD, AbsoluteTransformation);
		driver->drawIndexedTriangleList(triVer2, 3, triList2, 1);
	}
}

const core::aabbox3d<f32>& idsCylinderSceneNode::getBoundingBox() const
{
	return boundingBox;
}

u32 idsCylinderSceneNode::getMaterialCount() const
{
	return 1;
}

video::SMaterial& idsCylinderSceneNode::getMaterial(u32 i)
{
	return material;
}
