// Copyright (C) 2010 Kosei Demura
// This file is part of the irrDrawStuff library.
// Please read copyright notice in irrdrawstuff.h for conditions of distribution and usage.

#include "simLoop.h"
#include "eventReceiver.h"
#include "drawPrimitives.h"
#include "drawMesh.h"
#include "vision.h"
#include "misc.h"
#include <string>
#include <sstream>
#include <iostream>


using namespace irr::gui;

idsEventReceiver receiver;

void SimLoop::init()
{
	frameCount   = 0;   // フレーム数の初期化
	currentState = 0;   // 現在の状態
	run2         = 1;   // 1 if simulation run
	useTextures  = 1;	// 1 if textures to be drawn
	useShadows   = 1;	// 1 if shadows to be drawn
}

double SimLoop::getFrameCount()
{
	return frameCount;
}

int SimLoop::getCurrentState()
{
	return currentState;
}

void SimLoop::setCurrentState(int state)
{
	currentState = state;
}

bool SimLoop::getUseShadows()
{
	return useShadows;
}

void SimLoop::setUseShadows(bool a)
{
	useShadows = a;
}

bool SimLoop::getUseTextures()
{
	return useTextures;
}

void SimLoop::setUseTextures(bool a)
{
	useTextures = a;
}

bool SimLoop::getRun2()
{
	return run2;
}

void SimLoop::setRun2(bool a)
{
	run2 = a;
}

double SimLoop::elapsedTime()
{
    static double prev=0.0;
    double currentTime;
    double curr;

#ifdef WIN32
    currentTime = timeGetTime();
    curr = currentTime/1000.0;
#else
    timeval tv ;
    gettimeofday(&tv, 0);
    curr = tv.tv_sec + (double) tv.tv_usec / 1000000.0 ;
#endif

    if (!prev) prev = curr;
    double retval = curr-prev;
    prev=curr;
    return retval;
}

void SimLoop::diStartGraphics(int width, int height, diFunctions *fn)
{
    // DrawPrimitivesクラスのコンストラクタへ移動
	// 空，地面，木，格子テクスチャの割り当て
}

void SimLoop::showUsage()
{
    static bool first_time = true;

	if (first_time)
    {
        fprintf
        (
            stderr,
            "\n"
            //"\nODE with IrrDrawStuff %d.%02d\n"
            //"\nODE with IrrDrawStuff %d.%d.%d\n"
            "\nODE with IrrDrawStuff %s\n"
            "   Ctrl-P : pause / unpause (or say `-pause' on command line).\n"
            "   Ctrl-O : single step when paused.\n"
            "   Ctrl-T : toggle textures (or say `-notex' on command line).\n"
            "   Ctrl-S : toggle shadows (or say `-noshadow' on command line).\n"
            "   Ctrl-V : print current viewpoint coordinates (x,y,z,h,p,r).\n"
            //"   Ctrl-W : write frames to ppm files: frame/frameNNN.ppm\n"
            "   Ctrl-X : exit.\n"
            "	 Up Key:    Change the camera position to +X direction. \n"
            "	 Down Key:  Change the camera position to -X direction \n."
            "	 Right Key: Change the camera position to +Y direction. \n"
            "	 Left Key:  Change the camera position to -Y direction. \n"
            "\n"
            "Change the camera position by clicking + dragging in the window.\n"
            "   Left button - pan and tilt.\n"
            "   Right button - forward and sideways.\n"
            "   Left + Right button (or middle button) - sideways and up.\n"
            //"\n", (int) (DS_VERSION >> 8), (int) (DS_VERSION & 0xff)
           // "\n", (int) (DS_VERSION / 100), (DS_VERSION / 10) % 10, DS_VERSION % 100
            "\n", DS_VERSION
        );
        first_time = false;
    }
}


//　デバイスの生成;  Create a device
void SimLoop::createSetupDevice(const int window_width, const int window_height)
{

    // レンダリング環境、ウインドウサイズ、色深度(16,32),フルスクリーン、ステンシルバッファ(影）、vsync, イベント処理;
    // rendering environment, widow size, color depth, full screen, stencil buffer, vsynce, receiver

    // メニューバーの分だけウインドウの縦を15ピクセル増加している
	// OpenGL
	device = createDevice(EDT_OPENGL, dimension2d<u32>(window_width, window_height+20),16,
		false, true, false, 0);

    // Direct Xを使うと表示がおかしいバグあり
    //device = createDevice(EDT_DIRECT3D9, dimension2d<u32>(window_width, window_height+20),16,
	//	false, true, true, 0);

    if (device == 0)
    {
        cout << "device creation error \n" << endl;
        exit(1);
    }

	// ウインドウのサイズ変更を有効化; window resizable
	device->setResizable(true);
	//device->getFileSystem()->addFolderFileArchive("../../media/");
	device->setWindowCaption(L"irrDrawstuff - Loading...");


    // ビデオドライバ、シーンマネージャ,GUI環境;  Video driver, scene manager, gui environment
    driver = device->getVideoDriver();
	driver->setTextureCreationFlag(video::ETCF_ALWAYS_32_BIT, true);

	smgr = device->getSceneManager();
	smgr->getParameters()->setAttribute(scene::COLLADA_CREATE_SCENE_INSTANCES, true);
}


// GUIの作成
void SimLoop::createSetupGUI()
{
	IGUIContextMenu *submenu;

	env  = device->getGUIEnvironment();

	// フォントの設定
    IGUIFont* defaultFont = env->getFont("../../media/font.bmp");

	// set a font
	IGUISkin* skin = env->getSkin();
	//IGUIFont* font2 = env->getFont("../../fonthaettenschweiler.bmp");
	//IGUIFont* font2 = env->getFont("../../media/ariel10.xml");
	//IGUIFont* font2 = env->getFont("fontarialbrack26.bmp");
	IGUIFont* font2 = env->getFont("../../media/fontcourier.bmp");
	if (font2) skin->setFont(font2);

	// メニュー作成; create menu
	IGUIContextMenu* menu = env->addMenu();
	//core::rect<s32> r(0,0,50,50);
	//gui::IGUIContextMenu* menu = env->addContextMenu(r,0);
	menu->addItem(L"File", -1, true, true);
	menu->addItem(L"Simulation", -1, true, true);
	menu->addItem(L"Help", -1, true, true);


	submenu = menu->getSubMenu(0);
	submenu->addItem(L"Open Model File & Texture...", GUI_ID_OPEN_MODEL);
	submenu->addItem(L"Exit	Ctrl+X", GUI_ID_QUIT);

	submenu = menu->getSubMenu(1);
	submenu->addItem(L"Pause	Ctrl+P", GUI_ID_PAUSE, true, false, false);
	submenu->addItem(L"Single Step	Ctrol+O", GUI_ID_SINGLE_STEP, true, false, false);
	submenu->addItem(L"Performance Monitor", GUI_ID_MONITOR, false, false, false);
	submenu->addSeparator();
	submenu->addItem(L"Shadows	Ctrl+S", GUI_ID_SHADOWS, true, false, false);
	submenu->addItem(L"Textures	Ctrol+T", GUI_ID_TEXTURES, true, false, false);
	submenu->addItem(L"Save Settings",GUI_ID_SAVE_SETTINGS, false, false, false);

	submenu = menu->getSubMenu(2);
	submenu->addItem(L"About", GUI_ID_ABOUT);

	// disable alpha
	for (s32 i=0; i<gui::EGDC_COUNT ; ++i)
	{
		//video::SColor col = env->getSkin()->getColor((gui::EGUI_DEFAULT_COLOR)i);
		//col.setAlpha(255);
		//env->getSkin()->setColor((gui::EGUI_DEFAULT_COLOR)i, col);
		env->getSkin()->setColor( (gui::EGUI_DEFAULT_COLOR)i, SColor(200,200,200,200));
	}

	env->getSkin()->setColor( EGDC_3D_DARK_SHADOW, SColor( 200, 200, 200, 200) ); //暗い影
	env->getSkin()->setColor( EGDC_3D_FACE, SColor( 200, 250, 250, 250) );  // 表面
	env->getSkin()->setColor( EGDC_GRAY_TEXT, SColor( 255, 200, 200, 200) );  // 利用不能なテキスト
	env->getSkin()->setColor( EGDC_BUTTON_TEXT, SColor( 255, 0, 0, 0) );  // メニューのテキスト
	env->getSkin()->setColor( EGDC_HIGH_LIGHT_TEXT, SColor( 255, 255, 255, 255) );  // メニューのテキスト


}

std::string toString(double f, int digits)
//stringw toString(double f, int digits)
{
    std::ostringstream oss;
    //stringw oss;
    oss << setprecision(digits) << setiosflags(ios::fixed) << f;
    return oss.str();
}


// フレームの描画; Draw a frame
void SimLoop::diDrawFrame(int width, int height, diFunctions *fn, int pause)
{
    static int lastFPS = -1;
    static double sim_time  = 0; // simulation time
    static double real_time = 0; // real time

	assert(!(currentState < 1) && "Internal error at diDrawFrame");

	currentState = 2;

	// disable mouse cursor trap;
	// マウスカーソルがウインドウ内に補足されるのを止める
	vision->getCamera()->setInputReceiverEnabled(false);

    if (device->isWindowActive())
    {
        // 描画命令開始。描画命令はbeginSceneからendSceneに入れる。
        // beginシーンは色、深度をクリアする。
        // SColor(u32 a, u32 r, u32 g, u32 b) 値は0?255

        // setup viewport
        // driver->setViewPort(rect<s32>(0,0,width,height));

        // 毎回地面の色で初期化している。地面の色を変更するときはids->groundR,ids->groundG,_Bの値を変更する。
		irr::video::SColor color;

        if (useTextures)
        {
            color = 0xFF7E7C7B; //SColor(255, 129, 129, 129);
            driver->beginScene(true, true, color);
        }
        else
        {
            color = SColor((int) 255, (int) (pri->ground->red * 255),(int) (pri->ground->red * 255),
				(int) (pri->ground->blue * 255));
            driver->beginScene(true, true, color);
        }

        // カメラのセットアップ; Setup the camera
        vision->setupCamera(width,height);

		// ミップマップス有効;  mipmaps on
        driver->setTextureCreationFlag(video::ETCF_CREATE_MIP_MAPS, true);


		// 空の描画; draw the sky
        pri->drawSky(vision->view_xyz);

        // 地面の描画; draw the ground
        pri->drawGround();

		// ミップマップス無効;  mipmaps off
		driver->setTextureCreationFlag(video::ETCF_CREATE_MIP_MAPS, false);

        // ピラミッド形状のマーカーの描画; draw pyramidal grids
        pri->drawPyramidGrid();

        //  GUI環境の取得; Get the GUI environment
        env = device->getGUIEnvironment();

       	// 最後に描画した基本図形をリセット
		pri->resetLastDraw();

        // ステップ関数; Step function of ODE
        if (fn->step) fn->step(pause);
        sim_time += fn->sim_time_per_loop; // Simulation time; 経過時間の計測


        // シーンマネージャによる描画; draw by scene manager
        smgr->drawAll();

        // GUIの描画; draw by GUI
        env->drawAll();

         // 視点，視線の保存; save current xyz, hpr
        for (int i = 0; i < 3; i++)
        {
            vision->view_old_xyz[i] = vision->view_xyz[i];
            vision->view_old_hpr[i] = vision->view_hpr[i];
        }

        // 描画命令終了。画面に表示する。
        // 原因がわからないが１回目のループでは表示がおかしくなるのでスキップ
        if (frameCount > 1) driver->endScene();

        // FPSの取得; Get FPS
        int fps = driver->getFPS();

        // FPSの表示; Display FPS
        if (lastFPS != fps)
        {
          	core::stringw str = L"irrDrawStuff v";
          	static char str1[20], str2[20], str3[20];
			str += DS_VERSION;
			str += " [";
            str += driver->getName();
            str += "] FPS:";
            str += fps;
            if (fn->sim_time_per_loop != 0) {
                str += " Sim:";
                sprintf(str1,"%.1fs",sim_time);
                str += str1;
                str += " Real:";
                real_time += elapsedTime();
                sprintf(str2,"%.1fs",real_time);
                str += str2;
                str += " [";
                if (real_time != 0.0) sprintf(str3,"%.2fx",sim_time/real_time);
                str += str3;
                str += "]";
            }
            else {
                str += " Elapsed Time:";
                real_time += elapsedTime();
                //str += core::stringw(toString(real_time, 2));
                sprintf(str2,"%.1f[s]",real_time);
                str += str2;
            }
            device->setWindowCaption(str.c_str());
            lastFPS = fps;
        }
        frameCount++; // 描画フレームをカウン; Count frame number
    }
}


void SimLoop::diPlatformSimLoop(int window_width, int window_height, diFunctions *fn,
                               int initial_pause)
{
    // initMotionModel(); // diSimulationLoopから移動

 	// 使い方の表示; show usage
	showUsage();

 	// デバイスの生成と設定; create and setup a device
	createSetupDevice(window_width, window_height);

	// GUIの生成と設定; Create and setup GUI
	createSetupGUI();

	// ビジョンクラスの生成，コンストラクタでカメラの設定を行っている
	vision = new Vision(smgr);


	// レシーバークラスの生成;
	//receiver = new idsEventReceiver();
	device->setEventReceiver(&receiver);
	receiver.init(device, fn, initial_pause);

	// 基本描画クラスの生成
	pri = new DrawPrimitives(driver, smgr, env);

    // テクスチャパスをprefixに代入; Set texture path to prefix
	if (fn->path_to_textures) pri->setTexturePath(fn->path_to_textures);
    else  pri->setTexturePath((char *) DEFAULT_PATH_TO_TEXTURES);

	// メッシュクラスの生成
	mesh = new DrawMesh(smgr);

	// その他クラスの生成
	misc = new Misc(driver, device);

    // グラフィクスの開始; Start graphics;
    diStartGraphics(window_width, window_height, fn); // 使っていない

    // 視点，視線の設定; Set eye point and dirrection;
    if (fn->start) fn->start();

	// ライトの作成と設定; Create and setup a light
	vision->createSetupLight();

	// 3D モデルのロード
	//misc->loadModel("../../../irrlicht-1.6.1/media/dwarf.x");

	// カメラの作成と設定; Create and setup a camera
	vision->createSetupCamera();

    //　描画ループ ウインドウが閉じられるまで回る
    while (device->run() && run2 && driver)
    {
        if (device->isWindowActive())
		{
			// フレームの描画
			diDrawFrame(window_width, window_height, fn, receiver.getPause() && !receiver.getSingleStep());
			receiver.setSingleStep(0);
		}
		else {
			device->yield();
		}
    }

    if (fn->stop) fn->stop();
    device->drop();
}
