// Copyright (C) 2010 Kosei Demura
// This file is part of the irrDrawStuff library.
// Please read copyright notice in irrdrawstuff.h for conditions of distribution and usage. 

#include "vision.h"

namespace {
	SimLoop *sim = SimLoop::getInstance();
}

Vision::Vision(ISceneManager *smgr)
{
	view_xyz[0] =  0;  // ODE is 2
	view_xyz[1] =  1;  // ODE is 0
	view_xyz[2] = -2;  // ODE is 1
	view_hpr[0] =  0;  // ODE is 180
	view_hpr[1] =  0;  // ODE is 0
	view_hpr[2] = -180; // ODE is 0
	lightX = 0.4f;  // 0.4f
	lightZ = 1.0f;
	ismgr = smgr;
}

Vision::~Vision()
{
}

float Vision::getLightX()
{
	return lightX;
}

float Vision::getLightZ()
{
	return lightZ;
}

scene::ICameraSceneNode* Vision::getCamera()
{
	return camera;
}

scene::ILightSceneNode* Vision::getLight()
{
	return light;
}



void Vision::createSetupLight()
{
	// ライトの作成と設定; Create and setup a light
    // addLightScenNode(parent, position, color, radius, id)
    light = ismgr->addLightSceneNode(0);// ライトタイプの設定; set light type
	light->getLightData().Type = ELT_DIRECTIONAL; // ELT_POINT
	light->setPosition(vector3df(10*lightX, 10*1.0f, 10*lightZ)); // 位置設定
  	light->setRotation(vector3df(radToDeg(atan2(1,lightZ)),radToDeg(atan2(-lightX, lightZ)), 0)); // 方向
	//light->getLightData().Type = ELT_POINT; // ELT_POINT
    light->enableCastShadow(true); // 影の有効化; enable shadow
	
	int   value   = 127;
    ismgr->setAmbientLight(SColor(0,value, value, value));
}



void Vision::createSetupCamera()
{
	targetNode = ismgr->addEmptySceneNode(0);  // カメラのターゲットを作成, create a target for camera
    cameraman  = ismgr->addEmptySceneNode(targetNode);
    cameraman->setPosition(vector3df(view_xyz[0],view_xyz[1],view_xyz[2]));
    camera = ismgr->addCameraSceneNode(cameraman); //カメラ作成

    //camera->setPosition(vector3df(view_xyz[0], view_xyz[1], view_xyz[2]));
    targetNode->setRotation(vector3df(view_hpr[0],0.0f, 0.0f));
    targetNode->setRotation(vector3df(0.0f, 0.0f,view_hpr[2]));
    targetNode->setRotation(vector3df(0.0f,view_hpr[1],0.0f));

	float target_x, target_y, target_z;
    calcTargetPoint(&target_x, &target_y, &target_z,view_xyz[0],view_xyz[1],view_xyz[2],
                    view_ode_hpr[0], view_ode_hpr[1], view_ode_hpr[2]);
    camera->setTarget(vector3df(target_x, target_y, target_z));
}



// 視点の位置と方向からターゲット点を計算する．
// Calculate a target point with a gaze position and direction
void Vision::calcTargetPoint(float *target_x, float *target_y, float *target_z,
                            float x, float y, float z, float heading, float pitch, float roll)
{
    float r = degToRad(roll);
	float p = degToRad(pitch);
	float h = degToRad(heading);

	if (pitch <= -90)
    {
		*target_x = -sin(r)*cos(p) + x;
        *target_y =  sin(p) + y;
        *target_z = -cos(r)*cos(p) + z;
	 }
    else
    {
        *target_x = float( sin(h)*cos(r) - cos(h)*sin(p)*sin(r) + x);
        *target_y = float( - sin(r)*sin(h) - cos(h)*sin(p)*cos(r) + y);
        *target_z = float( -cos(h)*cos(p) + z);
    }
}

// Set a top, a bottom, a fov, and a projection matrix of the camera
void Vision::setupCamera(int width, int height)
{
    const float vnear  = 0.1f;   // 0.1f  near view plane
    const float vfar   = 100.0f; // 100.0f far view plane
    float fov; // field of view
    const float k = 0.8f;     // view scale, 1 = +/- 45 degrees
    matrix4 matrix;

    float top, bottom;
    float k2 = float(height)/float(width);
    top = vnear*k*k2;
    bottom = -vnear*k*k2;
    fov = (float) 2.0 * atan2((float) 2.0*(top-bottom), (float) vnear);
    matrix.buildProjectionMatrixPerspectiveLH(fov,k2,vnear,vfar);

    cameraman->setPosition(vector3df(view_xyz[0],view_xyz[1],view_xyz[2]));
    targetNode->setRotation(vector3df(view_hpr[0],0.0f, 0.0f));
    targetNode->setRotation(vector3df(0.0f, 0.0f,view_hpr[2]));
    targetNode->setRotation(vector3df(0.0f, view_hpr[1],0.0f));
}


void Vision::wrapCameraAngles(float view_hpr[3])
{
    for (int i=0; i<3; i++)
    {
        while (view_hpr[i] >  180) view_hpr[i] -= 360;
        while (view_hpr[i] < -180) view_hpr[i] += 360;
    }
}

void Vision::calcUpVector(float *up_x, float *up_y, float *up_z,
                         float heading, float pitch, float roll)
{
    heading*= -1;
    pitch *= -1;

	float r = degToRad(roll);
	float p = degToRad(pitch);
	float h = degToRad(heading);

    *up_x = float ( -cos(h)*sin(r) + sin(h)*sin(p)*sin(r));
    *up_y = float ( cos(p)*cos(r));
    *up_z = float ( sin(h)*sin(r)  + cos(h)*sin(p)*sin(r));

}


// Set the camera position, the target, and the upvector
void Vision::setCamera(float x, float y, float z)
{
    static float target_x = 0, target_y = 0, target_z = 0;
    static float up_x = 0, up_y = 1, up_z = 0;

    calcTargetPoint(&target_x, &target_y, &target_z, x, y, z,
                    view_ode_hpr[0], view_ode_hpr[1], view_ode_hpr[2]);
    camera->setTarget(vector3df(target_x, target_y, target_z));

    calcUpVector(&up_x, &up_y, &up_z,
                 view_ode_hpr[0], view_ode_hpr[1], view_ode_hpr[2]);
    camera->setUpVector(vector3df(up_x,up_y,up_z));
    camera->setPosition(vector3df(x,y,z));
}

void Vision::setViewpoint(float xyz[3], float hpr[3])
{
	assert(!(sim->getCurrentState() < 1) && "diSetViewpoint() called before simulation started");
	
    if (xyz)
    {
        view_xyz[0] =  xyz[1];
        view_xyz[1] =  xyz[2];
        view_xyz[2] = -xyz[0];
    }

    if (hpr)
    {
        //view_hpr[0] = -hpr[1];
        //view_hpr[1] = -hpr[2];
        //view_hpr[2] =  hpr[0];
        view_hpr[0] =  -hpr[1];
        view_hpr[1] =  -hpr[2];
        view_hpr[2] =   hpr[0];
        wrapCameraAngles(view_hpr);
    }

    // ＯＤＥの視点と視線方向を保存
    if (xyz)
    {
        view_ode_xyz[0] =  xyz[0];
        view_ode_xyz[1] =  xyz[1];
        view_ode_xyz[2] =  xyz[2];
    }

    if (hpr)
    {
        view_ode_hpr[0] =  hpr[0];
        view_ode_hpr[1] =  hpr[1];
        view_ode_hpr[2] =  hpr[2];
        wrapCameraAngles(view_ode_hpr);
    }
}

void Vision::getViewpoint(float xyz[3], float hpr[3])
{
    assert(!(sim->getCurrentState() < 1) && "dsGetViewpoint() called before simulation started");
	
    if (xyz)
    {
        xyz[1] =  view_xyz[0];
        xyz[2] =  view_xyz[1];
        xyz[0] = -view_xyz[2];
    }
    if (hpr)
    {
        hpr[1] = -view_hpr[0];
        hpr[2] = -view_hpr[1];
        hpr[0] =  view_hpr[2];
    }
}

// xyz, hprはODEの並び
extern "C" void diSetViewpoint(float xyz[3], float hpr[3])
{
    sim->vision->setViewpoint(xyz, hpr);	
}

// xyz, hprはODEの並び
extern "C" void diGetViewpoint(float xyz[3], float hpr[3])
{
	sim->vision->getViewpoint(xyz, hpr);
}

