/*
================================================================================================

Description	:	Performance scene rendering.
Author		:	J.M.P. van Waveren
Date		:	01/12/2017
Language	:	C99
Format		:	Real tabs with the tab size equal to 4 spaces.
Copyright	:	Copyright (c) 2016 Oculus VR, LLC. All Rights reserved.
			:	Portions copyright (c) 2016 The Brenwill Workshop Ltd. All Rights reserved.


LICENSE
=======

Copyright (c) 2016 Oculus VR, LLC.
Portions of macOS, iOS, functionality copyright (c) 2016 The Brenwill Workshop Ltd.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.


INTERFACE
=========

ksPerfScene

static void ksPerfScene_Create( ksGpuContext * context, ksPerfScene * scene, ksSceneSettings * settings, ksGpuRenderPass * renderPass );
static void ksPerfScene_Destroy( ksGpuContext * context, ksPerfScene * scene );

static void ksPerfScene_Simulate( ksPerfScene * scene, ksViewState * viewState, const ksNanoseconds time );
static void ksPerfScene_UpdateBuffers( ksGpuCommandBuffer * commandBuffer, ksPerfScene * scene, const ksViewState * viewState, const int eye );
static void ksPerfScene_Render( ksGpuCommandBuffer * commandBuffer, ksPerfScene * scene, const ksViewState * viewState );

================================================================================================================================
*/

typedef struct
{
	// assets
	ksGpuGeometry			geometry[MAX_SCENE_TRIANGLE_LEVELS];
	ksGpuGraphicsProgram	program[MAX_SCENE_FRAGMENT_LEVELS];
	ksGpuGraphicsPipeline	pipelines[MAX_SCENE_TRIANGLE_LEVELS][MAX_SCENE_FRAGMENT_LEVELS];
	ksGpuBuffer				sceneMatrices;
	ksGpuTexture			diffuseTexture;
	ksGpuTexture			specularTexture;
	ksGpuTexture			normalTexture;
	ksSceneSettings			settings;
	ksSceneSettings *		newSettings;
	// simulation state
	float					bigRotationX;
	float					bigRotationY;
	float					smallRotationX;
	float					smallRotationY;
	ksMatrix4x4f *			modelMatrix;
} ksPerfScene;

enum
{
	PROGRAM_UNIFORM_MODEL_MATRIX,
	PROGRAM_UNIFORM_SCENE_MATRICES,
	PROGRAM_TEXTURE_0,
	PROGRAM_TEXTURE_1,
	PROGRAM_TEXTURE_2
};

#if GRAPHICS_API_OPENGL == 1 || GRAPHICS_API_OPENGL_ES == 1

static ksGpuProgramParm flatShadedProgramParms[] =
{
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,	KS_GPU_PROGRAM_PARM_TYPE_PUSH_CONSTANT_FLOAT_MATRIX4X4,	KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_MODEL_MATRIX,		"ModelMatrix",		0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,	KS_GPU_PROGRAM_PARM_TYPE_BUFFER_UNIFORM,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_SCENE_MATRICES,		"SceneMatrices",	0 }
};

static const char flatShadedVertexProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"uniform mat4 ModelMatrix;\n"
	"uniform SceneMatrices\n"
	"{\n"
	"	mat4 ViewMatrix;\n"
	"	mat4 ProjectionMatrix;\n"
	"};\n"
	"in vec3 vertexPosition;\n"
	"in vec3 vertexNormal;\n"
	"out vec3 fragmentEyeDir;\n"
	"out vec3 fragmentNormal;\n"
	"out gl_PerVertex { vec4 gl_Position; };\n"
	"vec3 multiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[1].x * v.y + m[2].x * v.z,\n"
	"		m[0].y * v.x + m[1].y * v.y + m[2].y * v.z,\n"
	"		m[0].z * v.x + m[1].z * v.y + m[2].z * v.z );\n"
	"}\n"
	"vec3 transposeMultiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[0].y * v.y + m[0].z * v.z,\n"
	"		m[1].x * v.x + m[1].y * v.y + m[1].z * v.z,\n"
	"		m[2].x * v.x + m[2].y * v.y + m[2].z * v.z );\n"
	"}\n"
	"void main( void )\n"
	"{\n"
	"	vec4 vertexWorldPos = ModelMatrix * vec4( vertexPosition, 1.0 );\n"
	"	vec3 eyeWorldPos = transposeMultiply3x3( ViewMatrix, -vec3( ViewMatrix[3] ) );\n"
	"	gl_Position = ProjectionMatrix * ( ViewMatrix * vertexWorldPos );\n"
	"	fragmentEyeDir = eyeWorldPos - vec3( vertexWorldPos );\n"
	"	fragmentNormal = multiply3x3( ModelMatrix, vertexNormal );\n"
	"}\n";

static const char flatShadedMultiViewVertexProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"#define NUM_VIEWS 2\n"
	"#define VIEW_ID gl_ViewID_OVR\n"
	"#extension GL_OVR_multiview2 : require\n"
	"layout( num_views = NUM_VIEWS ) in;\n"
	"\n"
	"uniform mat4 ModelMatrix;\n"
	"uniform SceneMatrices\n"
	"{\n"
	"	mat4 ViewMatrix[NUM_VIEWS];\n"
	"	mat4 ProjectionMatrix[NUM_VIEWS];\n"
	"} ub;\n"
	"in vec3 vertexPosition;\n"
	"in vec3 vertexNormal;\n"
	"out vec3 fragmentEyeDir;\n"
	"out vec3 fragmentNormal;\n"
	"vec3 multiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[1].x * v.y + m[2].x * v.z,\n"
	"		m[0].y * v.x + m[1].y * v.y + m[2].y * v.z,\n"
	"		m[0].z * v.x + m[1].z * v.y + m[2].z * v.z );\n"
	"}\n"
	"vec3 transposeMultiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[0].y * v.y + m[0].z * v.z,\n"
	"		m[1].x * v.x + m[1].y * v.y + m[1].z * v.z,\n"
	"		m[2].x * v.x + m[2].y * v.y + m[2].z * v.z );\n"
	"}\n"
	"void main( void )\n"
	"{\n"
	"	vec4 vertexWorldPos = ModelMatrix * vec4( vertexPosition, 1.0 );\n"
	"	vec3 eyeWorldPos = transposeMultiply3x3( ub.ViewMatrix[VIEW_ID], -vec3( ub.ViewMatrix[VIEW_ID][3] ) );\n"
	"	gl_Position = ub.ProjectionMatrix[VIEW_ID] * ( ub.ViewMatrix[VIEW_ID] * vertexWorldPos );\n"
	"	fragmentEyeDir = eyeWorldPos - vec3( vertexWorldPos );\n"
	"	fragmentNormal = multiply3x3( ModelMatrix, vertexNormal );\n"
	"}\n";

static const char flatShadedFragmentProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"in lowp vec3 fragmentEyeDir;\n"
	"in lowp vec3 fragmentNormal;\n"
	"out lowp vec4 outColor;\n"
	"void main()\n"
	"{\n"
	"	lowp vec3 diffuseMap = vec3( 0.2, 0.2, 1.0 );\n"
	"	lowp vec3 specularMap = vec3( 0.5, 0.5, 0.5 );\n"
	"	lowp float specularPower = 10.0;\n"
	"	lowp vec3 eyeDir = normalize( fragmentEyeDir );\n"
	"	lowp vec3 normal = normalize( fragmentNormal );\n"
	"\n"
	"	lowp vec3 lightDir = normalize( vec3( -1.0, 1.0, 1.0 ) );\n"
	"	lowp vec3 lightReflection = normalize( 2.0 * dot( lightDir, normal ) * normal - lightDir );\n"
	"	lowp vec3 lightDiffuse = diffuseMap * ( max( dot( normal, lightDir ), 0.0 ) * 0.5 + 0.5 );\n"
	"	lowp vec3 lightSpecular = specularMap * pow( max( dot( lightReflection, eyeDir ), 0.0 ), specularPower );\n"
	"\n"
	"	outColor.xyz = lightDiffuse + lightSpecular;\n"
	"	outColor.w = 1.0;\n"
	"}\n";

static ksGpuProgramParm normalMappedProgramParms[] =
{
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,		KS_GPU_PROGRAM_PARM_TYPE_PUSH_CONSTANT_FLOAT_MATRIX4X4,	KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_MODEL_MATRIX,		"ModelMatrix",		0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,		KS_GPU_PROGRAM_PARM_TYPE_BUFFER_UNIFORM,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_SCENE_MATRICES,		"SceneMatrices",	0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_0,					"Texture0",			0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_1,					"Texture1",			1 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_2,					"Texture2",			2 }
};

static const char normalMappedVertexProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"uniform mat4 ModelMatrix;\n"
	"uniform SceneMatrices\n"
	"{\n"
	"	mat4 ViewMatrix;\n"
	"	mat4 ProjectionMatrix;\n"
	"};\n"
	"in vec3 vertexPosition;\n"
	"in vec3 vertexNormal;\n"
	"in vec3 vertexTangent;\n"
	"in vec3 vertexBinormal;\n"
	"in vec2 vertexUv0;\n"
	"out vec3 fragmentEyeDir;\n"
	"out vec3 fragmentNormal;\n"
	"out vec3 fragmentTangent;\n"
	"out vec3 fragmentBinormal;\n"
	"out vec2 fragmentUv0;\n"
	"out gl_PerVertex { vec4 gl_Position; };\n"
	"vec3 multiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[1].x * v.y + m[2].x * v.z,\n"
	"		m[0].y * v.x + m[1].y * v.y + m[2].y * v.z,\n"
	"		m[0].z * v.x + m[1].z * v.y + m[2].z * v.z );\n"
	"}\n"
	"vec3 transposeMultiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[0].y * v.y + m[0].z * v.z,\n"
	"		m[1].x * v.x + m[1].y * v.y + m[1].z * v.z,\n"
	"		m[2].x * v.x + m[2].y * v.y + m[2].z * v.z );\n"
	"}\n"
	"void main( void )\n"
	"{\n"
	"	vec4 vertexWorldPos = ModelMatrix * vec4( vertexPosition, 1.0 );\n"
	"	vec3 eyeWorldPos = transposeMultiply3x3( ViewMatrix, -vec3( ViewMatrix[3] ) );\n"
	"	gl_Position = ProjectionMatrix * ( ViewMatrix * vertexWorldPos );\n"
	"	fragmentEyeDir = eyeWorldPos - vec3( vertexWorldPos );\n"
	"	fragmentNormal = multiply3x3( ModelMatrix, vertexNormal );\n"
	"	fragmentTangent = multiply3x3( ModelMatrix, vertexTangent );\n"
	"	fragmentBinormal = multiply3x3( ModelMatrix, vertexBinormal );\n"
	"	fragmentUv0 = vertexUv0;\n"
	"}\n";

static const char normalMappedMultiViewVertexProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"#define NUM_VIEWS 2\n"
	"#define VIEW_ID gl_ViewID_OVR\n"
	"#extension GL_OVR_multiview2 : require\n"
	"layout( num_views = NUM_VIEWS ) in;\n"
	"\n"
	"uniform mat4 ModelMatrix;\n"
	"uniform SceneMatrices\n"
	"{\n"
	"	mat4 ViewMatrix[NUM_VIEWS];\n"
	"	mat4 ProjectionMatrix[NUM_VIEWS];\n"
	"} ub;\n"
	"in vec3 vertexPosition;\n"
	"in vec3 vertexNormal;\n"
	"in vec3 vertexTangent;\n"
	"in vec3 vertexBinormal;\n"
	"in vec2 vertexUv0;\n"
	"out vec3 fragmentEyeDir;\n"
	"out vec3 fragmentNormal;\n"
	"out vec3 fragmentTangent;\n"
	"out vec3 fragmentBinormal;\n"
	"out vec2 fragmentUv0;\n"
	"vec3 multiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[1].x * v.y + m[2].x * v.z,\n"
	"		m[0].y * v.x + m[1].y * v.y + m[2].y * v.z,\n"
	"		m[0].z * v.x + m[1].z * v.y + m[2].z * v.z );\n"
	"}\n"
	"vec3 transposeMultiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[0].y * v.y + m[0].z * v.z,\n"
	"		m[1].x * v.x + m[1].y * v.y + m[1].z * v.z,\n"
	"		m[2].x * v.x + m[2].y * v.y + m[2].z * v.z );\n"
	"}\n"
	"void main( void )\n"
	"{\n"
	"	vec4 vertexWorldPos = ModelMatrix * vec4( vertexPosition, 1.0 );\n"
	"	vec3 eyeWorldPos = transposeMultiply3x3( ub.ViewMatrix[VIEW_ID], -vec3( ub.ViewMatrix[VIEW_ID][3] ) );\n"
	"	gl_Position = ub.ProjectionMatrix[VIEW_ID] * ( ub.ViewMatrix[VIEW_ID] * vertexWorldPos );\n"
	"	fragmentEyeDir = eyeWorldPos - vec3( vertexWorldPos );\n"
	"	fragmentNormal = multiply3x3( ModelMatrix, vertexNormal );\n"
	"	fragmentTangent = multiply3x3( ModelMatrix, vertexTangent );\n"
	"	fragmentBinormal = multiply3x3( ModelMatrix, vertexBinormal );\n"
	"	fragmentUv0 = vertexUv0;\n"
	"}\n";

static const char normalMapped100LightsFragmentProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"uniform sampler2D Texture0;\n"
	"uniform sampler2D Texture1;\n"
	"uniform sampler2D Texture2;\n"
	"in lowp vec3 fragmentEyeDir;\n"
	"in lowp vec3 fragmentNormal;\n"
	"in lowp vec3 fragmentTangent;\n"
	"in lowp vec3 fragmentBinormal;\n"
	"in lowp vec2 fragmentUv0;\n"
	"out lowp vec4 outColor;\n"
	"void main()\n"
	"{\n"
	"	lowp vec3 diffuseMap = texture( Texture0, fragmentUv0 ).xyz;\n"
	"	lowp vec3 specularMap = texture( Texture1, fragmentUv0 ).xyz * 2.0;\n"
	"	lowp vec3 normalMap = texture( Texture2, fragmentUv0 ).xyz * 2.0 - 1.0;\n"
	"	lowp float specularPower = 10.0;\n"
	"	lowp vec3 eyeDir = normalize( fragmentEyeDir );\n"
	"	lowp vec3 normal = normalize( normalMap.x * fragmentTangent + normalMap.y * fragmentBinormal + normalMap.z * fragmentNormal );\n"
	"\n"
	"	lowp vec3 color = vec3( 0 );\n"
	"	for ( int i = 0; i < 100; i++ )\n"
	"	{\n"
	"		lowp vec3 lightDir = normalize( vec3( -1.0, 1.0, 1.0 ) );\n"
	"		lowp vec3 lightReflection = normalize( 2.0 * dot( lightDir, normal ) * normal - lightDir );\n"
	"		lowp vec3 lightDiffuse = diffuseMap * ( max( dot( normal, lightDir ), 0.0 ) * 0.5 + 0.5 );\n"
	"		lowp vec3 lightSpecular = specularMap * pow( max( dot( lightReflection, eyeDir ), 0.0 ), specularPower );\n"
	"		color += ( lightDiffuse + lightSpecular ) * ( 1.0 / 100.0 );\n"
	"	}\n"
	"\n"
	"	outColor.xyz = color;\n"
	"	outColor.w = 1.0;\n"
	"}\n";

static const char normalMapped1000LightsFragmentProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"uniform sampler2D Texture0;\n"
	"uniform sampler2D Texture1;\n"
	"uniform sampler2D Texture2;\n"
	"in lowp vec3 fragmentEyeDir;\n"
	"in lowp vec3 fragmentNormal;\n"
	"in lowp vec3 fragmentTangent;\n"
	"in lowp vec3 fragmentBinormal;\n"
	"in lowp vec2 fragmentUv0;\n"
	"out lowp vec4 outColor;\n"
	"void main()\n"
	"{\n"
	"	lowp vec3 diffuseMap = texture( Texture0, fragmentUv0 ).xyz;\n"
	"	lowp vec3 specularMap = texture( Texture1, fragmentUv0 ).xyz * 2.0;\n"
	"	lowp vec3 normalMap = texture( Texture2, fragmentUv0 ).xyz * 2.0 - 1.0;\n"
	"	lowp float specularPower = 10.0;\n"
	"	lowp vec3 eyeDir = normalize( fragmentEyeDir );\n"
	"	lowp vec3 normal = normalize( normalMap.x * fragmentTangent + normalMap.y * fragmentBinormal + normalMap.z * fragmentNormal );\n"
	"\n"
	"	lowp vec3 color = vec3( 0 );\n"
	"	for ( int i = 0; i < 1000; i++ )\n"
	"	{\n"
	"		lowp vec3 lightDir = normalize( vec3( -1.0, 1.0, 1.0 ) );\n"
	"		lowp vec3 lightReflection = normalize( 2.0 * dot( lightDir, normal ) * normal - lightDir );\n"
	"		lowp vec3 lightDiffuse = diffuseMap * ( max( dot( normal, lightDir ), 0.0 ) * 0.5 + 0.5 );\n"
	"		lowp vec3 lightSpecular = specularMap * pow( max( dot( lightReflection, eyeDir ), 0.0 ), specularPower );\n"
	"		color += ( lightDiffuse + lightSpecular ) * ( 1.0 / 1000.0 );\n"
	"	}\n"
	"\n"
	"	outColor.xyz = color;\n"
	"	outColor.w = 1.0;\n"
	"}\n";

static const char normalMapped2000LightsFragmentProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"uniform sampler2D Texture0;\n"
	"uniform sampler2D Texture1;\n"
	"uniform sampler2D Texture2;\n"
	"in lowp vec3 fragmentEyeDir;\n"
	"in lowp vec3 fragmentNormal;\n"
	"in lowp vec3 fragmentTangent;\n"
	"in lowp vec3 fragmentBinormal;\n"
	"in lowp vec2 fragmentUv0;\n"
	"out lowp vec4 outColor;\n"
	"void main()\n"
	"{\n"
	"	lowp vec3 diffuseMap = texture( Texture0, fragmentUv0 ).xyz;\n"
	"	lowp vec3 specularMap = texture( Texture1, fragmentUv0 ).xyz * 2.0;\n"
	"	lowp vec3 normalMap = texture( Texture2, fragmentUv0 ).xyz * 2.0 - 1.0;\n"
	"	lowp float specularPower = 10.0;\n"
	"	lowp vec3 eyeDir = normalize( fragmentEyeDir );\n"
	"	lowp vec3 normal = normalize( normalMap.x * fragmentTangent + normalMap.y * fragmentBinormal + normalMap.z * fragmentNormal );\n"
	"\n"
	"	lowp vec3 color = vec3( 0 );\n"
	"	for ( int i = 0; i < 2000; i++ )\n"
	"	{\n"
	"		lowp vec3 lightDir = normalize( vec3( -1.0, 1.0, 1.0 ) );\n"
	"		lowp vec3 lightReflection = normalize( 2.0 * dot( lightDir, normal ) * normal - lightDir );\n"
	"		lowp vec3 lightDiffuse = diffuseMap * ( max( dot( normal, lightDir ), 0.0 ) * 0.5 + 0.5 );\n"
	"		lowp vec3 lightSpecular = specularMap * pow( max( dot( lightReflection, eyeDir ), 0.0 ), specularPower );\n"
	"		color += ( lightDiffuse + lightSpecular ) * ( 1.0 / 2000.0 );\n"
	"	}\n"
	"\n"
	"	outColor.xyz = color;\n"
	"	outColor.w = 1.0;\n"
	"}\n";

#elif GRAPHICS_API_VULKAN == 1

static ksGpuProgramParm flatShadedProgramParms[] =
{
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,	KS_GPU_PROGRAM_PARM_TYPE_PUSH_CONSTANT_FLOAT_MATRIX4X4,	KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_MODEL_MATRIX,		"ModelMatrix",		0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,	KS_GPU_PROGRAM_PARM_TYPE_BUFFER_UNIFORM,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_SCENE_MATRICES,		"SceneMatrices",	0 }
};

static const char flatShadedVertexProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"layout( std430, push_constant ) uniform PushConstants\n"
	"{\n"
	"	layout( offset =   0 ) mat4 ModelMatrix;\n"
	"} pc;\n"
	"layout( std140, binding = 0 ) uniform SceneMatrices\n"
	"{\n"
	"	layout( offset =   0 ) mat4 ViewMatrix;\n"
	"	layout( offset =  64 ) mat4 ProjectionMatrix;\n"
	"};\n"
	"layout( location = 0 ) in vec3 vertexPosition;\n"
	"layout( location = 1 ) in vec3 vertexNormal;\n"
	"layout( location = 0 ) out vec3 fragmentEyeDir;\n"
	"layout( location = 1 ) out vec3 fragmentNormal;\n"
	"out gl_PerVertex { vec4 gl_Position; };\n"
	"vec3 multiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[1].x * v.y + m[2].x * v.z,\n"
	"		m[0].y * v.x + m[1].y * v.y + m[2].y * v.z,\n"
	"		m[0].z * v.x + m[1].z * v.y + m[2].z * v.z );\n"
	"}\n"
	"vec3 transposeMultiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[0].y * v.y + m[0].z * v.z,\n"
	"		m[1].x * v.x + m[1].y * v.y + m[1].z * v.z,\n"
	"		m[2].x * v.x + m[2].y * v.y + m[2].z * v.z );\n"
	"}\n"
	"void main( void )\n"
	"{\n"
	"	vec4 vertexWorldPos = pc.ModelMatrix * vec4( vertexPosition, 1.0 );\n"
	"	vec3 eyeWorldPos = transposeMultiply3x3( ViewMatrix, -vec3( ViewMatrix[3] ) );\n"
	"	gl_Position = ProjectionMatrix * ( ViewMatrix * vertexWorldPos );\n"
	"	fragmentEyeDir = eyeWorldPos - vec3( vertexWorldPos );\n"
	"	fragmentNormal = multiply3x3( pc.ModelMatrix, vertexNormal );\n"
	"}\n";

static const unsigned int flatShadedVertexProgramSPIRV[] =
{
	// SPIRV99.947 15-Feb-2016
	0x07230203,0x00010000,0x00080001,0x000000cb,0x00000000,0x00020011,0x00000001,0x0006000b,
	0x00000001,0x4c534c47,0x6474732e,0x3035342e,0x00000000,0x0003000e,0x00000000,0x00000001,
	0x000a000f,0x00000000,0x00000004,0x6e69616d,0x00000000,0x00000093,0x000000b0,0x000000bb,
	0x000000c3,0x000000c4,0x00030003,0x00000001,0x00000136,0x00040005,0x00000004,0x6e69616d,
	0x00000000,0x00080005,0x0000000f,0x746c756d,0x796c7069,0x28337833,0x3434666d,0x3366763b,
	0x0000003b,0x00030005,0x0000000d,0x0000006d,0x00030005,0x0000000e,0x00000076,0x000a0005,
	0x00000013,0x6e617274,0x736f7073,0x6c754d65,0x6c706974,0x33783379,0x34666d28,0x66763b34,
	0x00003b33,0x00030005,0x00000011,0x0000006d,0x00030005,0x00000012,0x00000076,0x00060005,
	0x0000008b,0x74726576,0x6f577865,0x50646c72,0x0000736f,0x00060005,0x0000008c,0x68737550,
	0x736e6f43,0x746e6174,0x00000073,0x00060006,0x0000008c,0x00000000,0x65646f4d,0x74614d6c,
	0x00786972,0x00030005,0x0000008e,0x00006370,0x00060005,0x00000093,0x74726576,0x6f507865,
	0x69746973,0x00006e6f,0x00050005,0x0000009b,0x57657965,0x646c726f,0x00736f50,0x00060005,
	0x0000009c,0x6e656353,0x74614d65,0x65636972,0x00000073,0x00060006,0x0000009c,0x00000000,
	0x77656956,0x7274614d,0x00007869,0x00080006,0x0000009c,0x00000001,0x6a6f7250,0x69746365,
	0x614d6e6f,0x78697274,0x00000000,0x00030005,0x0000009e,0x00006275,0x00040005,0x000000a8,
	0x61726170,0x0000006d,0x00040005,0x000000ac,0x61726170,0x0000006d,0x00060005,0x000000ae,
	0x505f6c67,0x65567265,0x78657472,0x00000000,0x00060006,0x000000ae,0x00000000,0x505f6c67,
	0x7469736f,0x006e6f69,0x00070006,0x000000ae,0x00000001,0x505f6c67,0x746e696f,0x657a6953,
	0x00000000,0x00030005,0x000000b0,0x00000000,0x00060005,0x000000bb,0x67617266,0x746e656d,
	0x44657945,0x00007269,0x00060005,0x000000c3,0x67617266,0x746e656d,0x6d726f4e,0x00006c61,
	0x00060005,0x000000c4,0x74726576,0x6f4e7865,0x6c616d72,0x00000000,0x00040005,0x000000c5,
	0x61726170,0x0000006d,0x00040005,0x000000c8,0x61726170,0x0000006d,0x00040048,0x0000008c,
	0x00000000,0x00000005,0x00050048,0x0000008c,0x00000000,0x00000023,0x00000000,0x00050048,
	0x0000008c,0x00000000,0x00000007,0x00000010,0x00030047,0x0000008c,0x00000002,0x00040047,
	0x00000093,0x0000001e,0x00000000,0x00040048,0x0000009c,0x00000000,0x00000005,0x00050048,
	0x0000009c,0x00000000,0x00000023,0x00000000,0x00050048,0x0000009c,0x00000000,0x00000007,
	0x00000010,0x00040048,0x0000009c,0x00000001,0x00000005,0x00050048,0x0000009c,0x00000001,
	0x00000023,0x00000040,0x00050048,0x0000009c,0x00000001,0x00000007,0x00000010,0x00030047,
	0x0000009c,0x00000002,0x00040047,0x0000009e,0x00000022,0x00000000,0x00040047,0x0000009e,
	0x00000021,0x00000000,0x00050048,0x000000ae,0x00000000,0x0000000b,0x00000000,0x00050048,
	0x000000ae,0x00000001,0x0000000b,0x00000001,0x00030047,0x000000ae,0x00000002,0x00040047,
	0x000000bb,0x0000001e,0x00000000,0x00040047,0x000000c3,0x0000001e,0x00000001,0x00040047,
	0x000000c4,0x0000001e,0x00000001,0x00020013,0x00000002,0x00030021,0x00000003,0x00000002,
	0x00030016,0x00000006,0x00000020,0x00040017,0x00000007,0x00000006,0x00000004,0x00040018,
	0x00000008,0x00000007,0x00000004,0x00040020,0x00000009,0x00000007,0x00000008,0x00040017,
	0x0000000a,0x00000006,0x00000003,0x00040020,0x0000000b,0x00000007,0x0000000a,0x00050021,
	0x0000000c,0x0000000a,0x00000009,0x0000000b,0x00040015,0x00000015,0x00000020,0x00000001,
	0x0004002b,0x00000015,0x00000016,0x00000000,0x00040015,0x00000017,0x00000020,0x00000000,
	0x0004002b,0x00000017,0x00000018,0x00000000,0x00040020,0x00000019,0x00000007,0x00000006,
	0x0004002b,0x00000015,0x0000001f,0x00000001,0x0004002b,0x00000017,0x00000022,0x00000001,
	0x0004002b,0x00000015,0x00000027,0x00000002,0x0004002b,0x00000017,0x0000002a,0x00000002,
	0x00040020,0x0000008a,0x00000007,0x00000007,0x0003001e,0x0000008c,0x00000008,0x00040020,
	0x0000008d,0x00000009,0x0000008c,0x0004003b,0x0000008d,0x0000008e,0x00000009,0x00040020,
	0x0000008f,0x00000009,0x00000008,0x00040020,0x00000092,0x00000001,0x0000000a,0x0004003b,
	0x00000092,0x00000093,0x00000001,0x0004002b,0x00000006,0x00000095,0x3f800000,0x0004001e,
	0x0000009c,0x00000008,0x00000008,0x00040020,0x0000009d,0x00000002,0x0000009c,0x0004003b,
	0x0000009d,0x0000009e,0x00000002,0x0004002b,0x00000015,0x0000009f,0x00000003,0x00040020,
	0x000000a0,0x00000002,0x00000007,0x00040020,0x000000a9,0x00000002,0x00000008,0x0004001e,
	0x000000ae,0x00000007,0x00000006,0x00040020,0x000000af,0x00000003,0x000000ae,0x0004003b,
	0x000000af,0x000000b0,0x00000003,0x00040020,0x000000b8,0x00000003,0x00000007,0x00040020,
	0x000000ba,0x00000003,0x0000000a,0x0004003b,0x000000ba,0x000000bb,0x00000003,0x0004003b,
	0x000000ba,0x000000c3,0x00000003,0x0004003b,0x00000092,0x000000c4,0x00000001,0x00050036,
	0x00000002,0x00000004,0x00000000,0x00000003,0x000200f8,0x00000005,0x0004003b,0x0000008a,
	0x0000008b,0x00000007,0x0004003b,0x0000000b,0x0000009b,0x00000007,0x0004003b,0x00000009,
	0x000000a8,0x00000007,0x0004003b,0x0000000b,0x000000ac,0x00000007,0x0004003b,0x00000009,
	0x000000c5,0x00000007,0x0004003b,0x0000000b,0x000000c8,0x00000007,0x00050041,0x0000008f,
	0x00000090,0x0000008e,0x00000016,0x0004003d,0x00000008,0x00000091,0x00000090,0x0004003d,
	0x0000000a,0x00000094,0x00000093,0x00050051,0x00000006,0x00000096,0x00000094,0x00000000,
	0x00050051,0x00000006,0x00000097,0x00000094,0x00000001,0x00050051,0x00000006,0x00000098,
	0x00000094,0x00000002,0x00070050,0x00000007,0x00000099,0x00000096,0x00000097,0x00000098,
	0x00000095,0x00050091,0x00000007,0x0000009a,0x00000091,0x00000099,0x0003003e,0x0000008b,
	0x0000009a,0x00060041,0x000000a0,0x000000a1,0x0000009e,0x00000016,0x0000009f,0x0004003d,
	0x00000007,0x000000a2,0x000000a1,0x00050051,0x00000006,0x000000a3,0x000000a2,0x00000000,
	0x00050051,0x00000006,0x000000a4,0x000000a2,0x00000001,0x00050051,0x00000006,0x000000a5,
	0x000000a2,0x00000002,0x00060050,0x0000000a,0x000000a6,0x000000a3,0x000000a4,0x000000a5,
	0x0004007f,0x0000000a,0x000000a7,0x000000a6,0x00050041,0x000000a9,0x000000aa,0x0000009e,
	0x00000016,0x0004003d,0x00000008,0x000000ab,0x000000aa,0x0003003e,0x000000a8,0x000000ab,
	0x0003003e,0x000000ac,0x000000a7,0x00060039,0x0000000a,0x000000ad,0x00000013,0x000000a8,
	0x000000ac,0x0003003e,0x0000009b,0x000000ad,0x00050041,0x000000a9,0x000000b1,0x0000009e,
	0x0000001f,0x0004003d,0x00000008,0x000000b2,0x000000b1,0x00050041,0x000000a9,0x000000b3,
	0x0000009e,0x00000016,0x0004003d,0x00000008,0x000000b4,0x000000b3,0x0004003d,0x00000007,
	0x000000b5,0x0000008b,0x00050091,0x00000007,0x000000b6,0x000000b4,0x000000b5,0x00050091,
	0x00000007,0x000000b7,0x000000b2,0x000000b6,0x00050041,0x000000b8,0x000000b9,0x000000b0,
	0x00000016,0x0003003e,0x000000b9,0x000000b7,0x0004003d,0x0000000a,0x000000bc,0x0000009b,
	0x0004003d,0x00000007,0x000000bd,0x0000008b,0x00050051,0x00000006,0x000000be,0x000000bd,
	0x00000000,0x00050051,0x00000006,0x000000bf,0x000000bd,0x00000001,0x00050051,0x00000006,
	0x000000c0,0x000000bd,0x00000002,0x00060050,0x0000000a,0x000000c1,0x000000be,0x000000bf,
	0x000000c0,0x00050083,0x0000000a,0x000000c2,0x000000bc,0x000000c1,0x0003003e,0x000000bb,
	0x000000c2,0x00050041,0x0000008f,0x000000c6,0x0000008e,0x00000016,0x0004003d,0x00000008,
	0x000000c7,0x000000c6,0x0003003e,0x000000c5,0x000000c7,0x0004003d,0x0000000a,0x000000c9,
	0x000000c4,0x0003003e,0x000000c8,0x000000c9,0x00060039,0x0000000a,0x000000ca,0x0000000f,
	0x000000c5,0x000000c8,0x0003003e,0x000000c3,0x000000ca,0x000100fd,0x00010038,0x00050036,
	0x0000000a,0x0000000f,0x00000000,0x0000000c,0x00030037,0x00000009,0x0000000d,0x00030037,
	0x0000000b,0x0000000e,0x000200f8,0x00000010,0x00060041,0x00000019,0x0000001a,0x0000000d,
	0x00000016,0x00000018,0x0004003d,0x00000006,0x0000001b,0x0000001a,0x00050041,0x00000019,
	0x0000001c,0x0000000e,0x00000018,0x0004003d,0x00000006,0x0000001d,0x0000001c,0x00050085,
	0x00000006,0x0000001e,0x0000001b,0x0000001d,0x00060041,0x00000019,0x00000020,0x0000000d,
	0x0000001f,0x00000018,0x0004003d,0x00000006,0x00000021,0x00000020,0x00050041,0x00000019,
	0x00000023,0x0000000e,0x00000022,0x0004003d,0x00000006,0x00000024,0x00000023,0x00050085,
	0x00000006,0x00000025,0x00000021,0x00000024,0x00050081,0x00000006,0x00000026,0x0000001e,
	0x00000025,0x00060041,0x00000019,0x00000028,0x0000000d,0x00000027,0x00000018,0x0004003d,
	0x00000006,0x00000029,0x00000028,0x00050041,0x00000019,0x0000002b,0x0000000e,0x0000002a,
	0x0004003d,0x00000006,0x0000002c,0x0000002b,0x00050085,0x00000006,0x0000002d,0x00000029,
	0x0000002c,0x00050081,0x00000006,0x0000002e,0x00000026,0x0000002d,0x00060041,0x00000019,
	0x0000002f,0x0000000d,0x00000016,0x00000022,0x0004003d,0x00000006,0x00000030,0x0000002f,
	0x00050041,0x00000019,0x00000031,0x0000000e,0x00000018,0x0004003d,0x00000006,0x00000032,
	0x00000031,0x00050085,0x00000006,0x00000033,0x00000030,0x00000032,0x00060041,0x00000019,
	0x00000034,0x0000000d,0x0000001f,0x00000022,0x0004003d,0x00000006,0x00000035,0x00000034,
	0x00050041,0x00000019,0x00000036,0x0000000e,0x00000022,0x0004003d,0x00000006,0x00000037,
	0x00000036,0x00050085,0x00000006,0x00000038,0x00000035,0x00000037,0x00050081,0x00000006,
	0x00000039,0x00000033,0x00000038,0x00060041,0x00000019,0x0000003a,0x0000000d,0x00000027,
	0x00000022,0x0004003d,0x00000006,0x0000003b,0x0000003a,0x00050041,0x00000019,0x0000003c,
	0x0000000e,0x0000002a,0x0004003d,0x00000006,0x0000003d,0x0000003c,0x00050085,0x00000006,
	0x0000003e,0x0000003b,0x0000003d,0x00050081,0x00000006,0x0000003f,0x00000039,0x0000003e,
	0x00060041,0x00000019,0x00000040,0x0000000d,0x00000016,0x0000002a,0x0004003d,0x00000006,
	0x00000041,0x00000040,0x00050041,0x00000019,0x00000042,0x0000000e,0x00000018,0x0004003d,
	0x00000006,0x00000043,0x00000042,0x00050085,0x00000006,0x00000044,0x00000041,0x00000043,
	0x00060041,0x00000019,0x00000045,0x0000000d,0x0000001f,0x0000002a,0x0004003d,0x00000006,
	0x00000046,0x00000045,0x00050041,0x00000019,0x00000047,0x0000000e,0x00000022,0x0004003d,
	0x00000006,0x00000048,0x00000047,0x00050085,0x00000006,0x00000049,0x00000046,0x00000048,
	0x00050081,0x00000006,0x0000004a,0x00000044,0x00000049,0x00060041,0x00000019,0x0000004b,
	0x0000000d,0x00000027,0x0000002a,0x0004003d,0x00000006,0x0000004c,0x0000004b,0x00050041,
	0x00000019,0x0000004d,0x0000000e,0x0000002a,0x0004003d,0x00000006,0x0000004e,0x0000004d,
	0x00050085,0x00000006,0x0000004f,0x0000004c,0x0000004e,0x00050081,0x00000006,0x00000050,
	0x0000004a,0x0000004f,0x00060050,0x0000000a,0x00000051,0x0000002e,0x0000003f,0x00000050,
	0x000200fe,0x00000051,0x00010038,0x00050036,0x0000000a,0x00000013,0x00000000,0x0000000c,
	0x00030037,0x00000009,0x00000011,0x00030037,0x0000000b,0x00000012,0x000200f8,0x00000014,
	0x00060041,0x00000019,0x00000054,0x00000011,0x00000016,0x00000018,0x0004003d,0x00000006,
	0x00000055,0x00000054,0x00050041,0x00000019,0x00000056,0x00000012,0x00000018,0x0004003d,
	0x00000006,0x00000057,0x00000056,0x00050085,0x00000006,0x00000058,0x00000055,0x00000057,
	0x00060041,0x00000019,0x00000059,0x00000011,0x00000016,0x00000022,0x0004003d,0x00000006,
	0x0000005a,0x00000059,0x00050041,0x00000019,0x0000005b,0x00000012,0x00000022,0x0004003d,
	0x00000006,0x0000005c,0x0000005b,0x00050085,0x00000006,0x0000005d,0x0000005a,0x0000005c,
	0x00050081,0x00000006,0x0000005e,0x00000058,0x0000005d,0x00060041,0x00000019,0x0000005f,
	0x00000011,0x00000016,0x0000002a,0x0004003d,0x00000006,0x00000060,0x0000005f,0x00050041,
	0x00000019,0x00000061,0x00000012,0x0000002a,0x0004003d,0x00000006,0x00000062,0x00000061,
	0x00050085,0x00000006,0x00000063,0x00000060,0x00000062,0x00050081,0x00000006,0x00000064,
	0x0000005e,0x00000063,0x00060041,0x00000019,0x00000065,0x00000011,0x0000001f,0x00000018,
	0x0004003d,0x00000006,0x00000066,0x00000065,0x00050041,0x00000019,0x00000067,0x00000012,
	0x00000018,0x0004003d,0x00000006,0x00000068,0x00000067,0x00050085,0x00000006,0x00000069,
	0x00000066,0x00000068,0x00060041,0x00000019,0x0000006a,0x00000011,0x0000001f,0x00000022,
	0x0004003d,0x00000006,0x0000006b,0x0000006a,0x00050041,0x00000019,0x0000006c,0x00000012,
	0x00000022,0x0004003d,0x00000006,0x0000006d,0x0000006c,0x00050085,0x00000006,0x0000006e,
	0x0000006b,0x0000006d,0x00050081,0x00000006,0x0000006f,0x00000069,0x0000006e,0x00060041,
	0x00000019,0x00000070,0x00000011,0x0000001f,0x0000002a,0x0004003d,0x00000006,0x00000071,
	0x00000070,0x00050041,0x00000019,0x00000072,0x00000012,0x0000002a,0x0004003d,0x00000006,
	0x00000073,0x00000072,0x00050085,0x00000006,0x00000074,0x00000071,0x00000073,0x00050081,
	0x00000006,0x00000075,0x0000006f,0x00000074,0x00060041,0x00000019,0x00000076,0x00000011,
	0x00000027,0x00000018,0x0004003d,0x00000006,0x00000077,0x00000076,0x00050041,0x00000019,
	0x00000078,0x00000012,0x00000018,0x0004003d,0x00000006,0x00000079,0x00000078,0x00050085,
	0x00000006,0x0000007a,0x00000077,0x00000079,0x00060041,0x00000019,0x0000007b,0x00000011,
	0x00000027,0x00000022,0x0004003d,0x00000006,0x0000007c,0x0000007b,0x00050041,0x00000019,
	0x0000007d,0x00000012,0x00000022,0x0004003d,0x00000006,0x0000007e,0x0000007d,0x00050085,
	0x00000006,0x0000007f,0x0000007c,0x0000007e,0x00050081,0x00000006,0x00000080,0x0000007a,
	0x0000007f,0x00060041,0x00000019,0x00000081,0x00000011,0x00000027,0x0000002a,0x0004003d,
	0x00000006,0x00000082,0x00000081,0x00050041,0x00000019,0x00000083,0x00000012,0x0000002a,
	0x0004003d,0x00000006,0x00000084,0x00000083,0x00050085,0x00000006,0x00000085,0x00000082,
	0x00000084,0x00050081,0x00000006,0x00000086,0x00000080,0x00000085,0x00060050,0x0000000a,
	0x00000087,0x00000064,0x00000075,0x00000086,0x000200fe,0x00000087,0x00010038
};

static const char flatShadedMultiViewVertexProgramGLSL[] =
	"";

static const unsigned int flatShadedMultiViewVertexProgramSPIRV[] =
{
	0x00000000
};

static const char flatShadedFragmentProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"layout( location = 0 ) in lowp vec3 fragmentEyeDir;\n"
	"layout( location = 1 ) in lowp vec3 fragmentNormal;\n"
	"layout( location = 0 ) out lowp vec4 outColor;\n"
	"void main()\n"
	"{\n"
	"	lowp vec3 diffuseMap = vec3( 0.2, 0.2, 1.0 );\n"
	"	lowp vec3 specularMap = vec3( 0.5, 0.5, 0.5 );\n"
	"	lowp float specularPower = 10.0;\n"
	"	lowp vec3 eyeDir = normalize( fragmentEyeDir );\n"
	"	lowp vec3 normal = normalize( fragmentNormal );\n"
	"\n"
	"	lowp vec3 lightDir = normalize( vec3( -1.0, 1.0, 1.0 ) );\n"
	"	lowp vec3 lightReflection = normalize( 2.0 * dot( lightDir, normal ) * normal - lightDir );\n"
	"	lowp vec3 lightDiffuse = diffuseMap * ( max( dot( normal, lightDir ), 0.0 ) * 0.5 + 0.5 );\n"
	"	lowp vec3 lightSpecular = specularMap * pow( max( dot( lightReflection, eyeDir ), 0.0 ), specularPower );\n"
	"\n"
	"	outColor.xyz = lightDiffuse + lightSpecular;\n"
	"	outColor.w = 1.0;\n"
	"}\n";

static const unsigned int flatShadedFragmentProgramSPIRV[] =
{
	// SPIRV99.947 15-Feb-2016
	0x07230203,0x00010000,0x00080001,0x0000004a,0x00000000,0x00020011,0x00000001,0x0006000b,
	0x00000001,0x4c534c47,0x6474732e,0x3035342e,0x00000000,0x0003000e,0x00000000,0x00000001,
	0x0008000f,0x00000004,0x00000004,0x6e69616d,0x00000000,0x00000015,0x00000019,0x00000040,
	0x00030010,0x00000004,0x00000007,0x00030003,0x00000001,0x00000136,0x00040005,0x00000004,
	0x6e69616d,0x00000000,0x00050005,0x00000009,0x66666964,0x4d657375,0x00007061,0x00050005,
	0x0000000d,0x63657073,0x72616c75,0x0070614d,0x00060005,0x00000011,0x63657073,0x72616c75,
	0x65776f50,0x00000072,0x00040005,0x00000013,0x44657965,0x00007269,0x00060005,0x00000015,
	0x67617266,0x746e656d,0x44657945,0x00007269,0x00040005,0x00000018,0x6d726f6e,0x00006c61,
	0x00060005,0x00000019,0x67617266,0x746e656d,0x6d726f4e,0x00006c61,0x00050005,0x0000001c,
	0x6867696c,0x72694474,0x00000000,0x00060005,0x00000020,0x6867696c,0x66655274,0x7463656c,
	0x006e6f69,0x00060005,0x0000002b,0x6867696c,0x66694474,0x65737566,0x00000000,0x00060005,
	0x00000035,0x6867696c,0x65705374,0x616c7563,0x00000072,0x00050005,0x00000040,0x4374756f,
	0x726f6c6f,0x00000000,0x00030047,0x00000009,0x00000000,0x00030047,0x0000000d,0x00000000,
	0x00030047,0x00000011,0x00000000,0x00030047,0x00000013,0x00000000,0x00030047,0x00000015,
	0x00000000,0x00040047,0x00000015,0x0000001e,0x00000000,0x00030047,0x00000016,0x00000000,
	0x00030047,0x00000017,0x00000000,0x00030047,0x00000018,0x00000000,0x00030047,0x00000019,
	0x00000000,0x00040047,0x00000019,0x0000001e,0x00000001,0x00030047,0x0000001a,0x00000000,
	0x00030047,0x0000001b,0x00000000,0x00030047,0x0000001c,0x00000000,0x00030047,0x00000020,
	0x00000000,0x00030047,0x00000022,0x00000000,0x00030047,0x00000023,0x00000000,0x00030047,
	0x00000024,0x00000000,0x00030047,0x00000025,0x00000000,0x00030047,0x00000026,0x00000000,
	0x00030047,0x00000027,0x00000000,0x00030047,0x00000028,0x00000000,0x00030047,0x00000029,
	0x00000000,0x00030047,0x0000002a,0x00000000,0x00030047,0x0000002b,0x00000000,0x00030047,
	0x0000002c,0x00000000,0x00030047,0x0000002d,0x00000000,0x00030047,0x0000002e,0x00000000,
	0x00030047,0x0000002f,0x00000000,0x00030047,0x00000031,0x00000000,0x00030047,0x00000032,
	0x00000000,0x00030047,0x00000033,0x00000000,0x00030047,0x00000034,0x00000000,0x00030047,
	0x00000035,0x00000000,0x00030047,0x00000036,0x00000000,0x00030047,0x00000037,0x00000000,
	0x00030047,0x00000038,0x00000000,0x00030047,0x00000039,0x00000000,0x00030047,0x0000003a,
	0x00000000,0x00030047,0x0000003b,0x00000000,0x00030047,0x0000003c,0x00000000,0x00030047,
	0x0000003d,0x00000000,0x00030047,0x00000040,0x00000000,0x00040047,0x00000040,0x0000001e,
	0x00000000,0x00030047,0x00000041,0x00000000,0x00030047,0x00000042,0x00000000,0x00030047,
	0x00000043,0x00000000,0x00020013,0x00000002,0x00030021,0x00000003,0x00000002,0x00030016,
	0x00000006,0x00000020,0x00040017,0x00000007,0x00000006,0x00000003,0x00040020,0x00000008,
	0x00000007,0x00000007,0x0004002b,0x00000006,0x0000000a,0x3e4ccccd,0x0004002b,0x00000006,
	0x0000000b,0x3f800000,0x0006002c,0x00000007,0x0000000c,0x0000000a,0x0000000a,0x0000000b,
	0x0004002b,0x00000006,0x0000000e,0x3f000000,0x0006002c,0x00000007,0x0000000f,0x0000000e,
	0x0000000e,0x0000000e,0x00040020,0x00000010,0x00000007,0x00000006,0x0004002b,0x00000006,
	0x00000012,0x41200000,0x00040020,0x00000014,0x00000001,0x00000007,0x0004003b,0x00000014,
	0x00000015,0x00000001,0x0004003b,0x00000014,0x00000019,0x00000001,0x0004002b,0x00000006,
	0x0000001d,0xbf13cd3a,0x0004002b,0x00000006,0x0000001e,0x3f13cd3a,0x0006002c,0x00000007,
	0x0000001f,0x0000001d,0x0000001e,0x0000001e,0x0004002b,0x00000006,0x00000021,0x40000000,
	0x0004002b,0x00000006,0x00000030,0x00000000,0x00040017,0x0000003e,0x00000006,0x00000004,
	0x00040020,0x0000003f,0x00000003,0x0000003e,0x0004003b,0x0000003f,0x00000040,0x00000003,
	0x00040015,0x00000046,0x00000020,0x00000000,0x0004002b,0x00000046,0x00000047,0x00000003,
	0x00040020,0x00000048,0x00000003,0x00000006,0x00050036,0x00000002,0x00000004,0x00000000,
	0x00000003,0x000200f8,0x00000005,0x0004003b,0x00000008,0x00000009,0x00000007,0x0004003b,
	0x00000008,0x0000000d,0x00000007,0x0004003b,0x00000010,0x00000011,0x00000007,0x0004003b,
	0x00000008,0x00000013,0x00000007,0x0004003b,0x00000008,0x00000018,0x00000007,0x0004003b,
	0x00000008,0x0000001c,0x00000007,0x0004003b,0x00000008,0x00000020,0x00000007,0x0004003b,
	0x00000008,0x0000002b,0x00000007,0x0004003b,0x00000008,0x00000035,0x00000007,0x0003003e,
	0x00000009,0x0000000c,0x0003003e,0x0000000d,0x0000000f,0x0003003e,0x00000011,0x00000012,
	0x0004003d,0x00000007,0x00000016,0x00000015,0x0006000c,0x00000007,0x00000017,0x00000001,
	0x00000045,0x00000016,0x0003003e,0x00000013,0x00000017,0x0004003d,0x00000007,0x0000001a,
	0x00000019,0x0006000c,0x00000007,0x0000001b,0x00000001,0x00000045,0x0000001a,0x0003003e,
	0x00000018,0x0000001b,0x0003003e,0x0000001c,0x0000001f,0x0004003d,0x00000007,0x00000022,
	0x0000001c,0x0004003d,0x00000007,0x00000023,0x00000018,0x00050094,0x00000006,0x00000024,
	0x00000022,0x00000023,0x00050085,0x00000006,0x00000025,0x00000021,0x00000024,0x0004003d,
	0x00000007,0x00000026,0x00000018,0x0005008e,0x00000007,0x00000027,0x00000026,0x00000025,
	0x0004003d,0x00000007,0x00000028,0x0000001c,0x00050083,0x00000007,0x00000029,0x00000027,
	0x00000028,0x0006000c,0x00000007,0x0000002a,0x00000001,0x00000045,0x00000029,0x0003003e,
	0x00000020,0x0000002a,0x0004003d,0x00000007,0x0000002c,0x00000009,0x0004003d,0x00000007,
	0x0000002d,0x00000018,0x0004003d,0x00000007,0x0000002e,0x0000001c,0x00050094,0x00000006,
	0x0000002f,0x0000002d,0x0000002e,0x0007000c,0x00000006,0x00000031,0x00000001,0x00000028,
	0x0000002f,0x00000030,0x00050085,0x00000006,0x00000032,0x00000031,0x0000000e,0x00050081,
	0x00000006,0x00000033,0x00000032,0x0000000e,0x0005008e,0x00000007,0x00000034,0x0000002c,
	0x00000033,0x0003003e,0x0000002b,0x00000034,0x0004003d,0x00000007,0x00000036,0x0000000d,
	0x0004003d,0x00000007,0x00000037,0x00000020,0x0004003d,0x00000007,0x00000038,0x00000013,
	0x00050094,0x00000006,0x00000039,0x00000037,0x00000038,0x0007000c,0x00000006,0x0000003a,
	0x00000001,0x00000028,0x00000039,0x00000030,0x0004003d,0x00000006,0x0000003b,0x00000011,
	0x0007000c,0x00000006,0x0000003c,0x00000001,0x0000001a,0x0000003a,0x0000003b,0x0005008e,
	0x00000007,0x0000003d,0x00000036,0x0000003c,0x0003003e,0x00000035,0x0000003d,0x0004003d,
	0x00000007,0x00000041,0x0000002b,0x0004003d,0x00000007,0x00000042,0x00000035,0x00050081,
	0x00000007,0x00000043,0x00000041,0x00000042,0x0004003d,0x0000003e,0x00000044,0x00000040,
	0x0009004f,0x0000003e,0x00000045,0x00000044,0x00000043,0x00000004,0x00000005,0x00000006,
	0x00000003,0x0003003e,0x00000040,0x00000045,0x00050041,0x00000048,0x00000049,0x00000040,
	0x00000047,0x0003003e,0x00000049,0x0000000b,0x000100fd,0x00010038
};

static ksGpuProgramParm normalMappedProgramParms[] =
{
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,		KS_GPU_PROGRAM_PARM_TYPE_PUSH_CONSTANT_FLOAT_MATRIX4X4,	KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_MODEL_MATRIX,		"ModelMatrix",		0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,		KS_GPU_PROGRAM_PARM_TYPE_BUFFER_UNIFORM,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_SCENE_MATRICES,		"SceneMatrices",	0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_0,					"Texture0",			1 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_1,					"Texture1",			2 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_2,					"Texture2",			3 }
};

static const char normalMappedVertexProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"layout( std430, push_constant ) uniform PushConstants\n"
	"{\n"
	"	layout( offset =   0 ) mat4 ModelMatrix;\n"
	"} pc;\n"
	"layout( std140, binding = 0 ) uniform SceneMatrices\n"
	"{\n"
	"	layout( offset =   0 ) mat4 ViewMatrix;\n"
	"	layout( offset =  64 ) mat4 ProjectionMatrix;\n"
	"};\n"
	"layout( location = 0 ) in vec3 vertexPosition;\n"
	"layout( location = 1 ) in vec3 vertexNormal;\n"
	"layout( location = 2 ) in vec3 vertexTangent;\n"
	"layout( location = 3 ) in vec3 vertexBinormal;\n"
	"layout( location = 4 ) in vec2 vertexUv0;\n"
	"layout( location = 0 ) out vec3 fragmentEyeDir;\n"
	"layout( location = 1 ) out vec3 fragmentNormal;\n"
	"layout( location = 2 ) out vec3 fragmentTangent;\n"
	"layout( location = 3 ) out vec3 fragmentBinormal;\n"
	"layout( location = 4 ) out vec2 fragmentUv0;\n"
	"out gl_PerVertex { vec4 gl_Position; };\n"
	"vec3 multiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[1].x * v.y + m[2].x * v.z,\n"
	"		m[0].y * v.x + m[1].y * v.y + m[2].y * v.z,\n"
	"		m[0].z * v.x + m[1].z * v.y + m[2].z * v.z );\n"
	"}\n"
	"vec3 transposeMultiply3x3( mat4 m, vec3 v )\n"
	"{\n"
	"	return vec3(\n"
	"		m[0].x * v.x + m[0].y * v.y + m[0].z * v.z,\n"
	"		m[1].x * v.x + m[1].y * v.y + m[1].z * v.z,\n"
	"		m[2].x * v.x + m[2].y * v.y + m[2].z * v.z );\n"
	"}\n"
	"void main( void )\n"
	"{\n"
	"	vec4 vertexWorldPos = pc.ModelMatrix * vec4( vertexPosition, 1.0 );\n"
	"	vec3 eyeWorldPos = transposeMultiply3x3( ViewMatrix, -vec3( ViewMatrix[3] ) );\n"
	"	gl_Position = ProjectionMatrix * ( ViewMatrix * vertexWorldPos );\n"
	"	fragmentEyeDir = eyeWorldPos - vec3( vertexWorldPos );\n"
	"	fragmentNormal = multiply3x3( pc.ModelMatrix, vertexNormal );\n"
	"	fragmentTangent = multiply3x3( pc.ModelMatrix, vertexTangent );\n"
	"	fragmentBinormal = multiply3x3( pc.ModelMatrix, vertexBinormal );\n"
	"	fragmentUv0 = vertexUv0;\n"
	"}\n";

static const unsigned int normalMappedVertexProgramSPIRV[] =
{
	// SPIRV99.947 15-Feb-2016
	0x07230203,0x00010000,0x00080001,0x000000e1,0x00000000,0x00020011,0x00000001,0x0006000b,
	0x00000001,0x4c534c47,0x6474732e,0x3035342e,0x00000000,0x0003000e,0x00000000,0x00000001,
	0x0010000f,0x00000000,0x00000004,0x6e69616d,0x00000000,0x00000093,0x000000b0,0x000000bb,
	0x000000c3,0x000000c4,0x000000cb,0x000000cc,0x000000d3,0x000000d4,0x000000dd,0x000000df,
	0x00030003,0x00000001,0x00000136,0x00040005,0x00000004,0x6e69616d,0x00000000,0x00080005,
	0x0000000f,0x746c756d,0x796c7069,0x28337833,0x3434666d,0x3366763b,0x0000003b,0x00030005,
	0x0000000d,0x0000006d,0x00030005,0x0000000e,0x00000076,0x000a0005,0x00000013,0x6e617274,
	0x736f7073,0x6c754d65,0x6c706974,0x33783379,0x34666d28,0x66763b34,0x00003b33,0x00030005,
	0x00000011,0x0000006d,0x00030005,0x00000012,0x00000076,0x00060005,0x0000008b,0x74726576,
	0x6f577865,0x50646c72,0x0000736f,0x00060005,0x0000008c,0x68737550,0x736e6f43,0x746e6174,
	0x00000073,0x00060006,0x0000008c,0x00000000,0x65646f4d,0x74614d6c,0x00786972,0x00030005,
	0x0000008e,0x00006370,0x00060005,0x00000093,0x74726576,0x6f507865,0x69746973,0x00006e6f,
	0x00050005,0x0000009b,0x57657965,0x646c726f,0x00736f50,0x00060005,0x0000009c,0x6e656353,
	0x74614d65,0x65636972,0x00000073,0x00060006,0x0000009c,0x00000000,0x77656956,0x7274614d,
	0x00007869,0x00080006,0x0000009c,0x00000001,0x6a6f7250,0x69746365,0x614d6e6f,0x78697274,
	0x00000000,0x00030005,0x0000009e,0x00006275,0x00040005,0x000000a8,0x61726170,0x0000006d,
	0x00040005,0x000000ac,0x61726170,0x0000006d,0x00060005,0x000000ae,0x505f6c67,0x65567265,
	0x78657472,0x00000000,0x00060006,0x000000ae,0x00000000,0x505f6c67,0x7469736f,0x006e6f69,
	0x00070006,0x000000ae,0x00000001,0x505f6c67,0x746e696f,0x657a6953,0x00000000,0x00030005,
	0x000000b0,0x00000000,0x00060005,0x000000bb,0x67617266,0x746e656d,0x44657945,0x00007269,
	0x00060005,0x000000c3,0x67617266,0x746e656d,0x6d726f4e,0x00006c61,0x00060005,0x000000c4,
	0x74726576,0x6f4e7865,0x6c616d72,0x00000000,0x00040005,0x000000c5,0x61726170,0x0000006d,
	0x00040005,0x000000c8,0x61726170,0x0000006d,0x00060005,0x000000cb,0x67617266,0x746e656d,
	0x676e6154,0x00746e65,0x00060005,0x000000cc,0x74726576,0x61547865,0x6e65676e,0x00000074,
	0x00040005,0x000000cd,0x61726170,0x0000006d,0x00040005,0x000000d0,0x61726170,0x0000006d,
	0x00070005,0x000000d3,0x67617266,0x746e656d,0x6f6e6942,0x6c616d72,0x00000000,0x00060005,
	0x000000d4,0x74726576,0x69427865,0x6d726f6e,0x00006c61,0x00040005,0x000000d5,0x61726170,
	0x0000006d,0x00040005,0x000000d8,0x61726170,0x0000006d,0x00050005,0x000000dd,0x67617266,
	0x746e656d,0x00307655,0x00050005,0x000000df,0x74726576,0x76557865,0x00000030,0x00040048,
	0x0000008c,0x00000000,0x00000005,0x00050048,0x0000008c,0x00000000,0x00000023,0x00000000,
	0x00050048,0x0000008c,0x00000000,0x00000007,0x00000010,0x00030047,0x0000008c,0x00000002,
	0x00040047,0x00000093,0x0000001e,0x00000000,0x00040048,0x0000009c,0x00000000,0x00000005,
	0x00050048,0x0000009c,0x00000000,0x00000023,0x00000000,0x00050048,0x0000009c,0x00000000,
	0x00000007,0x00000010,0x00040048,0x0000009c,0x00000001,0x00000005,0x00050048,0x0000009c,
	0x00000001,0x00000023,0x00000040,0x00050048,0x0000009c,0x00000001,0x00000007,0x00000010,
	0x00030047,0x0000009c,0x00000002,0x00040047,0x0000009e,0x00000022,0x00000000,0x00040047,
	0x0000009e,0x00000021,0x00000000,0x00050048,0x000000ae,0x00000000,0x0000000b,0x00000000,
	0x00050048,0x000000ae,0x00000001,0x0000000b,0x00000001,0x00030047,0x000000ae,0x00000002,
	0x00040047,0x000000bb,0x0000001e,0x00000000,0x00040047,0x000000c3,0x0000001e,0x00000001,
	0x00040047,0x000000c4,0x0000001e,0x00000001,0x00040047,0x000000cb,0x0000001e,0x00000002,
	0x00040047,0x000000cc,0x0000001e,0x00000002,0x00040047,0x000000d3,0x0000001e,0x00000003,
	0x00040047,0x000000d4,0x0000001e,0x00000003,0x00040047,0x000000dd,0x0000001e,0x00000004,
	0x00040047,0x000000df,0x0000001e,0x00000004,0x00020013,0x00000002,0x00030021,0x00000003,
	0x00000002,0x00030016,0x00000006,0x00000020,0x00040017,0x00000007,0x00000006,0x00000004,
	0x00040018,0x00000008,0x00000007,0x00000004,0x00040020,0x00000009,0x00000007,0x00000008,
	0x00040017,0x0000000a,0x00000006,0x00000003,0x00040020,0x0000000b,0x00000007,0x0000000a,
	0x00050021,0x0000000c,0x0000000a,0x00000009,0x0000000b,0x00040015,0x00000015,0x00000020,
	0x00000001,0x0004002b,0x00000015,0x00000016,0x00000000,0x00040015,0x00000017,0x00000020,
	0x00000000,0x0004002b,0x00000017,0x00000018,0x00000000,0x00040020,0x00000019,0x00000007,
	0x00000006,0x0004002b,0x00000015,0x0000001f,0x00000001,0x0004002b,0x00000017,0x00000022,
	0x00000001,0x0004002b,0x00000015,0x00000027,0x00000002,0x0004002b,0x00000017,0x0000002a,
	0x00000002,0x00040020,0x0000008a,0x00000007,0x00000007,0x0003001e,0x0000008c,0x00000008,
	0x00040020,0x0000008d,0x00000009,0x0000008c,0x0004003b,0x0000008d,0x0000008e,0x00000009,
	0x00040020,0x0000008f,0x00000009,0x00000008,0x00040020,0x00000092,0x00000001,0x0000000a,
	0x0004003b,0x00000092,0x00000093,0x00000001,0x0004002b,0x00000006,0x00000095,0x3f800000,
	0x0004001e,0x0000009c,0x00000008,0x00000008,0x00040020,0x0000009d,0x00000002,0x0000009c,
	0x0004003b,0x0000009d,0x0000009e,0x00000002,0x0004002b,0x00000015,0x0000009f,0x00000003,
	0x00040020,0x000000a0,0x00000002,0x00000007,0x00040020,0x000000a9,0x00000002,0x00000008,
	0x0004001e,0x000000ae,0x00000007,0x00000006,0x00040020,0x000000af,0x00000003,0x000000ae,
	0x0004003b,0x000000af,0x000000b0,0x00000003,0x00040020,0x000000b8,0x00000003,0x00000007,
	0x00040020,0x000000ba,0x00000003,0x0000000a,0x0004003b,0x000000ba,0x000000bb,0x00000003,
	0x0004003b,0x000000ba,0x000000c3,0x00000003,0x0004003b,0x00000092,0x000000c4,0x00000001,
	0x0004003b,0x000000ba,0x000000cb,0x00000003,0x0004003b,0x00000092,0x000000cc,0x00000001,
	0x0004003b,0x000000ba,0x000000d3,0x00000003,0x0004003b,0x00000092,0x000000d4,0x00000001,
	0x00040017,0x000000db,0x00000006,0x00000002,0x00040020,0x000000dc,0x00000003,0x000000db,
	0x0004003b,0x000000dc,0x000000dd,0x00000003,0x00040020,0x000000de,0x00000001,0x000000db,
	0x0004003b,0x000000de,0x000000df,0x00000001,0x00050036,0x00000002,0x00000004,0x00000000,
	0x00000003,0x000200f8,0x00000005,0x0004003b,0x0000008a,0x0000008b,0x00000007,0x0004003b,
	0x0000000b,0x0000009b,0x00000007,0x0004003b,0x00000009,0x000000a8,0x00000007,0x0004003b,
	0x0000000b,0x000000ac,0x00000007,0x0004003b,0x00000009,0x000000c5,0x00000007,0x0004003b,
	0x0000000b,0x000000c8,0x00000007,0x0004003b,0x00000009,0x000000cd,0x00000007,0x0004003b,
	0x0000000b,0x000000d0,0x00000007,0x0004003b,0x00000009,0x000000d5,0x00000007,0x0004003b,
	0x0000000b,0x000000d8,0x00000007,0x00050041,0x0000008f,0x00000090,0x0000008e,0x00000016,
	0x0004003d,0x00000008,0x00000091,0x00000090,0x0004003d,0x0000000a,0x00000094,0x00000093,
	0x00050051,0x00000006,0x00000096,0x00000094,0x00000000,0x00050051,0x00000006,0x00000097,
	0x00000094,0x00000001,0x00050051,0x00000006,0x00000098,0x00000094,0x00000002,0x00070050,
	0x00000007,0x00000099,0x00000096,0x00000097,0x00000098,0x00000095,0x00050091,0x00000007,
	0x0000009a,0x00000091,0x00000099,0x0003003e,0x0000008b,0x0000009a,0x00060041,0x000000a0,
	0x000000a1,0x0000009e,0x00000016,0x0000009f,0x0004003d,0x00000007,0x000000a2,0x000000a1,
	0x00050051,0x00000006,0x000000a3,0x000000a2,0x00000000,0x00050051,0x00000006,0x000000a4,
	0x000000a2,0x00000001,0x00050051,0x00000006,0x000000a5,0x000000a2,0x00000002,0x00060050,
	0x0000000a,0x000000a6,0x000000a3,0x000000a4,0x000000a5,0x0004007f,0x0000000a,0x000000a7,
	0x000000a6,0x00050041,0x000000a9,0x000000aa,0x0000009e,0x00000016,0x0004003d,0x00000008,
	0x000000ab,0x000000aa,0x0003003e,0x000000a8,0x000000ab,0x0003003e,0x000000ac,0x000000a7,
	0x00060039,0x0000000a,0x000000ad,0x00000013,0x000000a8,0x000000ac,0x0003003e,0x0000009b,
	0x000000ad,0x00050041,0x000000a9,0x000000b1,0x0000009e,0x0000001f,0x0004003d,0x00000008,
	0x000000b2,0x000000b1,0x00050041,0x000000a9,0x000000b3,0x0000009e,0x00000016,0x0004003d,
	0x00000008,0x000000b4,0x000000b3,0x0004003d,0x00000007,0x000000b5,0x0000008b,0x00050091,
	0x00000007,0x000000b6,0x000000b4,0x000000b5,0x00050091,0x00000007,0x000000b7,0x000000b2,
	0x000000b6,0x00050041,0x000000b8,0x000000b9,0x000000b0,0x00000016,0x0003003e,0x000000b9,
	0x000000b7,0x0004003d,0x0000000a,0x000000bc,0x0000009b,0x0004003d,0x00000007,0x000000bd,
	0x0000008b,0x00050051,0x00000006,0x000000be,0x000000bd,0x00000000,0x00050051,0x00000006,
	0x000000bf,0x000000bd,0x00000001,0x00050051,0x00000006,0x000000c0,0x000000bd,0x00000002,
	0x00060050,0x0000000a,0x000000c1,0x000000be,0x000000bf,0x000000c0,0x00050083,0x0000000a,
	0x000000c2,0x000000bc,0x000000c1,0x0003003e,0x000000bb,0x000000c2,0x00050041,0x0000008f,
	0x000000c6,0x0000008e,0x00000016,0x0004003d,0x00000008,0x000000c7,0x000000c6,0x0003003e,
	0x000000c5,0x000000c7,0x0004003d,0x0000000a,0x000000c9,0x000000c4,0x0003003e,0x000000c8,
	0x000000c9,0x00060039,0x0000000a,0x000000ca,0x0000000f,0x000000c5,0x000000c8,0x0003003e,
	0x000000c3,0x000000ca,0x00050041,0x0000008f,0x000000ce,0x0000008e,0x00000016,0x0004003d,
	0x00000008,0x000000cf,0x000000ce,0x0003003e,0x000000cd,0x000000cf,0x0004003d,0x0000000a,
	0x000000d1,0x000000cc,0x0003003e,0x000000d0,0x000000d1,0x00060039,0x0000000a,0x000000d2,
	0x0000000f,0x000000cd,0x000000d0,0x0003003e,0x000000cb,0x000000d2,0x00050041,0x0000008f,
	0x000000d6,0x0000008e,0x00000016,0x0004003d,0x00000008,0x000000d7,0x000000d6,0x0003003e,
	0x000000d5,0x000000d7,0x0004003d,0x0000000a,0x000000d9,0x000000d4,0x0003003e,0x000000d8,
	0x000000d9,0x00060039,0x0000000a,0x000000da,0x0000000f,0x000000d5,0x000000d8,0x0003003e,
	0x000000d3,0x000000da,0x0004003d,0x000000db,0x000000e0,0x000000df,0x0003003e,0x000000dd,
	0x000000e0,0x000100fd,0x00010038,0x00050036,0x0000000a,0x0000000f,0x00000000,0x0000000c,
	0x00030037,0x00000009,0x0000000d,0x00030037,0x0000000b,0x0000000e,0x000200f8,0x00000010,
	0x00060041,0x00000019,0x0000001a,0x0000000d,0x00000016,0x00000018,0x0004003d,0x00000006,
	0x0000001b,0x0000001a,0x00050041,0x00000019,0x0000001c,0x0000000e,0x00000018,0x0004003d,
	0x00000006,0x0000001d,0x0000001c,0x00050085,0x00000006,0x0000001e,0x0000001b,0x0000001d,
	0x00060041,0x00000019,0x00000020,0x0000000d,0x0000001f,0x00000018,0x0004003d,0x00000006,
	0x00000021,0x00000020,0x00050041,0x00000019,0x00000023,0x0000000e,0x00000022,0x0004003d,
	0x00000006,0x00000024,0x00000023,0x00050085,0x00000006,0x00000025,0x00000021,0x00000024,
	0x00050081,0x00000006,0x00000026,0x0000001e,0x00000025,0x00060041,0x00000019,0x00000028,
	0x0000000d,0x00000027,0x00000018,0x0004003d,0x00000006,0x00000029,0x00000028,0x00050041,
	0x00000019,0x0000002b,0x0000000e,0x0000002a,0x0004003d,0x00000006,0x0000002c,0x0000002b,
	0x00050085,0x00000006,0x0000002d,0x00000029,0x0000002c,0x00050081,0x00000006,0x0000002e,
	0x00000026,0x0000002d,0x00060041,0x00000019,0x0000002f,0x0000000d,0x00000016,0x00000022,
	0x0004003d,0x00000006,0x00000030,0x0000002f,0x00050041,0x00000019,0x00000031,0x0000000e,
	0x00000018,0x0004003d,0x00000006,0x00000032,0x00000031,0x00050085,0x00000006,0x00000033,
	0x00000030,0x00000032,0x00060041,0x00000019,0x00000034,0x0000000d,0x0000001f,0x00000022,
	0x0004003d,0x00000006,0x00000035,0x00000034,0x00050041,0x00000019,0x00000036,0x0000000e,
	0x00000022,0x0004003d,0x00000006,0x00000037,0x00000036,0x00050085,0x00000006,0x00000038,
	0x00000035,0x00000037,0x00050081,0x00000006,0x00000039,0x00000033,0x00000038,0x00060041,
	0x00000019,0x0000003a,0x0000000d,0x00000027,0x00000022,0x0004003d,0x00000006,0x0000003b,
	0x0000003a,0x00050041,0x00000019,0x0000003c,0x0000000e,0x0000002a,0x0004003d,0x00000006,
	0x0000003d,0x0000003c,0x00050085,0x00000006,0x0000003e,0x0000003b,0x0000003d,0x00050081,
	0x00000006,0x0000003f,0x00000039,0x0000003e,0x00060041,0x00000019,0x00000040,0x0000000d,
	0x00000016,0x0000002a,0x0004003d,0x00000006,0x00000041,0x00000040,0x00050041,0x00000019,
	0x00000042,0x0000000e,0x00000018,0x0004003d,0x00000006,0x00000043,0x00000042,0x00050085,
	0x00000006,0x00000044,0x00000041,0x00000043,0x00060041,0x00000019,0x00000045,0x0000000d,
	0x0000001f,0x0000002a,0x0004003d,0x00000006,0x00000046,0x00000045,0x00050041,0x00000019,
	0x00000047,0x0000000e,0x00000022,0x0004003d,0x00000006,0x00000048,0x00000047,0x00050085,
	0x00000006,0x00000049,0x00000046,0x00000048,0x00050081,0x00000006,0x0000004a,0x00000044,
	0x00000049,0x00060041,0x00000019,0x0000004b,0x0000000d,0x00000027,0x0000002a,0x0004003d,
	0x00000006,0x0000004c,0x0000004b,0x00050041,0x00000019,0x0000004d,0x0000000e,0x0000002a,
	0x0004003d,0x00000006,0x0000004e,0x0000004d,0x00050085,0x00000006,0x0000004f,0x0000004c,
	0x0000004e,0x00050081,0x00000006,0x00000050,0x0000004a,0x0000004f,0x00060050,0x0000000a,
	0x00000051,0x0000002e,0x0000003f,0x00000050,0x000200fe,0x00000051,0x00010038,0x00050036,
	0x0000000a,0x00000013,0x00000000,0x0000000c,0x00030037,0x00000009,0x00000011,0x00030037,
	0x0000000b,0x00000012,0x000200f8,0x00000014,0x00060041,0x00000019,0x00000054,0x00000011,
	0x00000016,0x00000018,0x0004003d,0x00000006,0x00000055,0x00000054,0x00050041,0x00000019,
	0x00000056,0x00000012,0x00000018,0x0004003d,0x00000006,0x00000057,0x00000056,0x00050085,
	0x00000006,0x00000058,0x00000055,0x00000057,0x00060041,0x00000019,0x00000059,0x00000011,
	0x00000016,0x00000022,0x0004003d,0x00000006,0x0000005a,0x00000059,0x00050041,0x00000019,
	0x0000005b,0x00000012,0x00000022,0x0004003d,0x00000006,0x0000005c,0x0000005b,0x00050085,
	0x00000006,0x0000005d,0x0000005a,0x0000005c,0x00050081,0x00000006,0x0000005e,0x00000058,
	0x0000005d,0x00060041,0x00000019,0x0000005f,0x00000011,0x00000016,0x0000002a,0x0004003d,
	0x00000006,0x00000060,0x0000005f,0x00050041,0x00000019,0x00000061,0x00000012,0x0000002a,
	0x0004003d,0x00000006,0x00000062,0x00000061,0x00050085,0x00000006,0x00000063,0x00000060,
	0x00000062,0x00050081,0x00000006,0x00000064,0x0000005e,0x00000063,0x00060041,0x00000019,
	0x00000065,0x00000011,0x0000001f,0x00000018,0x0004003d,0x00000006,0x00000066,0x00000065,
	0x00050041,0x00000019,0x00000067,0x00000012,0x00000018,0x0004003d,0x00000006,0x00000068,
	0x00000067,0x00050085,0x00000006,0x00000069,0x00000066,0x00000068,0x00060041,0x00000019,
	0x0000006a,0x00000011,0x0000001f,0x00000022,0x0004003d,0x00000006,0x0000006b,0x0000006a,
	0x00050041,0x00000019,0x0000006c,0x00000012,0x00000022,0x0004003d,0x00000006,0x0000006d,
	0x0000006c,0x00050085,0x00000006,0x0000006e,0x0000006b,0x0000006d,0x00050081,0x00000006,
	0x0000006f,0x00000069,0x0000006e,0x00060041,0x00000019,0x00000070,0x00000011,0x0000001f,
	0x0000002a,0x0004003d,0x00000006,0x00000071,0x00000070,0x00050041,0x00000019,0x00000072,
	0x00000012,0x0000002a,0x0004003d,0x00000006,0x00000073,0x00000072,0x00050085,0x00000006,
	0x00000074,0x00000071,0x00000073,0x00050081,0x00000006,0x00000075,0x0000006f,0x00000074,
	0x00060041,0x00000019,0x00000076,0x00000011,0x00000027,0x00000018,0x0004003d,0x00000006,
	0x00000077,0x00000076,0x00050041,0x00000019,0x00000078,0x00000012,0x00000018,0x0004003d,
	0x00000006,0x00000079,0x00000078,0x00050085,0x00000006,0x0000007a,0x00000077,0x00000079,
	0x00060041,0x00000019,0x0000007b,0x00000011,0x00000027,0x00000022,0x0004003d,0x00000006,
	0x0000007c,0x0000007b,0x00050041,0x00000019,0x0000007d,0x00000012,0x00000022,0x0004003d,
	0x00000006,0x0000007e,0x0000007d,0x00050085,0x00000006,0x0000007f,0x0000007c,0x0000007e,
	0x00050081,0x00000006,0x00000080,0x0000007a,0x0000007f,0x00060041,0x00000019,0x00000081,
	0x00000011,0x00000027,0x0000002a,0x0004003d,0x00000006,0x00000082,0x00000081,0x00050041,
	0x00000019,0x00000083,0x00000012,0x0000002a,0x0004003d,0x00000006,0x00000084,0x00000083,
	0x00050085,0x00000006,0x00000085,0x00000082,0x00000084,0x00050081,0x00000006,0x00000086,
	0x00000080,0x00000085,0x00060050,0x0000000a,0x00000087,0x00000064,0x00000075,0x00000086,
	0x000200fe,0x00000087,0x00010038
};

static const char normalMappedMultiViewVertexProgramGLSL[] =
	"";

static const unsigned int normalMappedMultiViewVertexProgramSPIRV[] =
{
	0x00000000
};

static const char normalMapped100LightsFragmentProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"layout( binding = 1 ) uniform sampler2D Texture0;\n"
	"layout( binding = 2 ) uniform sampler2D Texture1;\n"
	"layout( binding = 3 ) uniform sampler2D Texture2;\n"
	"layout( location = 0 ) in lowp vec3 fragmentEyeDir;\n"
	"layout( location = 1 ) in lowp vec3 fragmentNormal;\n"
	"layout( location = 2 ) in lowp vec3 fragmentTangent;\n"
	"layout( location = 3 ) in lowp vec3 fragmentBinormal;\n"
	"layout( location = 4 ) in lowp vec2 fragmentUv0;\n"
	"layout( location = 0 ) out lowp vec4 outColor;\n"
	"void main()\n"
	"{\n"
	"	lowp vec3 diffuseMap = texture( Texture0, fragmentUv0 ).xyz;\n"
	"	lowp vec3 specularMap = texture( Texture1, fragmentUv0 ).xyz * 2.0;\n"
	"	lowp vec3 normalMap = texture( Texture2, fragmentUv0 ).xyz * 2.0 - 1.0;\n"
	"	lowp float specularPower = 10.0;\n"
	"	lowp vec3 eyeDir = normalize( fragmentEyeDir );\n"
	"	lowp vec3 normal = normalize( normalMap.x * fragmentTangent + normalMap.y * fragmentBinormal + normalMap.z * fragmentNormal );\n"
	"\n"
	"	lowp vec3 color = vec3( 0 );\n"
	"	for ( int i = 0; i < 100; i++ )\n"
	"	{\n"
	"		lowp vec3 lightDir = normalize( vec3( -1.0, 1.0, 1.0 ) );\n"
	"		lowp vec3 lightReflection = normalize( 2.0 * dot( lightDir, normal ) * normal - lightDir );\n"
	"		lowp vec3 lightDiffuse = diffuseMap * ( max( dot( normal, lightDir ), 0.0 ) * 0.5 + 0.5 );\n"
	"		lowp vec3 lightSpecular = specularMap * pow( max( dot( lightReflection, eyeDir ), 0.0 ), specularPower );\n"
	"		color += ( lightDiffuse + lightSpecular ) * ( 1.0 / 100.0 );\n"
	"	}\n"
	"\n"
	"	outColor.xyz = color;\n"
	"	outColor.w = 1.0;\n"
	"}\n";

static const unsigned int normalMapped100LightsFragmentProgramSPIRV[] =
{
	// SPIRV99.947 15-Feb-2016
	0x07230203,0x00010000,0x00080001,0x0000008a,0x00000000,0x00020011,0x00000001,0x0006000b,
	0x00000001,0x4c534c47,0x6474732e,0x3035342e,0x00000000,0x0003000e,0x00000000,0x00000001,
	0x000b000f,0x00000004,0x00000004,0x6e69616d,0x00000000,0x00000011,0x0000002d,0x00000035,
	0x0000003b,0x00000042,0x00000083,0x00030010,0x00000004,0x00000007,0x00030003,0x00000001,
	0x00000136,0x00040005,0x00000004,0x6e69616d,0x00000000,0x00050005,0x00000009,0x66666964,
	0x4d657375,0x00007061,0x00050005,0x0000000d,0x74786554,0x30657275,0x00000000,0x00050005,
	0x00000011,0x67617266,0x746e656d,0x00307655,0x00050005,0x00000016,0x63657073,0x72616c75,
	0x0070614d,0x00050005,0x00000017,0x74786554,0x31657275,0x00000000,0x00050005,0x0000001e,
	0x6d726f6e,0x614d6c61,0x00000070,0x00050005,0x0000001f,0x74786554,0x32657275,0x00000000,
	0x00060005,0x00000029,0x63657073,0x72616c75,0x65776f50,0x00000072,0x00040005,0x0000002b,
	0x44657965,0x00007269,0x00060005,0x0000002d,0x67617266,0x746e656d,0x44657945,0x00007269,
	0x00040005,0x00000030,0x6d726f6e,0x00006c61,0x00060005,0x00000035,0x67617266,0x746e656d,
	0x676e6154,0x00746e65,0x00070005,0x0000003b,0x67617266,0x746e656d,0x6f6e6942,0x6c616d72,
	0x00000000,0x00060005,0x00000042,0x67617266,0x746e656d,0x6d726f4e,0x00006c61,0x00040005,
	0x00000047,0x6f6c6f63,0x00000072,0x00030005,0x0000004c,0x00000069,0x00050005,0x00000057,
	0x6867696c,0x72694474,0x00000000,0x00060005,0x0000005b,0x6867696c,0x66655274,0x7463656c,
	0x006e6f69,0x00060005,0x00000065,0x6867696c,0x66694474,0x65737566,0x00000000,0x00060005,
	0x0000006f,0x6867696c,0x65705374,0x616c7563,0x00000072,0x00050005,0x00000083,0x4374756f,
	0x726f6c6f,0x00000000,0x00030047,0x00000009,0x00000000,0x00030047,0x0000000d,0x00000000,
	0x00040047,0x0000000d,0x00000022,0x00000000,0x00040047,0x0000000d,0x00000021,0x00000001,
	0x00030047,0x0000000e,0x00000000,0x00030047,0x00000011,0x00000000,0x00040047,0x00000011,
	0x0000001e,0x00000004,0x00030047,0x00000012,0x00000000,0x00030047,0x00000014,0x00000000,
	0x00030047,0x00000015,0x00000000,0x00030047,0x00000016,0x00000000,0x00030047,0x00000017,
	0x00000000,0x00040047,0x00000017,0x00000022,0x00000000,0x00040047,0x00000017,0x00000021,
	0x00000002,0x00030047,0x00000018,0x00000000,0x00030047,0x00000019,0x00000000,0x00030047,
	0x0000001a,0x00000000,0x00030047,0x0000001b,0x00000000,0x00030047,0x0000001d,0x00000000,
	0x00030047,0x0000001e,0x00000000,0x00030047,0x0000001f,0x00000000,0x00040047,0x0000001f,
	0x00000022,0x00000000,0x00040047,0x0000001f,0x00000021,0x00000003,0x00030047,0x00000020,
	0x00000000,0x00030047,0x00000021,0x00000000,0x00030047,0x00000022,0x00000000,0x00030047,
	0x00000023,0x00000000,0x00030047,0x00000024,0x00000000,0x00030047,0x00000026,0x00000000,
	0x00030047,0x00000027,0x00000000,0x00030047,0x00000029,0x00000000,0x00030047,0x0000002b,
	0x00000000,0x00030047,0x0000002d,0x00000000,0x00040047,0x0000002d,0x0000001e,0x00000000,
	0x00030047,0x0000002e,0x00000000,0x00030047,0x0000002f,0x00000000,0x00030047,0x00000030,
	0x00000000,0x00030047,0x00000034,0x00000000,0x00030047,0x00000035,0x00000000,0x00040047,
	0x00000035,0x0000001e,0x00000002,0x00030047,0x00000036,0x00000000,0x00030047,0x00000037,
	0x00000000,0x00030047,0x0000003a,0x00000000,0x00030047,0x0000003b,0x00000000,0x00040047,
	0x0000003b,0x0000001e,0x00000003,0x00030047,0x0000003c,0x00000000,0x00030047,0x0000003d,
	0x00000000,0x00030047,0x0000003e,0x00000000,0x00030047,0x00000041,0x00000000,0x00030047,
	0x00000042,0x00000000,0x00040047,0x00000042,0x0000001e,0x00000001,0x00030047,0x00000043,
	0x00000000,0x00030047,0x00000044,0x00000000,0x00030047,0x00000045,0x00000000,0x00030047,
	0x00000046,0x00000000,0x00030047,0x00000047,0x00000000,0x00030047,0x0000004c,0x00000000,
	0x00030047,0x00000053,0x00000000,0x00030047,0x00000057,0x00000000,0x00030047,0x0000005b,
	0x00000000,0x00030047,0x0000005c,0x00000000,0x00030047,0x0000005d,0x00000000,0x00030047,
	0x0000005e,0x00000000,0x00030047,0x0000005f,0x00000000,0x00030047,0x00000060,0x00000000,
	0x00030047,0x00000061,0x00000000,0x00030047,0x00000062,0x00000000,0x00030047,0x00000063,
	0x00000000,0x00030047,0x00000064,0x00000000,0x00030047,0x00000065,0x00000000,0x00030047,
	0x00000066,0x00000000,0x00030047,0x00000067,0x00000000,0x00030047,0x00000068,0x00000000,
	0x00030047,0x00000069,0x00000000,0x00030047,0x0000006a,0x00000000,0x00030047,0x0000006c,
	0x00000000,0x00030047,0x0000006d,0x00000000,0x00030047,0x0000006e,0x00000000,0x00030047,
	0x0000006f,0x00000000,0x00030047,0x00000070,0x00000000,0x00030047,0x00000071,0x00000000,
	0x00030047,0x00000072,0x00000000,0x00030047,0x00000073,0x00000000,0x00030047,0x00000074,
	0x00000000,0x00030047,0x00000075,0x00000000,0x00030047,0x00000076,0x00000000,0x00030047,
	0x00000077,0x00000000,0x00030047,0x00000078,0x00000000,0x00030047,0x00000079,0x00000000,
	0x00030047,0x0000007a,0x00000000,0x00030047,0x0000007c,0x00000000,0x00030047,0x0000007d,
	0x00000000,0x00030047,0x0000007e,0x00000000,0x00030047,0x0000007f,0x00000000,0x00030047,
	0x00000081,0x00000000,0x00030047,0x00000083,0x00000000,0x00040047,0x00000083,0x0000001e,
	0x00000000,0x00030047,0x00000084,0x00000000,0x00020013,0x00000002,0x00030021,0x00000003,
	0x00000002,0x00030016,0x00000006,0x00000020,0x00040017,0x00000007,0x00000006,0x00000003,
	0x00040020,0x00000008,0x00000007,0x00000007,0x00090019,0x0000000a,0x00000006,0x00000001,
	0x00000000,0x00000000,0x00000000,0x00000001,0x00000000,0x0003001b,0x0000000b,0x0000000a,
	0x00040020,0x0000000c,0x00000000,0x0000000b,0x0004003b,0x0000000c,0x0000000d,0x00000000,
	0x00040017,0x0000000f,0x00000006,0x00000002,0x00040020,0x00000010,0x00000001,0x0000000f,
	0x0004003b,0x00000010,0x00000011,0x00000001,0x00040017,0x00000013,0x00000006,0x00000004,
	0x0004003b,0x0000000c,0x00000017,0x00000000,0x0004002b,0x00000006,0x0000001c,0x40000000,
	0x0004003b,0x0000000c,0x0000001f,0x00000000,0x0004002b,0x00000006,0x00000025,0x3f800000,
	0x00040020,0x00000028,0x00000007,0x00000006,0x0004002b,0x00000006,0x0000002a,0x41200000,
	0x00040020,0x0000002c,0x00000001,0x00000007,0x0004003b,0x0000002c,0x0000002d,0x00000001,
	0x00040015,0x00000031,0x00000020,0x00000000,0x0004002b,0x00000031,0x00000032,0x00000000,
	0x0004003b,0x0000002c,0x00000035,0x00000001,0x0004002b,0x00000031,0x00000038,0x00000001,
	0x0004003b,0x0000002c,0x0000003b,0x00000001,0x0004002b,0x00000031,0x0000003f,0x00000002,
	0x0004003b,0x0000002c,0x00000042,0x00000001,0x0004002b,0x00000006,0x00000048,0x00000000,
	0x0006002c,0x00000007,0x00000049,0x00000048,0x00000048,0x00000048,0x00040015,0x0000004a,
	0x00000020,0x00000001,0x00040020,0x0000004b,0x00000007,0x0000004a,0x0004002b,0x0000004a,
	0x0000004d,0x00000000,0x0004002b,0x0000004a,0x00000054,0x00000064,0x00020014,0x00000055,
	0x0004002b,0x00000006,0x00000058,0xbf13cd3a,0x0004002b,0x00000006,0x00000059,0x3f13cd3a,
	0x0006002c,0x00000007,0x0000005a,0x00000058,0x00000059,0x00000059,0x0004002b,0x00000006,
	0x0000006b,0x3f000000,0x0004002b,0x00000006,0x0000007b,0x3c23d70a,0x0004002b,0x0000004a,
	0x00000080,0x00000001,0x00040020,0x00000082,0x00000003,0x00000013,0x0004003b,0x00000082,
	0x00000083,0x00000003,0x0004002b,0x00000031,0x00000087,0x00000003,0x00040020,0x00000088,
	0x00000003,0x00000006,0x00050036,0x00000002,0x00000004,0x00000000,0x00000003,0x000200f8,
	0x00000005,0x0004003b,0x00000008,0x00000009,0x00000007,0x0004003b,0x00000008,0x00000016,
	0x00000007,0x0004003b,0x00000008,0x0000001e,0x00000007,0x0004003b,0x00000028,0x00000029,
	0x00000007,0x0004003b,0x00000008,0x0000002b,0x00000007,0x0004003b,0x00000008,0x00000030,
	0x00000007,0x0004003b,0x00000008,0x00000047,0x00000007,0x0004003b,0x0000004b,0x0000004c,
	0x00000007,0x0004003b,0x00000008,0x00000057,0x00000007,0x0004003b,0x00000008,0x0000005b,
	0x00000007,0x0004003b,0x00000008,0x00000065,0x00000007,0x0004003b,0x00000008,0x0000006f,
	0x00000007,0x0004003d,0x0000000b,0x0000000e,0x0000000d,0x0004003d,0x0000000f,0x00000012,
	0x00000011,0x00050057,0x00000013,0x00000014,0x0000000e,0x00000012,0x0008004f,0x00000007,
	0x00000015,0x00000014,0x00000014,0x00000000,0x00000001,0x00000002,0x0003003e,0x00000009,
	0x00000015,0x0004003d,0x0000000b,0x00000018,0x00000017,0x0004003d,0x0000000f,0x00000019,
	0x00000011,0x00050057,0x00000013,0x0000001a,0x00000018,0x00000019,0x0008004f,0x00000007,
	0x0000001b,0x0000001a,0x0000001a,0x00000000,0x00000001,0x00000002,0x0005008e,0x00000007,
	0x0000001d,0x0000001b,0x0000001c,0x0003003e,0x00000016,0x0000001d,0x0004003d,0x0000000b,
	0x00000020,0x0000001f,0x0004003d,0x0000000f,0x00000021,0x00000011,0x00050057,0x00000013,
	0x00000022,0x00000020,0x00000021,0x0008004f,0x00000007,0x00000023,0x00000022,0x00000022,
	0x00000000,0x00000001,0x00000002,0x0005008e,0x00000007,0x00000024,0x00000023,0x0000001c,
	0x00060050,0x00000007,0x00000026,0x00000025,0x00000025,0x00000025,0x00050083,0x00000007,
	0x00000027,0x00000024,0x00000026,0x0003003e,0x0000001e,0x00000027,0x0003003e,0x00000029,
	0x0000002a,0x0004003d,0x00000007,0x0000002e,0x0000002d,0x0006000c,0x00000007,0x0000002f,
	0x00000001,0x00000045,0x0000002e,0x0003003e,0x0000002b,0x0000002f,0x00050041,0x00000028,
	0x00000033,0x0000001e,0x00000032,0x0004003d,0x00000006,0x00000034,0x00000033,0x0004003d,
	0x00000007,0x00000036,0x00000035,0x0005008e,0x00000007,0x00000037,0x00000036,0x00000034,
	0x00050041,0x00000028,0x00000039,0x0000001e,0x00000038,0x0004003d,0x00000006,0x0000003a,
	0x00000039,0x0004003d,0x00000007,0x0000003c,0x0000003b,0x0005008e,0x00000007,0x0000003d,
	0x0000003c,0x0000003a,0x00050081,0x00000007,0x0000003e,0x00000037,0x0000003d,0x00050041,
	0x00000028,0x00000040,0x0000001e,0x0000003f,0x0004003d,0x00000006,0x00000041,0x00000040,
	0x0004003d,0x00000007,0x00000043,0x00000042,0x0005008e,0x00000007,0x00000044,0x00000043,
	0x00000041,0x00050081,0x00000007,0x00000045,0x0000003e,0x00000044,0x0006000c,0x00000007,
	0x00000046,0x00000001,0x00000045,0x00000045,0x0003003e,0x00000030,0x00000046,0x0003003e,
	0x00000047,0x00000049,0x0003003e,0x0000004c,0x0000004d,0x000200f9,0x0000004e,0x000200f8,
	0x0000004e,0x000400f6,0x00000050,0x00000051,0x00000000,0x000200f9,0x00000052,0x000200f8,
	0x00000052,0x0004003d,0x0000004a,0x00000053,0x0000004c,0x000500b1,0x00000055,0x00000056,
	0x00000053,0x00000054,0x000400fa,0x00000056,0x0000004f,0x00000050,0x000200f8,0x0000004f,
	0x0003003e,0x00000057,0x0000005a,0x0004003d,0x00000007,0x0000005c,0x00000057,0x0004003d,
	0x00000007,0x0000005d,0x00000030,0x00050094,0x00000006,0x0000005e,0x0000005c,0x0000005d,
	0x00050085,0x00000006,0x0000005f,0x0000001c,0x0000005e,0x0004003d,0x00000007,0x00000060,
	0x00000030,0x0005008e,0x00000007,0x00000061,0x00000060,0x0000005f,0x0004003d,0x00000007,
	0x00000062,0x00000057,0x00050083,0x00000007,0x00000063,0x00000061,0x00000062,0x0006000c,
	0x00000007,0x00000064,0x00000001,0x00000045,0x00000063,0x0003003e,0x0000005b,0x00000064,
	0x0004003d,0x00000007,0x00000066,0x00000009,0x0004003d,0x00000007,0x00000067,0x00000030,
	0x0004003d,0x00000007,0x00000068,0x00000057,0x00050094,0x00000006,0x00000069,0x00000067,
	0x00000068,0x0007000c,0x00000006,0x0000006a,0x00000001,0x00000028,0x00000069,0x00000048,
	0x00050085,0x00000006,0x0000006c,0x0000006a,0x0000006b,0x00050081,0x00000006,0x0000006d,
	0x0000006c,0x0000006b,0x0005008e,0x00000007,0x0000006e,0x00000066,0x0000006d,0x0003003e,
	0x00000065,0x0000006e,0x0004003d,0x00000007,0x00000070,0x00000016,0x0004003d,0x00000007,
	0x00000071,0x0000005b,0x0004003d,0x00000007,0x00000072,0x0000002b,0x00050094,0x00000006,
	0x00000073,0x00000071,0x00000072,0x0007000c,0x00000006,0x00000074,0x00000001,0x00000028,
	0x00000073,0x00000048,0x0004003d,0x00000006,0x00000075,0x00000029,0x0007000c,0x00000006,
	0x00000076,0x00000001,0x0000001a,0x00000074,0x00000075,0x0005008e,0x00000007,0x00000077,
	0x00000070,0x00000076,0x0003003e,0x0000006f,0x00000077,0x0004003d,0x00000007,0x00000078,
	0x00000065,0x0004003d,0x00000007,0x00000079,0x0000006f,0x00050081,0x00000007,0x0000007a,
	0x00000078,0x00000079,0x0005008e,0x00000007,0x0000007c,0x0000007a,0x0000007b,0x0004003d,
	0x00000007,0x0000007d,0x00000047,0x00050081,0x00000007,0x0000007e,0x0000007d,0x0000007c,
	0x0003003e,0x00000047,0x0000007e,0x000200f9,0x00000051,0x000200f8,0x00000051,0x0004003d,
	0x0000004a,0x0000007f,0x0000004c,0x00050080,0x0000004a,0x00000081,0x0000007f,0x00000080,
	0x0003003e,0x0000004c,0x00000081,0x000200f9,0x0000004e,0x000200f8,0x00000050,0x0004003d,
	0x00000007,0x00000084,0x00000047,0x0004003d,0x00000013,0x00000085,0x00000083,0x0009004f,
	0x00000013,0x00000086,0x00000085,0x00000084,0x00000004,0x00000005,0x00000006,0x00000003,
	0x0003003e,0x00000083,0x00000086,0x00050041,0x00000088,0x00000089,0x00000083,0x00000087,
	0x0003003e,0x00000089,0x00000025,0x000100fd,0x00010038
};

static const char normalMapped1000LightsFragmentProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"layout( binding = 1 ) uniform sampler2D Texture0;\n"
	"layout( binding = 2 ) uniform sampler2D Texture1;\n"
	"layout( binding = 3 ) uniform sampler2D Texture2;\n"
	"layout( location = 0 ) in lowp vec3 fragmentEyeDir;\n"
	"layout( location = 1 ) in lowp vec3 fragmentNormal;\n"
	"layout( location = 2 ) in lowp vec3 fragmentTangent;\n"
	"layout( location = 3 ) in lowp vec3 fragmentBinormal;\n"
	"layout( location = 4 ) in lowp vec2 fragmentUv0;\n"
	"layout( location = 0 ) out lowp vec4 outColor;\n"
	"void main()\n"
	"{\n"
	"	lowp vec3 diffuseMap = texture( Texture0, fragmentUv0 ).xyz;\n"
	"	lowp vec3 specularMap = texture( Texture1, fragmentUv0 ).xyz * 2.0;\n"
	"	lowp vec3 normalMap = texture( Texture2, fragmentUv0 ).xyz * 2.0 - 1.0;\n"
	"	lowp float specularPower = 10.0;\n"
	"	lowp vec3 eyeDir = normalize( fragmentEyeDir );\n"
	"	lowp vec3 normal = normalize( normalMap.x * fragmentTangent + normalMap.y * fragmentBinormal + normalMap.z * fragmentNormal );\n"
	"\n"
	"	lowp vec3 color = vec3( 0 );\n"
	"	for ( int i = 0; i < 1000; i++ )\n"
	"	{\n"
	"		lowp vec3 lightDir = normalize( vec3( -1.0, 1.0, 1.0 ) );\n"
	"		lowp vec3 lightReflection = normalize( 2.0 * dot( lightDir, normal ) * normal - lightDir );\n"
	"		lowp vec3 lightDiffuse = diffuseMap * ( max( dot( normal, lightDir ), 0.0 ) * 0.5 + 0.5 );\n"
	"		lowp vec3 lightSpecular = specularMap * pow( max( dot( lightReflection, eyeDir ), 0.0 ), specularPower );\n"
	"		color += ( lightDiffuse + lightSpecular ) * ( 1.0 / 1000.0 );\n"
	"	}\n"
	"\n"
	"	outColor.xyz = color;\n"
	"	outColor.w = 1.0;\n"
	"}\n";

static const unsigned int normalMapped1000LightsFragmentProgramSPIRV[] =
{
	// SPIRV99.947 15-Feb-2016
	0x07230203,0x00010000,0x00080001,0x0000008a,0x00000000,0x00020011,0x00000001,0x0006000b,
	0x00000001,0x4c534c47,0x6474732e,0x3035342e,0x00000000,0x0003000e,0x00000000,0x00000001,
	0x000b000f,0x00000004,0x00000004,0x6e69616d,0x00000000,0x00000011,0x0000002d,0x00000035,
	0x0000003b,0x00000042,0x00000083,0x00030010,0x00000004,0x00000007,0x00030003,0x00000001,
	0x00000136,0x00040005,0x00000004,0x6e69616d,0x00000000,0x00050005,0x00000009,0x66666964,
	0x4d657375,0x00007061,0x00050005,0x0000000d,0x74786554,0x30657275,0x00000000,0x00050005,
	0x00000011,0x67617266,0x746e656d,0x00307655,0x00050005,0x00000016,0x63657073,0x72616c75,
	0x0070614d,0x00050005,0x00000017,0x74786554,0x31657275,0x00000000,0x00050005,0x0000001e,
	0x6d726f6e,0x614d6c61,0x00000070,0x00050005,0x0000001f,0x74786554,0x32657275,0x00000000,
	0x00060005,0x00000029,0x63657073,0x72616c75,0x65776f50,0x00000072,0x00040005,0x0000002b,
	0x44657965,0x00007269,0x00060005,0x0000002d,0x67617266,0x746e656d,0x44657945,0x00007269,
	0x00040005,0x00000030,0x6d726f6e,0x00006c61,0x00060005,0x00000035,0x67617266,0x746e656d,
	0x676e6154,0x00746e65,0x00070005,0x0000003b,0x67617266,0x746e656d,0x6f6e6942,0x6c616d72,
	0x00000000,0x00060005,0x00000042,0x67617266,0x746e656d,0x6d726f4e,0x00006c61,0x00040005,
	0x00000047,0x6f6c6f63,0x00000072,0x00030005,0x0000004c,0x00000069,0x00050005,0x00000057,
	0x6867696c,0x72694474,0x00000000,0x00060005,0x0000005b,0x6867696c,0x66655274,0x7463656c,
	0x006e6f69,0x00060005,0x00000065,0x6867696c,0x66694474,0x65737566,0x00000000,0x00060005,
	0x0000006f,0x6867696c,0x65705374,0x616c7563,0x00000072,0x00050005,0x00000083,0x4374756f,
	0x726f6c6f,0x00000000,0x00030047,0x00000009,0x00000000,0x00030047,0x0000000d,0x00000000,
	0x00040047,0x0000000d,0x00000022,0x00000000,0x00040047,0x0000000d,0x00000021,0x00000001,
	0x00030047,0x0000000e,0x00000000,0x00030047,0x00000011,0x00000000,0x00040047,0x00000011,
	0x0000001e,0x00000004,0x00030047,0x00000012,0x00000000,0x00030047,0x00000014,0x00000000,
	0x00030047,0x00000015,0x00000000,0x00030047,0x00000016,0x00000000,0x00030047,0x00000017,
	0x00000000,0x00040047,0x00000017,0x00000022,0x00000000,0x00040047,0x00000017,0x00000021,
	0x00000002,0x00030047,0x00000018,0x00000000,0x00030047,0x00000019,0x00000000,0x00030047,
	0x0000001a,0x00000000,0x00030047,0x0000001b,0x00000000,0x00030047,0x0000001d,0x00000000,
	0x00030047,0x0000001e,0x00000000,0x00030047,0x0000001f,0x00000000,0x00040047,0x0000001f,
	0x00000022,0x00000000,0x00040047,0x0000001f,0x00000021,0x00000003,0x00030047,0x00000020,
	0x00000000,0x00030047,0x00000021,0x00000000,0x00030047,0x00000022,0x00000000,0x00030047,
	0x00000023,0x00000000,0x00030047,0x00000024,0x00000000,0x00030047,0x00000026,0x00000000,
	0x00030047,0x00000027,0x00000000,0x00030047,0x00000029,0x00000000,0x00030047,0x0000002b,
	0x00000000,0x00030047,0x0000002d,0x00000000,0x00040047,0x0000002d,0x0000001e,0x00000000,
	0x00030047,0x0000002e,0x00000000,0x00030047,0x0000002f,0x00000000,0x00030047,0x00000030,
	0x00000000,0x00030047,0x00000034,0x00000000,0x00030047,0x00000035,0x00000000,0x00040047,
	0x00000035,0x0000001e,0x00000002,0x00030047,0x00000036,0x00000000,0x00030047,0x00000037,
	0x00000000,0x00030047,0x0000003a,0x00000000,0x00030047,0x0000003b,0x00000000,0x00040047,
	0x0000003b,0x0000001e,0x00000003,0x00030047,0x0000003c,0x00000000,0x00030047,0x0000003d,
	0x00000000,0x00030047,0x0000003e,0x00000000,0x00030047,0x00000041,0x00000000,0x00030047,
	0x00000042,0x00000000,0x00040047,0x00000042,0x0000001e,0x00000001,0x00030047,0x00000043,
	0x00000000,0x00030047,0x00000044,0x00000000,0x00030047,0x00000045,0x00000000,0x00030047,
	0x00000046,0x00000000,0x00030047,0x00000047,0x00000000,0x00030047,0x0000004c,0x00000000,
	0x00030047,0x00000053,0x00000000,0x00030047,0x00000057,0x00000000,0x00030047,0x0000005b,
	0x00000000,0x00030047,0x0000005c,0x00000000,0x00030047,0x0000005d,0x00000000,0x00030047,
	0x0000005e,0x00000000,0x00030047,0x0000005f,0x00000000,0x00030047,0x00000060,0x00000000,
	0x00030047,0x00000061,0x00000000,0x00030047,0x00000062,0x00000000,0x00030047,0x00000063,
	0x00000000,0x00030047,0x00000064,0x00000000,0x00030047,0x00000065,0x00000000,0x00030047,
	0x00000066,0x00000000,0x00030047,0x00000067,0x00000000,0x00030047,0x00000068,0x00000000,
	0x00030047,0x00000069,0x00000000,0x00030047,0x0000006a,0x00000000,0x00030047,0x0000006c,
	0x00000000,0x00030047,0x0000006d,0x00000000,0x00030047,0x0000006e,0x00000000,0x00030047,
	0x0000006f,0x00000000,0x00030047,0x00000070,0x00000000,0x00030047,0x00000071,0x00000000,
	0x00030047,0x00000072,0x00000000,0x00030047,0x00000073,0x00000000,0x00030047,0x00000074,
	0x00000000,0x00030047,0x00000075,0x00000000,0x00030047,0x00000076,0x00000000,0x00030047,
	0x00000077,0x00000000,0x00030047,0x00000078,0x00000000,0x00030047,0x00000079,0x00000000,
	0x00030047,0x0000007a,0x00000000,0x00030047,0x0000007c,0x00000000,0x00030047,0x0000007d,
	0x00000000,0x00030047,0x0000007e,0x00000000,0x00030047,0x0000007f,0x00000000,0x00030047,
	0x00000081,0x00000000,0x00030047,0x00000083,0x00000000,0x00040047,0x00000083,0x0000001e,
	0x00000000,0x00030047,0x00000084,0x00000000,0x00020013,0x00000002,0x00030021,0x00000003,
	0x00000002,0x00030016,0x00000006,0x00000020,0x00040017,0x00000007,0x00000006,0x00000003,
	0x00040020,0x00000008,0x00000007,0x00000007,0x00090019,0x0000000a,0x00000006,0x00000001,
	0x00000000,0x00000000,0x00000000,0x00000001,0x00000000,0x0003001b,0x0000000b,0x0000000a,
	0x00040020,0x0000000c,0x00000000,0x0000000b,0x0004003b,0x0000000c,0x0000000d,0x00000000,
	0x00040017,0x0000000f,0x00000006,0x00000002,0x00040020,0x00000010,0x00000001,0x0000000f,
	0x0004003b,0x00000010,0x00000011,0x00000001,0x00040017,0x00000013,0x00000006,0x00000004,
	0x0004003b,0x0000000c,0x00000017,0x00000000,0x0004002b,0x00000006,0x0000001c,0x40000000,
	0x0004003b,0x0000000c,0x0000001f,0x00000000,0x0004002b,0x00000006,0x00000025,0x3f800000,
	0x00040020,0x00000028,0x00000007,0x00000006,0x0004002b,0x00000006,0x0000002a,0x41200000,
	0x00040020,0x0000002c,0x00000001,0x00000007,0x0004003b,0x0000002c,0x0000002d,0x00000001,
	0x00040015,0x00000031,0x00000020,0x00000000,0x0004002b,0x00000031,0x00000032,0x00000000,
	0x0004003b,0x0000002c,0x00000035,0x00000001,0x0004002b,0x00000031,0x00000038,0x00000001,
	0x0004003b,0x0000002c,0x0000003b,0x00000001,0x0004002b,0x00000031,0x0000003f,0x00000002,
	0x0004003b,0x0000002c,0x00000042,0x00000001,0x0004002b,0x00000006,0x00000048,0x00000000,
	0x0006002c,0x00000007,0x00000049,0x00000048,0x00000048,0x00000048,0x00040015,0x0000004a,
	0x00000020,0x00000001,0x00040020,0x0000004b,0x00000007,0x0000004a,0x0004002b,0x0000004a,
	0x0000004d,0x00000000,0x0004002b,0x0000004a,0x00000054,0x000003e8,0x00020014,0x00000055,
	0x0004002b,0x00000006,0x00000058,0xbf13cd3a,0x0004002b,0x00000006,0x00000059,0x3f13cd3a,
	0x0006002c,0x00000007,0x0000005a,0x00000058,0x00000059,0x00000059,0x0004002b,0x00000006,
	0x0000006b,0x3f000000,0x0004002b,0x00000006,0x0000007b,0x3a83126f,0x0004002b,0x0000004a,
	0x00000080,0x00000001,0x00040020,0x00000082,0x00000003,0x00000013,0x0004003b,0x00000082,
	0x00000083,0x00000003,0x0004002b,0x00000031,0x00000087,0x00000003,0x00040020,0x00000088,
	0x00000003,0x00000006,0x00050036,0x00000002,0x00000004,0x00000000,0x00000003,0x000200f8,
	0x00000005,0x0004003b,0x00000008,0x00000009,0x00000007,0x0004003b,0x00000008,0x00000016,
	0x00000007,0x0004003b,0x00000008,0x0000001e,0x00000007,0x0004003b,0x00000028,0x00000029,
	0x00000007,0x0004003b,0x00000008,0x0000002b,0x00000007,0x0004003b,0x00000008,0x00000030,
	0x00000007,0x0004003b,0x00000008,0x00000047,0x00000007,0x0004003b,0x0000004b,0x0000004c,
	0x00000007,0x0004003b,0x00000008,0x00000057,0x00000007,0x0004003b,0x00000008,0x0000005b,
	0x00000007,0x0004003b,0x00000008,0x00000065,0x00000007,0x0004003b,0x00000008,0x0000006f,
	0x00000007,0x0004003d,0x0000000b,0x0000000e,0x0000000d,0x0004003d,0x0000000f,0x00000012,
	0x00000011,0x00050057,0x00000013,0x00000014,0x0000000e,0x00000012,0x0008004f,0x00000007,
	0x00000015,0x00000014,0x00000014,0x00000000,0x00000001,0x00000002,0x0003003e,0x00000009,
	0x00000015,0x0004003d,0x0000000b,0x00000018,0x00000017,0x0004003d,0x0000000f,0x00000019,
	0x00000011,0x00050057,0x00000013,0x0000001a,0x00000018,0x00000019,0x0008004f,0x00000007,
	0x0000001b,0x0000001a,0x0000001a,0x00000000,0x00000001,0x00000002,0x0005008e,0x00000007,
	0x0000001d,0x0000001b,0x0000001c,0x0003003e,0x00000016,0x0000001d,0x0004003d,0x0000000b,
	0x00000020,0x0000001f,0x0004003d,0x0000000f,0x00000021,0x00000011,0x00050057,0x00000013,
	0x00000022,0x00000020,0x00000021,0x0008004f,0x00000007,0x00000023,0x00000022,0x00000022,
	0x00000000,0x00000001,0x00000002,0x0005008e,0x00000007,0x00000024,0x00000023,0x0000001c,
	0x00060050,0x00000007,0x00000026,0x00000025,0x00000025,0x00000025,0x00050083,0x00000007,
	0x00000027,0x00000024,0x00000026,0x0003003e,0x0000001e,0x00000027,0x0003003e,0x00000029,
	0x0000002a,0x0004003d,0x00000007,0x0000002e,0x0000002d,0x0006000c,0x00000007,0x0000002f,
	0x00000001,0x00000045,0x0000002e,0x0003003e,0x0000002b,0x0000002f,0x00050041,0x00000028,
	0x00000033,0x0000001e,0x00000032,0x0004003d,0x00000006,0x00000034,0x00000033,0x0004003d,
	0x00000007,0x00000036,0x00000035,0x0005008e,0x00000007,0x00000037,0x00000036,0x00000034,
	0x00050041,0x00000028,0x00000039,0x0000001e,0x00000038,0x0004003d,0x00000006,0x0000003a,
	0x00000039,0x0004003d,0x00000007,0x0000003c,0x0000003b,0x0005008e,0x00000007,0x0000003d,
	0x0000003c,0x0000003a,0x00050081,0x00000007,0x0000003e,0x00000037,0x0000003d,0x00050041,
	0x00000028,0x00000040,0x0000001e,0x0000003f,0x0004003d,0x00000006,0x00000041,0x00000040,
	0x0004003d,0x00000007,0x00000043,0x00000042,0x0005008e,0x00000007,0x00000044,0x00000043,
	0x00000041,0x00050081,0x00000007,0x00000045,0x0000003e,0x00000044,0x0006000c,0x00000007,
	0x00000046,0x00000001,0x00000045,0x00000045,0x0003003e,0x00000030,0x00000046,0x0003003e,
	0x00000047,0x00000049,0x0003003e,0x0000004c,0x0000004d,0x000200f9,0x0000004e,0x000200f8,
	0x0000004e,0x000400f6,0x00000050,0x00000051,0x00000000,0x000200f9,0x00000052,0x000200f8,
	0x00000052,0x0004003d,0x0000004a,0x00000053,0x0000004c,0x000500b1,0x00000055,0x00000056,
	0x00000053,0x00000054,0x000400fa,0x00000056,0x0000004f,0x00000050,0x000200f8,0x0000004f,
	0x0003003e,0x00000057,0x0000005a,0x0004003d,0x00000007,0x0000005c,0x00000057,0x0004003d,
	0x00000007,0x0000005d,0x00000030,0x00050094,0x00000006,0x0000005e,0x0000005c,0x0000005d,
	0x00050085,0x00000006,0x0000005f,0x0000001c,0x0000005e,0x0004003d,0x00000007,0x00000060,
	0x00000030,0x0005008e,0x00000007,0x00000061,0x00000060,0x0000005f,0x0004003d,0x00000007,
	0x00000062,0x00000057,0x00050083,0x00000007,0x00000063,0x00000061,0x00000062,0x0006000c,
	0x00000007,0x00000064,0x00000001,0x00000045,0x00000063,0x0003003e,0x0000005b,0x00000064,
	0x0004003d,0x00000007,0x00000066,0x00000009,0x0004003d,0x00000007,0x00000067,0x00000030,
	0x0004003d,0x00000007,0x00000068,0x00000057,0x00050094,0x00000006,0x00000069,0x00000067,
	0x00000068,0x0007000c,0x00000006,0x0000006a,0x00000001,0x00000028,0x00000069,0x00000048,
	0x00050085,0x00000006,0x0000006c,0x0000006a,0x0000006b,0x00050081,0x00000006,0x0000006d,
	0x0000006c,0x0000006b,0x0005008e,0x00000007,0x0000006e,0x00000066,0x0000006d,0x0003003e,
	0x00000065,0x0000006e,0x0004003d,0x00000007,0x00000070,0x00000016,0x0004003d,0x00000007,
	0x00000071,0x0000005b,0x0004003d,0x00000007,0x00000072,0x0000002b,0x00050094,0x00000006,
	0x00000073,0x00000071,0x00000072,0x0007000c,0x00000006,0x00000074,0x00000001,0x00000028,
	0x00000073,0x00000048,0x0004003d,0x00000006,0x00000075,0x00000029,0x0007000c,0x00000006,
	0x00000076,0x00000001,0x0000001a,0x00000074,0x00000075,0x0005008e,0x00000007,0x00000077,
	0x00000070,0x00000076,0x0003003e,0x0000006f,0x00000077,0x0004003d,0x00000007,0x00000078,
	0x00000065,0x0004003d,0x00000007,0x00000079,0x0000006f,0x00050081,0x00000007,0x0000007a,
	0x00000078,0x00000079,0x0005008e,0x00000007,0x0000007c,0x0000007a,0x0000007b,0x0004003d,
	0x00000007,0x0000007d,0x00000047,0x00050081,0x00000007,0x0000007e,0x0000007d,0x0000007c,
	0x0003003e,0x00000047,0x0000007e,0x000200f9,0x00000051,0x000200f8,0x00000051,0x0004003d,
	0x0000004a,0x0000007f,0x0000004c,0x00050080,0x0000004a,0x00000081,0x0000007f,0x00000080,
	0x0003003e,0x0000004c,0x00000081,0x000200f9,0x0000004e,0x000200f8,0x00000050,0x0004003d,
	0x00000007,0x00000084,0x00000047,0x0004003d,0x00000013,0x00000085,0x00000083,0x0009004f,
	0x00000013,0x00000086,0x00000085,0x00000084,0x00000004,0x00000005,0x00000006,0x00000003,
	0x0003003e,0x00000083,0x00000086,0x00050041,0x00000088,0x00000089,0x00000083,0x00000087,
	0x0003003e,0x00000089,0x00000025,0x000100fd,0x00010038
};

static const char normalMapped2000LightsFragmentProgramGLSL[] =
	"#version " GLSL_VERSION "\n"
	GLSL_EXTENSIONS
	"layout( binding = 1 ) uniform sampler2D Texture0;\n"
	"layout( binding = 2 ) uniform sampler2D Texture1;\n"
	"layout( binding = 3 ) uniform sampler2D Texture2;\n"
	"layout( location = 0 ) in lowp vec3 fragmentEyeDir;\n"
	"layout( location = 1 ) in lowp vec3 fragmentNormal;\n"
	"layout( location = 2 ) in lowp vec3 fragmentTangent;\n"
	"layout( location = 3 ) in lowp vec3 fragmentBinormal;\n"
	"layout( location = 4 ) in lowp vec2 fragmentUv0;\n"
	"layout( location = 0 ) out lowp vec4 outColor;\n"
	"void main()\n"
	"{\n"
	"	lowp vec3 diffuseMap = texture( Texture0, fragmentUv0 ).xyz;\n"
	"	lowp vec3 specularMap = texture( Texture1, fragmentUv0 ).xyz * 2.0;\n"
	"	lowp vec3 normalMap = texture( Texture2, fragmentUv0 ).xyz * 2.0 - 1.0;\n"
	"	lowp float specularPower = 10.0;\n"
	"	lowp vec3 eyeDir = normalize( fragmentEyeDir );\n"
	"	lowp vec3 normal = normalize( normalMap.x * fragmentTangent + normalMap.y * fragmentBinormal + normalMap.z * fragmentNormal );\n"
	"\n"
	"	lowp vec3 color = vec3( 0 );\n"
	"	for ( int i = 0; i < 2000; i++ )\n"
	"	{\n"
	"		lowp vec3 lightDir = normalize( vec3( -1.0, 1.0, 1.0 ) );\n"
	"		lowp vec3 lightReflection = normalize( 2.0 * dot( lightDir, normal ) * normal - lightDir );\n"
	"		lowp vec3 lightDiffuse = diffuseMap * ( max( dot( normal, lightDir ), 0.0 ) * 0.5 + 0.5 );\n"
	"		lowp vec3 lightSpecular = specularMap * pow( max( dot( lightReflection, eyeDir ), 0.0 ), specularPower );\n"
	"		color += ( lightDiffuse + lightSpecular ) * ( 1.0 / 2000.0 );\n"
	"	}\n"
	"\n"
	"	outColor.xyz = color;\n"
	"	outColor.w = 1.0;\n"
	"}\n";

static const unsigned int normalMapped2000LightsFragmentProgramSPIRV[] =
{
	// SPIRV99.947 15-Feb-2016
	0x07230203,0x00010000,0x00080001,0x0000008a,0x00000000,0x00020011,0x00000001,0x0006000b,
	0x00000001,0x4c534c47,0x6474732e,0x3035342e,0x00000000,0x0003000e,0x00000000,0x00000001,
	0x000b000f,0x00000004,0x00000004,0x6e69616d,0x00000000,0x00000011,0x0000002d,0x00000035,
	0x0000003b,0x00000042,0x00000083,0x00030010,0x00000004,0x00000007,0x00030003,0x00000001,
	0x00000136,0x00040005,0x00000004,0x6e69616d,0x00000000,0x00050005,0x00000009,0x66666964,
	0x4d657375,0x00007061,0x00050005,0x0000000d,0x74786554,0x30657275,0x00000000,0x00050005,
	0x00000011,0x67617266,0x746e656d,0x00307655,0x00050005,0x00000016,0x63657073,0x72616c75,
	0x0070614d,0x00050005,0x00000017,0x74786554,0x31657275,0x00000000,0x00050005,0x0000001e,
	0x6d726f6e,0x614d6c61,0x00000070,0x00050005,0x0000001f,0x74786554,0x32657275,0x00000000,
	0x00060005,0x00000029,0x63657073,0x72616c75,0x65776f50,0x00000072,0x00040005,0x0000002b,
	0x44657965,0x00007269,0x00060005,0x0000002d,0x67617266,0x746e656d,0x44657945,0x00007269,
	0x00040005,0x00000030,0x6d726f6e,0x00006c61,0x00060005,0x00000035,0x67617266,0x746e656d,
	0x676e6154,0x00746e65,0x00070005,0x0000003b,0x67617266,0x746e656d,0x6f6e6942,0x6c616d72,
	0x00000000,0x00060005,0x00000042,0x67617266,0x746e656d,0x6d726f4e,0x00006c61,0x00040005,
	0x00000047,0x6f6c6f63,0x00000072,0x00030005,0x0000004c,0x00000069,0x00050005,0x00000057,
	0x6867696c,0x72694474,0x00000000,0x00060005,0x0000005b,0x6867696c,0x66655274,0x7463656c,
	0x006e6f69,0x00060005,0x00000065,0x6867696c,0x66694474,0x65737566,0x00000000,0x00060005,
	0x0000006f,0x6867696c,0x65705374,0x616c7563,0x00000072,0x00050005,0x00000083,0x4374756f,
	0x726f6c6f,0x00000000,0x00030047,0x00000009,0x00000000,0x00030047,0x0000000d,0x00000000,
	0x00040047,0x0000000d,0x00000022,0x00000000,0x00040047,0x0000000d,0x00000021,0x00000001,
	0x00030047,0x0000000e,0x00000000,0x00030047,0x00000011,0x00000000,0x00040047,0x00000011,
	0x0000001e,0x00000004,0x00030047,0x00000012,0x00000000,0x00030047,0x00000014,0x00000000,
	0x00030047,0x00000015,0x00000000,0x00030047,0x00000016,0x00000000,0x00030047,0x00000017,
	0x00000000,0x00040047,0x00000017,0x00000022,0x00000000,0x00040047,0x00000017,0x00000021,
	0x00000002,0x00030047,0x00000018,0x00000000,0x00030047,0x00000019,0x00000000,0x00030047,
	0x0000001a,0x00000000,0x00030047,0x0000001b,0x00000000,0x00030047,0x0000001d,0x00000000,
	0x00030047,0x0000001e,0x00000000,0x00030047,0x0000001f,0x00000000,0x00040047,0x0000001f,
	0x00000022,0x00000000,0x00040047,0x0000001f,0x00000021,0x00000003,0x00030047,0x00000020,
	0x00000000,0x00030047,0x00000021,0x00000000,0x00030047,0x00000022,0x00000000,0x00030047,
	0x00000023,0x00000000,0x00030047,0x00000024,0x00000000,0x00030047,0x00000026,0x00000000,
	0x00030047,0x00000027,0x00000000,0x00030047,0x00000029,0x00000000,0x00030047,0x0000002b,
	0x00000000,0x00030047,0x0000002d,0x00000000,0x00040047,0x0000002d,0x0000001e,0x00000000,
	0x00030047,0x0000002e,0x00000000,0x00030047,0x0000002f,0x00000000,0x00030047,0x00000030,
	0x00000000,0x00030047,0x00000034,0x00000000,0x00030047,0x00000035,0x00000000,0x00040047,
	0x00000035,0x0000001e,0x00000002,0x00030047,0x00000036,0x00000000,0x00030047,0x00000037,
	0x00000000,0x00030047,0x0000003a,0x00000000,0x00030047,0x0000003b,0x00000000,0x00040047,
	0x0000003b,0x0000001e,0x00000003,0x00030047,0x0000003c,0x00000000,0x00030047,0x0000003d,
	0x00000000,0x00030047,0x0000003e,0x00000000,0x00030047,0x00000041,0x00000000,0x00030047,
	0x00000042,0x00000000,0x00040047,0x00000042,0x0000001e,0x00000001,0x00030047,0x00000043,
	0x00000000,0x00030047,0x00000044,0x00000000,0x00030047,0x00000045,0x00000000,0x00030047,
	0x00000046,0x00000000,0x00030047,0x00000047,0x00000000,0x00030047,0x0000004c,0x00000000,
	0x00030047,0x00000053,0x00000000,0x00030047,0x00000057,0x00000000,0x00030047,0x0000005b,
	0x00000000,0x00030047,0x0000005c,0x00000000,0x00030047,0x0000005d,0x00000000,0x00030047,
	0x0000005e,0x00000000,0x00030047,0x0000005f,0x00000000,0x00030047,0x00000060,0x00000000,
	0x00030047,0x00000061,0x00000000,0x00030047,0x00000062,0x00000000,0x00030047,0x00000063,
	0x00000000,0x00030047,0x00000064,0x00000000,0x00030047,0x00000065,0x00000000,0x00030047,
	0x00000066,0x00000000,0x00030047,0x00000067,0x00000000,0x00030047,0x00000068,0x00000000,
	0x00030047,0x00000069,0x00000000,0x00030047,0x0000006a,0x00000000,0x00030047,0x0000006c,
	0x00000000,0x00030047,0x0000006d,0x00000000,0x00030047,0x0000006e,0x00000000,0x00030047,
	0x0000006f,0x00000000,0x00030047,0x00000070,0x00000000,0x00030047,0x00000071,0x00000000,
	0x00030047,0x00000072,0x00000000,0x00030047,0x00000073,0x00000000,0x00030047,0x00000074,
	0x00000000,0x00030047,0x00000075,0x00000000,0x00030047,0x00000076,0x00000000,0x00030047,
	0x00000077,0x00000000,0x00030047,0x00000078,0x00000000,0x00030047,0x00000079,0x00000000,
	0x00030047,0x0000007a,0x00000000,0x00030047,0x0000007c,0x00000000,0x00030047,0x0000007d,
	0x00000000,0x00030047,0x0000007e,0x00000000,0x00030047,0x0000007f,0x00000000,0x00030047,
	0x00000081,0x00000000,0x00030047,0x00000083,0x00000000,0x00040047,0x00000083,0x0000001e,
	0x00000000,0x00030047,0x00000084,0x00000000,0x00020013,0x00000002,0x00030021,0x00000003,
	0x00000002,0x00030016,0x00000006,0x00000020,0x00040017,0x00000007,0x00000006,0x00000003,
	0x00040020,0x00000008,0x00000007,0x00000007,0x00090019,0x0000000a,0x00000006,0x00000001,
	0x00000000,0x00000000,0x00000000,0x00000001,0x00000000,0x0003001b,0x0000000b,0x0000000a,
	0x00040020,0x0000000c,0x00000000,0x0000000b,0x0004003b,0x0000000c,0x0000000d,0x00000000,
	0x00040017,0x0000000f,0x00000006,0x00000002,0x00040020,0x00000010,0x00000001,0x0000000f,
	0x0004003b,0x00000010,0x00000011,0x00000001,0x00040017,0x00000013,0x00000006,0x00000004,
	0x0004003b,0x0000000c,0x00000017,0x00000000,0x0004002b,0x00000006,0x0000001c,0x40000000,
	0x0004003b,0x0000000c,0x0000001f,0x00000000,0x0004002b,0x00000006,0x00000025,0x3f800000,
	0x00040020,0x00000028,0x00000007,0x00000006,0x0004002b,0x00000006,0x0000002a,0x41200000,
	0x00040020,0x0000002c,0x00000001,0x00000007,0x0004003b,0x0000002c,0x0000002d,0x00000001,
	0x00040015,0x00000031,0x00000020,0x00000000,0x0004002b,0x00000031,0x00000032,0x00000000,
	0x0004003b,0x0000002c,0x00000035,0x00000001,0x0004002b,0x00000031,0x00000038,0x00000001,
	0x0004003b,0x0000002c,0x0000003b,0x00000001,0x0004002b,0x00000031,0x0000003f,0x00000002,
	0x0004003b,0x0000002c,0x00000042,0x00000001,0x0004002b,0x00000006,0x00000048,0x00000000,
	0x0006002c,0x00000007,0x00000049,0x00000048,0x00000048,0x00000048,0x00040015,0x0000004a,
	0x00000020,0x00000001,0x00040020,0x0000004b,0x00000007,0x0000004a,0x0004002b,0x0000004a,
	0x0000004d,0x00000000,0x0004002b,0x0000004a,0x00000054,0x000007d0,0x00020014,0x00000055,
	0x0004002b,0x00000006,0x00000058,0xbf13cd3a,0x0004002b,0x00000006,0x00000059,0x3f13cd3a,
	0x0006002c,0x00000007,0x0000005a,0x00000058,0x00000059,0x00000059,0x0004002b,0x00000006,
	0x0000006b,0x3f000000,0x0004002b,0x00000006,0x0000007b,0x3a03126f,0x0004002b,0x0000004a,
	0x00000080,0x00000001,0x00040020,0x00000082,0x00000003,0x00000013,0x0004003b,0x00000082,
	0x00000083,0x00000003,0x0004002b,0x00000031,0x00000087,0x00000003,0x00040020,0x00000088,
	0x00000003,0x00000006,0x00050036,0x00000002,0x00000004,0x00000000,0x00000003,0x000200f8,
	0x00000005,0x0004003b,0x00000008,0x00000009,0x00000007,0x0004003b,0x00000008,0x00000016,
	0x00000007,0x0004003b,0x00000008,0x0000001e,0x00000007,0x0004003b,0x00000028,0x00000029,
	0x00000007,0x0004003b,0x00000008,0x0000002b,0x00000007,0x0004003b,0x00000008,0x00000030,
	0x00000007,0x0004003b,0x00000008,0x00000047,0x00000007,0x0004003b,0x0000004b,0x0000004c,
	0x00000007,0x0004003b,0x00000008,0x00000057,0x00000007,0x0004003b,0x00000008,0x0000005b,
	0x00000007,0x0004003b,0x00000008,0x00000065,0x00000007,0x0004003b,0x00000008,0x0000006f,
	0x00000007,0x0004003d,0x0000000b,0x0000000e,0x0000000d,0x0004003d,0x0000000f,0x00000012,
	0x00000011,0x00050057,0x00000013,0x00000014,0x0000000e,0x00000012,0x0008004f,0x00000007,
	0x00000015,0x00000014,0x00000014,0x00000000,0x00000001,0x00000002,0x0003003e,0x00000009,
	0x00000015,0x0004003d,0x0000000b,0x00000018,0x00000017,0x0004003d,0x0000000f,0x00000019,
	0x00000011,0x00050057,0x00000013,0x0000001a,0x00000018,0x00000019,0x0008004f,0x00000007,
	0x0000001b,0x0000001a,0x0000001a,0x00000000,0x00000001,0x00000002,0x0005008e,0x00000007,
	0x0000001d,0x0000001b,0x0000001c,0x0003003e,0x00000016,0x0000001d,0x0004003d,0x0000000b,
	0x00000020,0x0000001f,0x0004003d,0x0000000f,0x00000021,0x00000011,0x00050057,0x00000013,
	0x00000022,0x00000020,0x00000021,0x0008004f,0x00000007,0x00000023,0x00000022,0x00000022,
	0x00000000,0x00000001,0x00000002,0x0005008e,0x00000007,0x00000024,0x00000023,0x0000001c,
	0x00060050,0x00000007,0x00000026,0x00000025,0x00000025,0x00000025,0x00050083,0x00000007,
	0x00000027,0x00000024,0x00000026,0x0003003e,0x0000001e,0x00000027,0x0003003e,0x00000029,
	0x0000002a,0x0004003d,0x00000007,0x0000002e,0x0000002d,0x0006000c,0x00000007,0x0000002f,
	0x00000001,0x00000045,0x0000002e,0x0003003e,0x0000002b,0x0000002f,0x00050041,0x00000028,
	0x00000033,0x0000001e,0x00000032,0x0004003d,0x00000006,0x00000034,0x00000033,0x0004003d,
	0x00000007,0x00000036,0x00000035,0x0005008e,0x00000007,0x00000037,0x00000036,0x00000034,
	0x00050041,0x00000028,0x00000039,0x0000001e,0x00000038,0x0004003d,0x00000006,0x0000003a,
	0x00000039,0x0004003d,0x00000007,0x0000003c,0x0000003b,0x0005008e,0x00000007,0x0000003d,
	0x0000003c,0x0000003a,0x00050081,0x00000007,0x0000003e,0x00000037,0x0000003d,0x00050041,
	0x00000028,0x00000040,0x0000001e,0x0000003f,0x0004003d,0x00000006,0x00000041,0x00000040,
	0x0004003d,0x00000007,0x00000043,0x00000042,0x0005008e,0x00000007,0x00000044,0x00000043,
	0x00000041,0x00050081,0x00000007,0x00000045,0x0000003e,0x00000044,0x0006000c,0x00000007,
	0x00000046,0x00000001,0x00000045,0x00000045,0x0003003e,0x00000030,0x00000046,0x0003003e,
	0x00000047,0x00000049,0x0003003e,0x0000004c,0x0000004d,0x000200f9,0x0000004e,0x000200f8,
	0x0000004e,0x000400f6,0x00000050,0x00000051,0x00000000,0x000200f9,0x00000052,0x000200f8,
	0x00000052,0x0004003d,0x0000004a,0x00000053,0x0000004c,0x000500b1,0x00000055,0x00000056,
	0x00000053,0x00000054,0x000400fa,0x00000056,0x0000004f,0x00000050,0x000200f8,0x0000004f,
	0x0003003e,0x00000057,0x0000005a,0x0004003d,0x00000007,0x0000005c,0x00000057,0x0004003d,
	0x00000007,0x0000005d,0x00000030,0x00050094,0x00000006,0x0000005e,0x0000005c,0x0000005d,
	0x00050085,0x00000006,0x0000005f,0x0000001c,0x0000005e,0x0004003d,0x00000007,0x00000060,
	0x00000030,0x0005008e,0x00000007,0x00000061,0x00000060,0x0000005f,0x0004003d,0x00000007,
	0x00000062,0x00000057,0x00050083,0x00000007,0x00000063,0x00000061,0x00000062,0x0006000c,
	0x00000007,0x00000064,0x00000001,0x00000045,0x00000063,0x0003003e,0x0000005b,0x00000064,
	0x0004003d,0x00000007,0x00000066,0x00000009,0x0004003d,0x00000007,0x00000067,0x00000030,
	0x0004003d,0x00000007,0x00000068,0x00000057,0x00050094,0x00000006,0x00000069,0x00000067,
	0x00000068,0x0007000c,0x00000006,0x0000006a,0x00000001,0x00000028,0x00000069,0x00000048,
	0x00050085,0x00000006,0x0000006c,0x0000006a,0x0000006b,0x00050081,0x00000006,0x0000006d,
	0x0000006c,0x0000006b,0x0005008e,0x00000007,0x0000006e,0x00000066,0x0000006d,0x0003003e,
	0x00000065,0x0000006e,0x0004003d,0x00000007,0x00000070,0x00000016,0x0004003d,0x00000007,
	0x00000071,0x0000005b,0x0004003d,0x00000007,0x00000072,0x0000002b,0x00050094,0x00000006,
	0x00000073,0x00000071,0x00000072,0x0007000c,0x00000006,0x00000074,0x00000001,0x00000028,
	0x00000073,0x00000048,0x0004003d,0x00000006,0x00000075,0x00000029,0x0007000c,0x00000006,
	0x00000076,0x00000001,0x0000001a,0x00000074,0x00000075,0x0005008e,0x00000007,0x00000077,
	0x00000070,0x00000076,0x0003003e,0x0000006f,0x00000077,0x0004003d,0x00000007,0x00000078,
	0x00000065,0x0004003d,0x00000007,0x00000079,0x0000006f,0x00050081,0x00000007,0x0000007a,
	0x00000078,0x00000079,0x0005008e,0x00000007,0x0000007c,0x0000007a,0x0000007b,0x0004003d,
	0x00000007,0x0000007d,0x00000047,0x00050081,0x00000007,0x0000007e,0x0000007d,0x0000007c,
	0x0003003e,0x00000047,0x0000007e,0x000200f9,0x00000051,0x000200f8,0x00000051,0x0004003d,
	0x0000004a,0x0000007f,0x0000004c,0x00050080,0x0000004a,0x00000081,0x0000007f,0x00000080,
	0x0003003e,0x0000004c,0x00000081,0x000200f9,0x0000004e,0x000200f8,0x00000050,0x0004003d,
	0x00000007,0x00000084,0x00000047,0x0004003d,0x00000013,0x00000085,0x00000083,0x0009004f,
	0x00000013,0x00000086,0x00000085,0x00000084,0x00000004,0x00000005,0x00000006,0x00000003,
	0x0003003e,0x00000083,0x00000086,0x00050041,0x00000088,0x00000089,0x00000083,0x00000087,
	0x0003003e,0x00000089,0x00000025,0x000100fd,0x00010038
};

#elif GRAPHICS_API_D3D == 1

static ksGpuProgramParm flatShadedProgramParms[] =
{
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,	KS_GPU_PROGRAM_PARM_TYPE_PUSH_CONSTANT_FLOAT_MATRIX4X4,	KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_MODEL_MATRIX,		"ModelMatrix",		0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,	KS_GPU_PROGRAM_PARM_TYPE_BUFFER_UNIFORM,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_SCENE_MATRICES,		"SceneMatrices",	0 }
};

static const char flatShadedVertexProgramHLSL[] =
	"";

static const char flatShadedMultiViewVertexProgramHLSL[] =
	"";

static const char flatShadedFragmentProgramHLSL[] =
	"";

static ksGpuProgramParm normalMappedProgramParms[] =
{
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,		KS_GPU_PROGRAM_PARM_TYPE_PUSH_CONSTANT_FLOAT_MATRIX4X4,	KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_MODEL_MATRIX,		"ModelMatrix",		0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,		KS_GPU_PROGRAM_PARM_TYPE_BUFFER_UNIFORM,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_SCENE_MATRICES,		"SceneMatrices",	0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_0,					"Texture0",			0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_1,					"Texture1",			1 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_2,					"Texture2",			2 }
};

static const char normalMappedVertexProgramHLSL[] =
	"";

static const char normalMappedMultiViewVertexProgramHLSL[] =
	"";

static const char normalMapped100LightsFragmentProgramHLSL[] =
	"";

static const char normalMapped1000LightsFragmentProgramHLSL[] =
	"";

static const char normalMapped2000LightsFragmentProgramHLSL[] =
	"";

#elif GRAPHICS_API_METAL == 1

static ksGpuProgramParm flatShadedProgramParms[] =
{
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,	KS_GPU_PROGRAM_PARM_TYPE_PUSH_CONSTANT_FLOAT_MATRIX4X4,	KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_MODEL_MATRIX,		"ModelMatrix",		0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,	KS_GPU_PROGRAM_PARM_TYPE_BUFFER_UNIFORM,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_SCENE_MATRICES,		"SceneMatrices",	0 }
};

static const char flatShadedVertexProgramMetalSL[] =
	"";

static const char flatShadedMultiViewVertexProgramMetalSL[] =
	"";

static const char flatShadedFragmentProgramMetalSL[] =
	"";

static ksGpuProgramParm normalMappedProgramParms[] =
{
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,		KS_GPU_PROGRAM_PARM_TYPE_PUSH_CONSTANT_FLOAT_MATRIX4X4,	KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_MODEL_MATRIX,		"ModelMatrix",		0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_VERTEX,		KS_GPU_PROGRAM_PARM_TYPE_BUFFER_UNIFORM,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_UNIFORM_SCENE_MATRICES,		"SceneMatrices",	0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_0,					"Texture0",			0 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_1,					"Texture1",			1 },
	{ KS_GPU_PROGRAM_STAGE_FLAG_FRAGMENT,	KS_GPU_PROGRAM_PARM_TYPE_TEXTURE_SAMPLED,				KS_GPU_PROGRAM_PARM_ACCESS_READ_ONLY,	PROGRAM_TEXTURE_2,					"Texture2",			2 }
};

static const char normalMappedVertexProgramMetalSL[] =
	"";

static const char normalMappedMultiViewVertexProgramMetalSL[] =
	"";

static const char normalMapped100LightsFragmentProgramMetalSL[] =
	"";

static const char normalMapped1000LightsFragmentProgramMetalSL[] =
	"";

static const char normalMapped2000LightsFragmentProgramMetalSL[] =
	"";

#endif

static void ksPerfScene_Create( ksGpuContext * context, ksPerfScene * scene, ksSceneSettings * settings, ksGpuRenderPass * renderPass )
{
	memset( scene, 0, sizeof( ksPerfScene ) );

	ksGpuGeometry_CreateCube( context, &scene->geometry[0], 0.0f, 0.5f );			// 12 triangles
	ksGpuGeometry_CreateTorus( context, &scene->geometry[1], 8, 0.0f, 1.0f );		// 128 triangles
	ksGpuGeometry_CreateTorus( context, &scene->geometry[2], 16, 0.0f, 1.0f );	// 512 triangles
	ksGpuGeometry_CreateTorus( context, &scene->geometry[3], 32, 0.0f, 1.0f );	// 2048 triangles

	ksGpuGraphicsProgram_Create( context, &scene->program[0],
								settings->useMultiView ? PROGRAM( flatShadedMultiViewVertexProgram ) : PROGRAM( flatShadedVertexProgram ),
								settings->useMultiView ? sizeof( PROGRAM( flatShadedMultiViewVertexProgram ) ) : sizeof( PROGRAM( flatShadedVertexProgram ) ),
								PROGRAM( flatShadedFragmentProgram ),
								sizeof( PROGRAM( flatShadedFragmentProgram ) ),
								flatShadedProgramParms, ARRAY_SIZE( flatShadedProgramParms ),
								scene->geometry[0].layout, VERTEX_ATTRIBUTE_FLAG_POSITION | VERTEX_ATTRIBUTE_FLAG_NORMAL );
	ksGpuGraphicsProgram_Create( context, &scene->program[1],
								settings->useMultiView ? PROGRAM( normalMappedMultiViewVertexProgram ) : PROGRAM( normalMappedVertexProgram ),
								settings->useMultiView ? sizeof( PROGRAM( normalMappedMultiViewVertexProgram ) ) : sizeof( PROGRAM( normalMappedVertexProgram ) ),
								PROGRAM( normalMapped100LightsFragmentProgram ),
								sizeof( PROGRAM( normalMapped100LightsFragmentProgram ) ),
								normalMappedProgramParms, ARRAY_SIZE( normalMappedProgramParms ),
								scene->geometry[0].layout, VERTEX_ATTRIBUTE_FLAG_POSITION | VERTEX_ATTRIBUTE_FLAG_NORMAL |
								VERTEX_ATTRIBUTE_FLAG_TANGENT | VERTEX_ATTRIBUTE_FLAG_BINORMAL |
								VERTEX_ATTRIBUTE_FLAG_UV0 );
	ksGpuGraphicsProgram_Create( context, &scene->program[2],
								settings->useMultiView ? PROGRAM( normalMappedMultiViewVertexProgram ) : PROGRAM( normalMappedVertexProgram ),
								settings->useMultiView ? sizeof( PROGRAM( normalMappedMultiViewVertexProgram ) ) : sizeof( PROGRAM( normalMappedVertexProgram ) ),
								PROGRAM( normalMapped1000LightsFragmentProgram ),
								sizeof( PROGRAM( normalMapped1000LightsFragmentProgram ) ),
								normalMappedProgramParms, ARRAY_SIZE( normalMappedProgramParms ),
								scene->geometry[0].layout, VERTEX_ATTRIBUTE_FLAG_POSITION | VERTEX_ATTRIBUTE_FLAG_NORMAL |
								VERTEX_ATTRIBUTE_FLAG_TANGENT | VERTEX_ATTRIBUTE_FLAG_BINORMAL |
								VERTEX_ATTRIBUTE_FLAG_UV0 );
	ksGpuGraphicsProgram_Create( context, &scene->program[3],
								settings->useMultiView ? PROGRAM( normalMappedMultiViewVertexProgram ) : PROGRAM( normalMappedVertexProgram ),
								settings->useMultiView ? sizeof( PROGRAM( normalMappedMultiViewVertexProgram ) ) : sizeof( PROGRAM( normalMappedVertexProgram ) ),
								PROGRAM( normalMapped2000LightsFragmentProgram ),
								sizeof( PROGRAM( normalMapped2000LightsFragmentProgram ) ),
								normalMappedProgramParms, ARRAY_SIZE( normalMappedProgramParms ),
								scene->geometry[0].layout, VERTEX_ATTRIBUTE_FLAG_POSITION | VERTEX_ATTRIBUTE_FLAG_NORMAL |
								VERTEX_ATTRIBUTE_FLAG_TANGENT | VERTEX_ATTRIBUTE_FLAG_BINORMAL |
								VERTEX_ATTRIBUTE_FLAG_UV0 );

	for ( int i = 0; i < MAX_SCENE_TRIANGLE_LEVELS; i++ )
	{
		for ( int j = 0; j < MAX_SCENE_FRAGMENT_LEVELS; j++ )
		{
			ksGpuGraphicsPipelineParms pipelineParms;
			ksGpuGraphicsPipelineParms_Init( &pipelineParms );

			pipelineParms.renderPass = renderPass;
			pipelineParms.program = &scene->program[j];
			pipelineParms.geometry = &scene->geometry[i];

			ksGpuGraphicsPipeline_Create( context, &scene->pipelines[i][j], &pipelineParms );
		}
	}

	ksGpuBuffer_Create( context, &scene->sceneMatrices, KS_GPU_BUFFER_TYPE_UNIFORM, ( settings->useMultiView ? 4 : 2 ) * sizeof( ksMatrix4x4f ), NULL, false );

	ksGpuTexture_CreateDefault( context, &scene->diffuseTexture, KS_GPU_TEXTURE_DEFAULT_CHECKERBOARD, 256, 256, 0, 0, 1, true, false );
	ksGpuTexture_CreateDefault( context, &scene->specularTexture, KS_GPU_TEXTURE_DEFAULT_CHECKERBOARD, 256, 256, 0, 0, 1, true, false );
	ksGpuTexture_CreateDefault( context, &scene->normalTexture, KS_GPU_TEXTURE_DEFAULT_PYRAMIDS, 256, 256, 0, 0, 1, true, false );

	scene->settings = *settings;
	scene->newSettings = settings;

	const int maxDimension = 2 * ( 1 << ( MAX_SCENE_DRAWCALL_LEVELS - 1 ) );

	scene->bigRotationX = 0.0f;
	scene->bigRotationY = 0.0f;
	scene->smallRotationX = 0.0f;
	scene->smallRotationY = 0.0f;

	scene->modelMatrix = (ksMatrix4x4f *) AllocAlignedMemory( maxDimension * maxDimension * maxDimension * sizeof( ksMatrix4x4f ), sizeof( ksMatrix4x4f ) );
}

static void ksPerfScene_Destroy( ksGpuContext * context, ksPerfScene * scene )
{
	ksGpuContext_WaitIdle( context );

	for ( int i = 0; i < MAX_SCENE_TRIANGLE_LEVELS; i++ )
	{
		for ( int j = 0; j < MAX_SCENE_FRAGMENT_LEVELS; j++ )
		{
			ksGpuGraphicsPipeline_Destroy( context, &scene->pipelines[i][j] );
		}
	}

	for ( int i = 0; i < MAX_SCENE_TRIANGLE_LEVELS; i++ )
	{
		ksGpuGeometry_Destroy( context, &scene->geometry[i] );
	}

	for ( int i = 0; i < MAX_SCENE_FRAGMENT_LEVELS; i++ )
	{
		ksGpuGraphicsProgram_Destroy( context, &scene->program[i] );
	}

	ksGpuBuffer_Destroy( context, &scene->sceneMatrices );

	ksGpuTexture_Destroy( context, &scene->diffuseTexture );
	ksGpuTexture_Destroy( context, &scene->specularTexture );
	ksGpuTexture_Destroy( context, &scene->normalTexture );

	FreeAlignedMemory( scene->modelMatrix );
	scene->modelMatrix = NULL;
}

static void ksPerfScene_Simulate( ksPerfScene * scene, ksViewState * viewState, const ksNanoseconds time )
{
	// Must recreate the scene if multi-view is enabled/disabled.
	assert( scene->settings.useMultiView == scene->newSettings->useMultiView );
	scene->settings = *scene->newSettings;

	ksViewState_HandleHmd( viewState, time );

	if ( !scene->settings.simulationPaused )
	{
		// FIXME: use double?
		const float offset = time * ( MATH_PI / 1000.0f / 1000.0f / 1000.0f );
		scene->bigRotationX = 20.0f * offset;
		scene->bigRotationY = 10.0f * offset;
		scene->smallRotationX = -60.0f * offset;
		scene->smallRotationY = -40.0f * offset;
	}
}

static void ksPerfScene_UpdateBuffers( ksGpuCommandBuffer * commandBuffer, ksPerfScene * scene, const ksViewState * viewState, const int eye )
{
	ksMatrix4x4f * sceneMatrices = NULL;
	ksGpuBuffer * sceneMatricesBuffer = ksGpuCommandBuffer_MapBuffer( commandBuffer, &scene->sceneMatrices, (void **)&sceneMatrices );
	const int count = ( eye == 2 ) ? 2 : 1;
	memcpy( sceneMatrices + 0 * count, &viewState->viewMatrix[eye], count * sizeof( ksMatrix4x4f ) );
	memcpy( sceneMatrices + 1 * count, &viewState->projectionMatrix[eye], count * sizeof( ksMatrix4x4f ) );
	ksGpuCommandBuffer_UnmapBuffer( commandBuffer, &scene->sceneMatrices, sceneMatricesBuffer, KS_GPU_BUFFER_UNMAP_TYPE_COPY_BACK );
}

static void ksPerfScene_Render( ksGpuCommandBuffer * commandBuffer, ksPerfScene * scene, const ksViewState * viewState )
{
	UNUSED_PARM( viewState );

	const int dimension = 2 * ( 1 << scene->settings.drawCallLevel );
	const float cubeOffset = ( dimension - 1.0f ) * 0.5f;
	const float cubeScale = 2.0f;

	ksMatrix4x4f bigRotationMatrix;
	ksMatrix4x4f_CreateRotation( &bigRotationMatrix, scene->bigRotationX, scene->bigRotationY, 0.0f );

	ksMatrix4x4f bigTranslationMatrix;
	ksMatrix4x4f_CreateTranslation( &bigTranslationMatrix, 0.0f, 0.0f, - 2.5f * dimension );

	ksMatrix4x4f bigTransformMatrix;
	ksMatrix4x4f_Multiply( &bigTransformMatrix, &bigTranslationMatrix, &bigRotationMatrix );

	ksMatrix4x4f smallRotationMatrix;
	ksMatrix4x4f_CreateRotation( &smallRotationMatrix, scene->smallRotationX, scene->smallRotationY, 0.0f );

	ksGpuGraphicsCommand command;
	ksGpuGraphicsCommand_Init( &command );
	ksGpuGraphicsCommand_SetPipeline( &command, &scene->pipelines[scene->settings.triangleLevel][scene->settings.fragmentLevel] );
	ksGpuGraphicsCommand_SetParmBufferUniform( &command, PROGRAM_UNIFORM_SCENE_MATRICES, &scene->sceneMatrices );
	ksGpuGraphicsCommand_SetParmTextureSampled( &command, PROGRAM_TEXTURE_0, ( scene->settings.fragmentLevel >= 1 ) ? &scene->diffuseTexture : NULL );
	ksGpuGraphicsCommand_SetParmTextureSampled( &command, PROGRAM_TEXTURE_1, ( scene->settings.fragmentLevel >= 1 ) ? &scene->specularTexture : NULL );
	ksGpuGraphicsCommand_SetParmTextureSampled( &command, PROGRAM_TEXTURE_2, ( scene->settings.fragmentLevel >= 1 ) ? &scene->normalTexture : NULL );

	for ( int x = 0; x < dimension; x++ )
	{
		for ( int y = 0; y < dimension; y++ )
		{
			for ( int z = 0; z < dimension; z++ )
			{
				ksMatrix4x4f smallTranslationMatrix;
				ksMatrix4x4f_CreateTranslation( &smallTranslationMatrix, cubeScale * ( x - cubeOffset ), cubeScale * ( y - cubeOffset ), cubeScale * ( z - cubeOffset ) );

				ksMatrix4x4f smallTransformMatrix;
				ksMatrix4x4f_Multiply( &smallTransformMatrix, &smallTranslationMatrix, &smallRotationMatrix );

				ksMatrix4x4f * modelMatrix = &scene->modelMatrix[( x * dimension + y ) * dimension + z];
				ksMatrix4x4f_Multiply( modelMatrix, &bigTransformMatrix, &smallTransformMatrix );

				ksGpuGraphicsCommand_SetParmFloatMatrix4x4( &command, PROGRAM_UNIFORM_MODEL_MATRIX, modelMatrix );

				ksGpuCommandBuffer_SubmitGraphicsCommand( commandBuffer, &command );
			}
		}
	}
}
