﻿//--------------------------------------------------------------------------------------------------
// File : SampleTarget.h
// Desc : Targa Module.
// Copyright(c) Project Asura. All right reserved.
//--------------------------------------------------------------------------------------------------
#pragma once

//--------------------------------------------------------------------------------------------------
// Includes
//--------------------------------------------------------------------------------------------------
#include <cstdint>


////////////////////////////////////////////////////////////////////////////////////////////////////
// TGA_FORMA_TYPE enum
////////////////////////////////////////////////////////////////////////////////////////////////////
enum TGA_FORMAT_TYPE
{
    TGA_FORMAT_NONE             = 0,        //!< イメージなし.
    TGA_FORMAT_INDEXCOLOR       = 1,        //!< インデックスカラー(256色).
    TGA_FORMAT_FULLCOLOR        = 2,        //!< フルカラー
    TGA_FORMAT_GRAYSCALE        = 3,        //!< 白黒.
    TGA_FORMAT_RLE_INDEXCOLOR   = 9,        //!< RLE圧縮インデックスカラー.
    TGA_FORMAT_RLE_FULLCOLOR    = 10,       //!< RLE圧縮フルカラー.
    TGA_FORMAT_RLE_GRAYSCALE    = 11,       //!< RLE圧縮白黒.
};


////////////////////////////////////////////////////////////////////////////////////////////////////
// TGA_HEADER structure
////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct TGA_HEADER
{
    uint8_t  IdFieldLength;      // IDフィードのサイズ(範囲は0～255).
    uint8_t  HasColorMap;        // カラーマップ有無(0=なし, 1=あり)
    uint8_t  Format;             // 画像形式.
    uint16_t ColorMapEntry;      // カラーマップエントリー.
    uint16_t ColorMapLength;     // カラーマップのエントリーの総数.
    uint8_t  ColorMapEntrySize;  // カラーマップの1エントリー当たりのビット数.
    uint16_t OffsetX;            // 画像のX座標.
    uint16_t OffsetY;            // 画像のY座標.
    uint16_t Width;              // 画像の横幅.
    uint16_t Height;             // 画像の縦幅.
    uint8_t  BitPerPixel;        // ビットの深さ.
    uint8_t  ImageDescriptor;    // (0~3bit) : 属性, 4bit : 格納方向(0=左から右,1=右から左), 5bit : 格納方向(0=下から上, 1=上から下), 6~7bit : インタリーブ(使用不可).
};
#pragma pack( pop )


////////////////////////////////////////////////////////////////////////////////////////////////////
// TGA_FOOTER structure
////////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct TGA_FOOTER
{
    uint32_t  OffsetExt;      // 拡張データへのオフセット(byte数) [オフセットはファイルの先頭から].
    uint32_t  OffsetDev;      // ディベロッパーエリアへのオフセット(byte数)[オフセットはファイルの先頭から].
    char Tag[18];        // 'TRUEVISION-XFILE.\0'
};
#pragma pack( pop )


///////////////////////////////////////////////////////////////////////////////////////////////////
// TGA_EXTENSION structure
///////////////////////////////////////////////////////////////////////////////////////////////////
#pragma pack( push, 1 )
struct TGA_EXTENSION
{
    uint16_t    Size;                       //!< サイズ.
    char        AuthorName[ 41 ];           //!< 著作者名.
    char        AuthorComment[ 324 ];       //!< 著作者コメント.
    uint16_t    StampMonth;                 //!< タイムスタンプ　月(1-12).
    uint16_t    StampDay;                   //!< タイムスタンプ　日(1-31).
    uint16_t    StampYear;                  //!< タイムスタンプ　年(4桁, 例1989).
    uint16_t    StampHour;                  //!< タイムスタンプ　時(0-23).
    uint16_t    StampMinute;                //!< タイムスタンプ　分(0-59).
    uint16_t    StampSecond;                //!< タイムスタンプ　秒(0-59).
    char        JobName[ 41 ];              //!< ジョブ名 (最後のバイトはゼロが必須).
    uint16_t    JobHour;                    //!< ジョブ時間  時(0-65535)
    uint16_t    JobMinute;                  //!< ジョブ時間　分(0-59)
    uint16_t    JobSecond;                  //!< ジョブ時間　秒(0-59)
    char        SoftwareId[ 41 ];           //!< ソフトウェアID (最後のバイトはゼロが必須).
    uint16_t    VersionNumber;              //!< ソフトウェアバージョン    VersionNumber * 100になる.
    uint8_t     VersionLetter;              //!< ソフトウェアバージョン
    uint32_t    KeyColor;                   //!< キーカラー.
    uint16_t    PixelNumerator;             //!< ピクセル比分子　ピクセル横幅.
    uint16_t    PixelDenominator;           //!< ピクセル比分母　ピクセル縦幅.
    uint16_t    GammaNumerator;             //!< ガンマ値分子.
    uint16_t    GammaDenominator;           //!< ガンマ値分母
    uint32_t    ColorCorrectionOffset;      //!< 色補正テーブルへのオフセット.
    uint32_t    StampOffset;                //!< ポステージスタンプ画像へのオフセット.
    uint32_t    ScanLineOffset;             //!< スキャンラインオフセット.
    uint8_t     AttributeType;              //!< アルファチャンネルデータのタイプ
};
#pragma pack( pop )


////////////////////////////////////////////////////////////////////////////////////////////////////
// Targa class
////////////////////////////////////////////////////////////////////////////////////////////////////
class Targa
{
    //==============================================================================================
    // list of friend classes and methods.
    //==============================================================================================
    /* NOTHING */

public:
    //==============================================================================================
    // public variables.
    //==============================================================================================
    /* NOTHING */

    //==============================================================================================
    // public methods.
    //==============================================================================================

    //----------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //----------------------------------------------------------------------------------------------
    Targa();

    //----------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //----------------------------------------------------------------------------------------------
    virtual ~Targa();

    //----------------------------------------------------------------------------------------------
    //! @brief      ファイルから読み込みします.
    //!
    //! @param[in]      filename        ファイル名です.
    //! @retval true    読み込み成功.
    //! @retval false   読み込み失敗.
    //----------------------------------------------------------------------------------------------
    bool Load( const char* filename );

    //----------------------------------------------------------------------------------------------
    //! @brief      メモリを解放します.
    //----------------------------------------------------------------------------------------------
    void Release();

    //----------------------------------------------------------------------------------------------
    //! @brief      画像の横幅を取得します.
    //!
    //! @return     画像の横幅を返却します.
    //----------------------------------------------------------------------------------------------
    uint32_t GetWidth() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      画像の縦幅を取得します.
    //!
    //! @return     画像の縦幅を返却します.
    //----------------------------------------------------------------------------------------------
    uint32_t GetHeight() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      画像の形式を取得します.
    //!
    //! @return     画像の形式を返却します.
    //----------------------------------------------------------------------------------------------
    TGA_FORMAT_TYPE GetFormat() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      1ピクセルあたりのビット数を取得します.
    //!
    //! @return     1ピクセルあたりのビット数を返却します.
    //----------------------------------------------------------------------------------------------
    uint8_t GetBitPerPixel() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      1ピクセルあたりのバイト数を取得します.
    //!
    //! @return     1ピクセルあたりのバイト数を取得します.
    //----------------------------------------------------------------------------------------------
    uint8_t GetBytePerPixel() const;

    //----------------------------------------------------------------------------------------------
    //! @brief      ピクセルデータを取得します.
    //!
    //! @return     ピクセルデータを返却します.
    //----------------------------------------------------------------------------------------------
    const uint8_t* GetPixels() const;


protected:
    //==============================================================================================
    // protected variables.
    //==============================================================================================
    uint32_t        m_Width;            //!< 画像の横幅です.
    uint32_t        m_Height;           //!< 画像の縦幅です.
    uint8_t         m_BitPerPixel;      //!< 1ピクセルあたりのビットの深さです.
    TGA_FORMAT_TYPE m_Format;           //!< 画像形式です.
    uint8_t*        m_pPixels;          //!< ピクセルデータです.

    //==============================================================================================
    // protected methods.
    //==============================================================================================
    /* NOTHING */

private:
    //==============================================================================================
    // private variables.
    //==============================================================================================
    /* NOTHING */

    //==============================================================================================
    // private methods.
    //==============================================================================================
    Targa           (const Targa&) = delete;
    void operator = (const Targa&) = delete;
};
