﻿//-------------------------------------------------------------------------------------------------
// File : a3dFrameBuffer.h
// Desc : Frame Buffer Implmentation.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once


namespace a3d {

///////////////////////////////////////////////////////////////////////////////////////////////////
// FrameBuffer class
///////////////////////////////////////////////////////////////////////////////////////////////////
class A3D_API FrameBuffer : public IFrameBuffer, public BaseAllocator
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      生成処理を行います.
    //!
    //! @param[in]      pDevice         デバイスです.
    //! @param[in]      pDesc           構成設定です.
    //! @param[out]     ppFrameBuffer   フレームバッファの格納先です.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //---------------------------------------------------------------------------------------------
    static bool A3D_APIENTRY Create(
        IDevice*                pDevice,
        const FrameBufferDesc*  pDesc, 
        IFrameBuffer**          ppFrameBuffer);

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを増やします.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY AddRef() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      解放処理を行います.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY Release() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを取得します.
    //!
    //! @return     参照カウントを返却します.
    //---------------------------------------------------------------------------------------------
    uint32_t A3D_APIENTRY GetCount() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      デバイスを取得します.
    //!
    //! @param[out]     ppDevice        デバイスの格納先です.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY GetDevice(IDevice** ppDevice) override;

    //---------------------------------------------------------------------------------------------
    //! @brief      構成設定を取得します.
    //!
    //! @return     構成設定を返却します.
    //---------------------------------------------------------------------------------------------
    FrameBufferDesc A3D_APIENTRY GetDesc() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      フレームバッファを設定する描画コマンドを発行します.
    //!
    //! @param[in]      pCommandList        コマンドリストです.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY Bind(ICommandList* pCommandList);

    //---------------------------------------------------------------------------------------------
    //! @brief      フレームバッファをクリアする描画コマンドを発行します.
    //!
    //! @param[in]      pCommandList        コマンドリストです.
    //! @param[in]      clearColorCount     クリアカラー数です.
    //! @param[in]      pClearColors        クリアカラーの配列です.
    //! @param[in]      pClearDepthStencil  クリア深度ステンシルです.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY Clear(
        ICommandList*                   pCommand,
        uint32_t                        clearColorCount,
        const ClearColorValue*          pClearColors,
        const ClearDepthStencilValue*   pClearDepthStencil);

    //---------------------------------------------------------------------------------------------
    //! @brief      カラーターゲット数を取得します.
    //!
    //! @return     カラーターゲット数を返却します.
    //---------------------------------------------------------------------------------------------
    uint32_t A3D_APIENTRY GetColorCount() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      カラーターゲットを持つかどうかチェックします.
    //!
    //! @retval true    カラーターゲットを持ちます.
    //! @retval false   カラーターゲットはありません.
    //---------------------------------------------------------------------------------------------
    bool A3D_APIENTRY HasColorTarget() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      深度ターゲットを持つかどうかチェックします.
    //!
    //! @retval true    深度ターゲットを持ちます.
    //! @retval false   深度ターゲットはありません.
    //---------------------------------------------------------------------------------------------
    bool A3D_APIENTRY HasDepthTarget() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      レンダーターゲットビューのフォーマットの配列を取得します.
    //!
    //! @return     レンダーターゲットビューのフォーマットの配列を返却します.
    //---------------------------------------------------------------------------------------------
    const DXGI_FORMAT* A3D_APIENTRY GetD3D12RenderTargetViewFormats() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      深度ステンシルビューのフォーマットを取得します.
    //!
    //! @return     深度ステンシルビューのフォーマットを返却します.
    //---------------------------------------------------------------------------------------------
    DXGI_FORMAT A3D_APIENTRY GetD3D12DepthStencilViewFormat() const;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    std::atomic<uint32_t>       m_RefCount;             //!< 参照カウンタです.
    Device*                     m_pDevice;              //!< デバイスです.
    FrameBufferDesc             m_Desc;                 //!< 構成設定です.
    DXGI_FORMAT                 m_RTVFormats[8];        //!< レンダーターゲットビューのフォーマットです.
    DXGI_FORMAT                 m_DSVFormat;            //!< 深度ステンシルビューのフォーマットです.
    D3D12_CPU_DESCRIPTOR_HANDLE m_RTVHandles[8];        //!< レンダーターゲットビューハンドルです.
    D3D12_CPU_DESCRIPTOR_HANDLE m_DSVHandle;            //!< 深度ステンシルターゲットビューハンドルです.
    bool                        m_HasDepth;

    //=============================================================================================
    // private methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    A3D_APIENTRY FrameBuffer();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    A3D_APIENTRY ~FrameBuffer();

    //---------------------------------------------------------------------------------------------
    //! @brief      初期化処理を行います.
    //!
    //! @param[in]      pDevice     デバイスです.
    //! @parma[in]      pDesc       構成設定です.
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //---------------------------------------------------------------------------------------------
    bool A3D_APIENTRY Init(IDevice* pDevice, const FrameBufferDesc* pDesc);

    //---------------------------------------------------------------------------------------------
    //! @brief      終了処理を行います.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY Term();

    FrameBuffer     (const FrameBuffer&) = delete;
    void operator = (const FrameBuffer&) = delete;
};

} // namespace a3d
