﻿//-------------------------------------------------------------------------------------------------
// File : a3dCommandBuffer.cpp
// Desc : Command Buffer Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------


namespace a3d {

///////////////////////////////////////////////////////////////////////////////////////////////////
// CommandBuffer class
///////////////////////////////////////////////////////////////////////////////////////////////////

//-------------------------------------------------------------------------------------------------
//      コンストラクタです.
//-------------------------------------------------------------------------------------------------
CommandBuffer::CommandBuffer()
: m_pBuffer (nullptr)
, m_pCmd    (nullptr)
, m_Size    (0)
, m_Enable  (true)
{ /* DO_NOTHING */ }

//-------------------------------------------------------------------------------------------------
//      デストラクタです.
//-------------------------------------------------------------------------------------------------
CommandBuffer::~CommandBuffer()
{ Term(); }

//-------------------------------------------------------------------------------------------------
//      初期化処理を行います.
//-------------------------------------------------------------------------------------------------
bool CommandBuffer::Init(size_t size)
{
    Term();

    m_pBuffer = static_cast<uint8_t*>(a3d_alloc( size, 4 ));
    if (m_pBuffer == nullptr)
    { return false; }

    m_Size  = size;
    m_pCmd   = m_pBuffer;
    m_Enable = true;

    return true;
}

//-------------------------------------------------------------------------------------------------
//      終了処理を行います.
//-------------------------------------------------------------------------------------------------
void CommandBuffer::Term()
{
    if (m_pBuffer != nullptr)
    {
        a3d_free(m_pBuffer);
        m_pBuffer = nullptr;
    }

    m_pCmd   = nullptr;
    m_Size   = 0;
    m_Enable = false;
}

//-------------------------------------------------------------------------------------------------
//      コマンドポインタをコマンドバッファの先頭に戻します.
//-------------------------------------------------------------------------------------------------
void CommandBuffer::Reset()
{
    m_pCmd = m_pBuffer;

    if (m_pBuffer != nullptr)
    { m_Enable = true; }
}

//-------------------------------------------------------------------------------------------------
//      コマンドバッファを閉じます.
//-------------------------------------------------------------------------------------------------
void CommandBuffer::Close()
{ m_Enable = false; }

//-------------------------------------------------------------------------------------------------
//      コマンドを追加します.
//-------------------------------------------------------------------------------------------------
void CommandBuffer::Push(const void* pData, size_t size)
{
    if (!m_Enable)
    { return; }

    auto usedSize = GetCmdSize();
    if (usedSize + size >= m_Size)
    {
        auto resize = static_cast<size_t>(m_Size * 1.5);
        if (resize >= usedSize + size)
        { resize = static_cast<size_t>((usedSize + size) * 1.5); }

        m_pBuffer = static_cast<uint8_t*>(a3d_realloc(m_pBuffer, resize, 4));
        m_Size    = resize;
        m_pCmd    = m_pBuffer;
        m_pCmd    += usedSize;
    }

    memcpy(m_pCmd, pData, size);
    m_pCmd += size;
}

//-------------------------------------------------------------------------------------------------
//      コマンドバッファを追加します.
//-------------------------------------------------------------------------------------------------
void CommandBuffer::Append(const CommandBuffer* pBuffer)
{
    if (!m_Enable)
    { return; }

    auto usedSize = GetCmdSize();
    auto bufSize  = pBuffer->GetCmdSize();
    if (usedSize + bufSize >= m_Size)
    {
        auto resize = static_cast<size_t>(m_Size * 1.5);
        if (resize >= usedSize + bufSize)
        { resize = static_cast<size_t>((usedSize + bufSize) * 1.5); }

        m_pBuffer = static_cast<uint8_t*>(a3d_realloc(m_pBuffer, resize, 4));
        m_Size    = resize;
        m_pCmd    = m_pBuffer;
        m_pCmd    += usedSize;
    }

    memcpy(m_pCmd, pBuffer->GetBuffer(), bufSize);
    m_pCmd += bufSize;
}

//-------------------------------------------------------------------------------------------------
//      コマンドバッファへのポインタを取得します.
//-------------------------------------------------------------------------------------------------
uint8_t* CommandBuffer::GetBuffer() const
{ return m_pBuffer; }

//-------------------------------------------------------------------------------------------------
//      コマンドポインタを取得します.
//-------------------------------------------------------------------------------------------------
uint8_t* CommandBuffer::GetCmdPtr() const
{ return m_pCmd; }

//-------------------------------------------------------------------------------------------------
//      コマンドバッファサイズを取得します.
//-------------------------------------------------------------------------------------------------
size_t CommandBuffer::GetBufferSize() const
{ return m_Size; }

//-------------------------------------------------------------------------------------------------
//      コマンドサイズを取得します.
//-------------------------------------------------------------------------------------------------
size_t CommandBuffer::GetCmdSize() const
{ return static_cast<size_t>(m_pCmd - m_pBuffer); }

} // namespace a3d
