﻿//-------------------------------------------------------------------------------------------------
// File : a3dCommandBuffer.h
// Desc : Command Buffer Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once


namespace a3d {

///////////////////////////////////////////////////////////////////////////////////////////////////
// CommandBuffer class
///////////////////////////////////////////////////////////////////////////////////////////////////
class CommandBuffer : public BaseAllocator
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    CommandBuffer();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    ~CommandBuffer();

    //---------------------------------------------------------------------------------------------
    //! @brief      初期化処理を行います.
    //!
    //! @param[in]      size        バッファの初期化サイズです.
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //---------------------------------------------------------------------------------------------
    bool Init(size_t size);

    //---------------------------------------------------------------------------------------------
    //! @brief      終了処理を行います.
    //---------------------------------------------------------------------------------------------
    void Term();

    //---------------------------------------------------------------------------------------------
    //! @brief      コマンドポイントを先頭に戻します.
    //---------------------------------------------------------------------------------------------
    void Reset();

    //---------------------------------------------------------------------------------------------
    //! @bief       コマンドバッファを閉じます.
    //---------------------------------------------------------------------------------------------
    void Close();

    //---------------------------------------------------------------------------------------------
    //! @brief      コマンドを追加し，コマンドポインタを進めます.
    //!
    //! @param[in]      pData       コマンドデータです.
    //! @param[in]      size        コマンドサイズです.
    //! @note       コマンドバッファの容量が足らない場合はメモリ再確保が実行されます.
    //!             メモリの再確保に失敗した場合はコマンドが追加されません.
    //---------------------------------------------------------------------------------------------
    void Push(const void* pData, size_t size);

    //---------------------------------------------------------------------------------------------
    //! @brief      コマンドバッファを追加します.
    //!
    //! @param[in]      pBuffer     追加するコマンドバッファです.
    //! @note       コマンドバッファの容量が足らない場合はメモリ再確保が実行されます.
    //!             メモリの再確保に失敗した場合はコマンドが追加されません.
    //---------------------------------------------------------------------------------------------
    void Append(const CommandBuffer* pBuffer);

    //---------------------------------------------------------------------------------------------
    //! @brief      コマンドバッファの先頭ポインタを取得します.
    //!
    //! @return     コマンドバッファの先頭ポインタを返却します.
    //---------------------------------------------------------------------------------------------
    uint8_t* GetBuffer() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      コマンドポインタを取得します.
    //!
    //! @return     コマンドポインタを返却します.
    //---------------------------------------------------------------------------------------------
    uint8_t* GetCmdPtr() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      コマンドバッファ全体のサイズを取得します.
    //!
    //! @return     コマンドバッファ全体のサイズを返却します.
    //! @note       コマンド未使用領域のサイズも含まれます.
    //---------------------------------------------------------------------------------------------
    size_t GetBufferSize() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      コマンドサイズを取得します.
    //!
    //! @return     コマンドバッファ先頭からコマンドポインタまでのサイズを返却します.
    //---------------------------------------------------------------------------------------------
    size_t GetCmdSize() const;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    uint8_t*    m_pBuffer;  //!< コマンドバッファです.
    uint8_t*    m_pCmd;     //!< コマンドポインタです.
    size_t      m_Size;     //!< コマンドバッファサイズです.
    bool        m_Enable;   //!< 記録可能かどうか?

    //=============================================================================================
    // private methods.
    //=============================================================================================
    CommandBuffer   (const CommandBuffer&) = delete;
    void operator = (const CommandBuffer&) = delete;
};

} // namespace a3d
