﻿//-------------------------------------------------------------------------------------------------
// File : a3dFrameBuffer.h
// Desc : Frame Buffer Implmentation.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once


namespace a3d {

///////////////////////////////////////////////////////////////////////////////////////////////////
// FrameBuffer class
///////////////////////////////////////////////////////////////////////////////////////////////////
class A3D_API FrameBuffer : public IFrameBuffer, public BaseAllocator
{
    //=============================================================================================
    // list of friend classes and methods.
    //=============================================================================================
    /* NOTHING */

public:
    //=============================================================================================
    // public variables.
    //=============================================================================================
    /* NOTHING */

    //=============================================================================================
    // public methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      生成処理を行います.
    //!
    //! @param[in]      pDevice         デバイスです.
    //! @param[in]      pDesc           構成設定です.
    //! @param[out]     ppFrameBuffer   フレームバッファの格納先です.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //---------------------------------------------------------------------------------------------
    static bool A3D_APIENTRY Create(
        IDevice*                pDevice,
        const FrameBufferDesc*  pDesc, 
        IFrameBuffer**          ppFrameBuffer);

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを増やします.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY AddRef() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      解放処理を行います.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY Release() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを取得します.
    //!
    //! @return     参照カウントを返却します.
    //---------------------------------------------------------------------------------------------
    uint32_t A3D_APIENTRY GetCount() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      デバイスを取得します.
    //!
    //! @param[out]     ppDevice        デバイスの格納先です.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY GetDevice(IDevice** ppDevice) override;

    //---------------------------------------------------------------------------------------------
    //! @brief      構成設定を取得します.
    //!
    //! @return     構成設定を返却します.
    //---------------------------------------------------------------------------------------------
    FrameBufferDesc A3D_APIENTRY GetDesc() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      フレームバッファを設定する描画コマンドを発行します.
    //!
    //! @param[in]      pCommandList        コマンドリストです.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY Bind(ICommandList* pCommandList);

    //---------------------------------------------------------------------------------------------
    //! @brief      フレームバッファをクリアする描画コマンドを発行します.
    //!
    //! @param[in]      pCommandList        コマンドリストです.
    //! @param[in]      count               クリアするターゲット数です.
    //! @param[in]      pClearColor         ターゲットのクリアカラーの配列です.
    //! @param[in]      pClearDepthStencil  ターゲットのクリア深度ステンシルの配列です.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY Clear(
        ICommandList*                   pCommandList,
        uint32_t                        clearColorCount,
        const ClearColorValue*          pClearColors,
        const ClearDepthStencilValue*   pClearDepthStencil);

    //---------------------------------------------------------------------------------------------
    //! @brief      横幅を取得します.
    //!
    //! @return     横幅を返却します.
    //---------------------------------------------------------------------------------------------
    uint32_t A3D_APIENTRY GetWidth() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      縦幅を取得します.
    //!
    //! @return     縦幅を返却します.
    //---------------------------------------------------------------------------------------------
    uint32_t A3D_APIENTRY GetHeight() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      カラーターゲット数を取得します.
    //!
    //! @return     カラーターゲット数を返却します.
    //--------------------------------------------------------------------------------------------
    uint32_t A3D_APIENTRY GetColorTargetCount() const;

    //---------------------------------------------------------------------------------------------
    //! @brief      レンダーパスを取得します.
    //!
    //! @return     レンダーパスを返却します.
    //---------------------------------------------------------------------------------------------
    VkRenderPass A3D_APIENTRY GetRenderPass() const;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    std::atomic<uint32_t>       m_RefCount;         //!< 参照カウンタです.
    Device*                     m_pDevice;          //!< デバイスです.
    FrameBufferDesc             m_Desc;             //!< 構成設定です.
    VkFramebuffer               m_FrameBuffer;      //!< フレームバッファです.
    VkRenderPass                m_RenderPass;       //!< レンダーパスです.
    VkRenderPassBeginInfo       m_BeginInfo;        //!< レンダーパス開始情報です.

    //=============================================================================================
    // private methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    A3D_APIENTRY FrameBuffer();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    A3D_APIENTRY ~FrameBuffer();

    //---------------------------------------------------------------------------------------------
    //! @brief      初期化処理を行います.
    //!
    //! @param[in]      pDevice     デバイスです.
    //! @parma[in]      pDesc       構成設定です.
    //! @retval true    初期化に成功.
    //! @retval false   初期化に失敗.
    //---------------------------------------------------------------------------------------------
    bool A3D_APIENTRY Init(IDevice* pDevice, const FrameBufferDesc* pDesc);

    //---------------------------------------------------------------------------------------------
    //! @brief      終了処理を行います.
    //---------------------------------------------------------------------------------------------
    void A3D_APIENTRY Term();

    FrameBuffer     (const FrameBuffer&) = delete;
    void operator = (const FrameBuffer&) = delete;
};

} // namespace a3d
