﻿//-------------------------------------------------------------------------------------------------
// File : wnd.h
// Desc : Window Wrapper
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <cstdint>


namespace wnd {

///////////////////////////////////////////////////////////////////////////////////////////////////
// MOUSE_BUTTON enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum MOUSE_BUTTON
{
    MOUSE_BUTTON_L = 0x0,       //!< 左ボタンです.
    MOUSE_BUTTON_R,             //!< 右ボタンです.
    MOUSE_BUTTON_M,             //!< 中ボタンです.
    MOUSE_BUTTON_X1,            //!< サイドボタン1です.
    MOUSE_BUTTON_X2,            //!< サイドボタン2です.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// KEYCODE enum
///////////////////////////////////////////////////////////////////////////////////////////////////
enum KEYCODE
{
    KEY_RETURN = 0x80,      //!< リターンキーです.
    KEY_TAB,                //!< Tabキーです.
    KEY_ESC,                //!< Escapeキーです.
    KEY_BACK,               //!< BackSpaceキーです.
    KEY_SHIFT,              //!< Shiftキーです.
    KEY_CONTROL,            //!< Controlキーです.
    KEY_ALT,                //!< Altキーです.
    KEY_F1,                 //!< F1キーです.
    KEY_F2,                 //!< F2キーです.
    KEY_F3,                 //!< F3キーです.
    KEY_F4,                 //!< F4キーです.
    KEY_F5,                 //!< F5キーです.
    KEY_F6,                 //!< F6キーです.
    KEY_F7,                 //!< F7キーです.
    KEY_F8,                 //!< F8キーです.
    KEY_F9,                 //!< F9キーです.
    KEY_F10,                //!< F10キーです.
    KEY_F11,                //!< F11キーです.
    KEY_F12,                //!< F12キーです.
    KEY_UP,                 //!< ↑キーです.
    KEY_DOWN,               //!< ↓キーです.
    KEY_LEFT,               //!< ←キーです.
    KEY_RIGHT,              //!< →キーです.
    KEY_NUM0,               //!< Num0キーです.
    KEY_NUM1,               //!< Num1キーです.
    KEY_NUM2,               //!< Num2キーです.
    KEY_NUM3,               //!< Num3キーです.
    KEY_NUM4,               //!< Num4キーです.
    KEY_NUM5,               //!< Num5キーです.
    KEY_NUM6,               //!< Num6キーです.
    KEY_NUM7,               //!< Num7キーです.
    KEY_NUM8,               //!< Num8キーです.
    KEY_NUM9,               //!< Num9キーです.
    KEY_INSERT,             //!< Insertキーです.
    KEY_DELETE,             //!< Deleteキーです.
    KEY_HOME,               //!< Homeキーです.
    KEY_END,                //!< Endキーです.
    KEY_PAGE_UP,            //!< PageUpキーです.
    KEY_PAGE_DOWN,          //!< PageDownキーです.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// MouseEvent structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct MouseEvent
{
    int     CursorX;        //!< マウスカーソルのX座標です.
    int     CursorY;        //!< マウスカーソルのY座標です.
    int     WheelDelta;     //!< マウスホイールの移動量です.
    bool    IsDown[5];      //!< ボタンが押されているかどうか?
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// KeyEvent structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct KeyEvent
{
    bool        IsKeyDown;      //!< キーが押されているかどうか?
    bool        IsAltDown;      //!< ALTキーがおされているかどうか?
    uint32_t    KeyCode;        //!< キーコードです.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// TypingEvent structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct TypingEvent
{
    uint32_t    KeyCode;        //!< キーコードです.
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// ResizerEvent structure
///////////////////////////////////////////////////////////////////////////////////////////////////
struct ResizeEvent
{
    uint32_t    Width;          //!< ウィンドウの横幅です.
    uint32_t    Height;         //!< ウィンドウの縦幅です.
    float       AspectRatio;    //!< ウィンドウのアスペクト比です.
};

//-------------------------------------------------------------------------------------------------
// Type definitions.
//-------------------------------------------------------------------------------------------------
typedef void (*MouseHandler)   (const MouseEvent&  args, void* pUser);
typedef void (*KeyboardHandler)(const KeyEvent&    args, void* pUser);
typedef void (*TypingHandler)  (const TypingEvent& args, void* pUser);
typedef void (*ResizeHandler)  (const ResizeEvent& args, void* pUser);


///////////////////////////////////////////////////////////////////////////////////////////////////
// IApp interface
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IApp
{
    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    virtual ~IApp()
    { /* DO_NOTHING */ }

    //---------------------------------------------------------------------------------------------
    //! @brief      解放処理を行います.
    //---------------------------------------------------------------------------------------------
    virtual void Release() = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを増やします.
    //---------------------------------------------------------------------------------------------
    virtual void AddRef() = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを取得します.
    //!
    //! @return     参照カウントを返却します.
    //---------------------------------------------------------------------------------------------
    virtual uint32_t GetCount() const = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      メインループを続行するかどうか.
    //!
    //! @retval true    メインループを続行します.
    //! @retval false   メインループを終了します.
    //---------------------------------------------------------------------------------------------
    virtual bool IsLoop() = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      終了要求を通知します.
    //---------------------------------------------------------------------------------------------
    virtual void RequestQuit() = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウの横幅を取得します.
    //!
    //! @return     ウィンドウの横幅を返却します.
    //---------------------------------------------------------------------------------------------
    virtual uint32_t GetWidth() const = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウの縦幅を取得します.
    //!
    //! @return     ウィンドウの縦幅を返却します.
    //---------------------------------------------------------------------------------------------
    virtual uint32_t GetHeight() const = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウのアスペクト比を取得します.
    //!
    //! @return     ウィンドウのアスペクト比を返却します.
    //---------------------------------------------------------------------------------------------
    virtual float GetAspectRatio() const = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      インスタンスハンドルを取得します.
    //!
    //! @return     インスタンスハンドルを返却します.
    //---------------------------------------------------------------------------------------------
    virtual void* GetInstanceHandle() const = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウハンドルを取得します.
    //!
    //! @return     ウィンドウハンドルを返却します.
    //---------------------------------------------------------------------------------------------
    virtual void* GetWindowHandle() const = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      マウスハンドラーを設定します.
    //!
    //! @param[in]      handler     ハンドラーです.
    //! @param[in]      pUser       ユーザーポインタです.
    //---------------------------------------------------------------------------------------------
    virtual void SetMouseHandler(MouseHandler handler, void* pUser) = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      キーボードハンドラーを設定します.
    //!
    //! @param[in]      handler     ハンドラーです.
    //! @param[in]      pUser       ユーザーポインタです.
    //---------------------------------------------------------------------------------------------
    virtual void SetKeyboardHandler(KeyboardHandler handler, void* pUser) = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      タイピングハンドラーを設定します.
    //!
    //! @param[in]      handler     ハンドラーです.
    //! @param[in]      pUser       ユーザーポインタです.
    //---------------------------------------------------------------------------------------------
    virtual void SetTypingHandler(TypingHandler handler, void* pUser) = 0;

    //---------------------------------------------------------------------------------------------
    //! @brief      リサイズハンドラーを設定します.
    //!
    //! @param[in]      handler     ハンドラーです.
    //! @param[in]      pUser       ユーザーポインタです.
    //---------------------------------------------------------------------------------------------
    virtual void SetResizeHandler(ResizeHandler handler, void* pUser) = 0;
};

//-------------------------------------------------------------------------------------------------
//! @brief      アプリケーションを生成します.
//!
//! @param[in]      title       タイトル名です.
//! @param[in]      width       ウィンドウの横幅です.
//! @param[in]      height      ウィンドウの縦幅です.
//! @param[out]     ppApp       アプリケーションの格納先です.
//! @retval true    生成に成功.
//! @retval false   生成に失敗.
//-------------------------------------------------------------------------------------------------
bool CreateApp(const char* title, uint32_t width, uint32_t height, IApp** ppApp);

} // namespace wnd
