﻿//-------------------------------------------------------------------------------------------------
// File : App.h
// Desc : Application Module.
// Copyright(c) Project Asura. All right reserved.
//-------------------------------------------------------------------------------------------------
#pragma once

//-------------------------------------------------------------------------------------------------
// Includes
//-------------------------------------------------------------------------------------------------
#include <wnd.h>
#include <atomic>
#include <Windows.h>


namespace wnd {

///////////////////////////////////////////////////////////////////////////////////////////////////
// App class
///////////////////////////////////////////////////////////////////////////////////////////////////
class App : IApp
{
public:
    //---------------------------------------------------------------------------------------------
    //! @brief      生成処理を行います.
    //!
    //! @param[in]      title       タイトル名です.
    //! @param[in]      width       ウィンドウの横幅です.
    //! @param[in]      height      ウィンドウの縦幅です.
    //! @param[out]     ppApp       アプリケーションの格納先です.
    //! @retval true    生成に成功.
    //! @retval false   生成に失敗.
    //---------------------------------------------------------------------------------------------
    static bool Create(const char* title, uint32_t width, uint32_t height, IApp** ppApp);

    //---------------------------------------------------------------------------------------------
    //! @brief      解放処理を行います.
    //---------------------------------------------------------------------------------------------
    void Release() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを増やします.
    //---------------------------------------------------------------------------------------------
    void AddRef() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      参照カウントを取得します.
    //!
    //! @return     参照カウントを返却します.
    //---------------------------------------------------------------------------------------------
    uint32_t GetCount() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      メインループを続行するかどうか.
    //!
    //! @retval true    メインループを続行します.
    //! @retval false   メインループを終了します.
    //---------------------------------------------------------------------------------------------
    bool IsLoop() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      終了要求を通知します.
    //---------------------------------------------------------------------------------------------
    void RequestQuit() override;

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウの横幅を取得します.
    //!
    //! @return     ウィンドウの横幅を返却します.
    //---------------------------------------------------------------------------------------------
    uint32_t GetWidth() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウの縦幅を取得します.
    //!
    //! @return     ウィンドウの縦幅を返却します.
    //---------------------------------------------------------------------------------------------
    uint32_t GetHeight() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウのアスペクト比を取得します.
    //!
    //! @return     ウィンドウのアスペクト比を返却します.
    //---------------------------------------------------------------------------------------------
    float GetAspectRatio() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      インスタンスハンドルを取得します.
    //!
    //! @return     インスタンスハンドルを返却します.
    //---------------------------------------------------------------------------------------------
    void* GetInstanceHandle() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウハンドルを取得します.
    //!
    //! @return     ウィンドウハンドルを返却します.
    //---------------------------------------------------------------------------------------------
    void* GetWindowHandle() const override;

    //---------------------------------------------------------------------------------------------
    //! @brief      マウスハンドラーを設定します.
    //!
    //! @param[in]      handler     ハンドラーです.
    //! @param[in]      pUser       ユーザーポインタです.
    //---------------------------------------------------------------------------------------------
    void SetMouseHandler(MouseHandler handler, void* pUser) override;

    //---------------------------------------------------------------------------------------------
    //! @brief      キーボードハンドラーを設定します.
    //!
    //! @param[in]      handler     ハンドラーです.
    //! @param[in]      pUser       ユーザーポインタです.
    //---------------------------------------------------------------------------------------------
    void SetKeyboardHandler(KeyboardHandler handler, void* pUser) override;

    //---------------------------------------------------------------------------------------------
    //! @brief      タイピングハンドラーを設定します.
    //!
    //! @param[in]      handler     ハンドラーです.
    //! @param[in]      pUser       ユーザーポインタです.
    //---------------------------------------------------------------------------------------------
    void SetTypingHandler(TypingHandler handler, void* pUser) override;

    //---------------------------------------------------------------------------------------------
    //! @brief      リサイズハンドラーを設定します.
    //!
    //! @param[in]      handler     ハンドラーです.
    //! @param[in]      pUser       ユーザーポインタです.
    //---------------------------------------------------------------------------------------------
    void SetResizeHandler(ResizeHandler handler, void* pUser) override;

private:
    //=============================================================================================
    // private variables.
    //=============================================================================================
    std::atomic<uint32_t>   m_RefCount;
    HINSTANCE               m_hInstance;
    HWND                    m_hWnd;
    MSG                     m_Msg;
    uint32_t                m_Width;
    uint32_t                m_Height;
    float                   m_AspectRatio;
    char                    m_Title[256];
    MouseHandler            m_MouseHandler;
    KeyboardHandler         m_KeyboardHandler;
    TypingHandler           m_TypingHandler;
    ResizeHandler           m_ResizeHandler;
    void*                   m_pUserMouse;
    void*                   m_pUserKeyboard;
    void*                   m_pUserTyping;
    void*                   m_pUserResize;

    //=============================================================================================
    // private methods.
    //=============================================================================================

    //---------------------------------------------------------------------------------------------
    //! @brief      コンストラクタです.
    //---------------------------------------------------------------------------------------------
    App();

    //---------------------------------------------------------------------------------------------
    //! @brief      デストラクタです.
    //---------------------------------------------------------------------------------------------
    ~App();

    //---------------------------------------------------------------------------------------------
    //! @brief      初期化を行います.
    //---------------------------------------------------------------------------------------------
    bool Init(const char* title, uint32_t width, uint32_t height);

    //---------------------------------------------------------------------------------------------
    //! @brief      終了処理を行います.
    //---------------------------------------------------------------------------------------------
    void Term();

    //---------------------------------------------------------------------------------------------
    //! @brief      ウィンドウプロシージャです.
    //---------------------------------------------------------------------------------------------
    static LRESULT CALLBACK WndProc(HWND hWnd, UINT msg, WPARAM wp, LPARAM lp);

    App             (const App&) = delete;
    void operator = (const App&) = delete;
};

} // namespace wnd
