/* -------------------------------------------------------------------------
 * diskd --- monitors shared disk.
 *   This applied pingd mechanism to disk monitor.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Copyright (c) 2008 NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 *
 * -------------------------------------------------------------------------
 */

#include <lha_internal.h>

#include <sys/param.h>

#include <crm/crm.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <unistd.h>

#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <libgen.h>

#include <heartbeat.h>
#include <hb_api.h>
#include <clplumbing/Gmain_timeout.h>
#include <clplumbing/lsb_exitcodes.h>

#include <crm/common/ipc.h>
#include <attrd.h>

#ifdef HAVE_GETOPT_H
#  include <getopt.h>
#endif

#define MIN_INTERVAL 1
#define MAX_INTERVAL 3600
#define MIN_TIMEOUT 1
#define MAX_TIMEOUT 3600
#define MIN_RETRY 	0
#define MAX_RETRY	10
#define MIN_RETRY_INTERVAL	1
#define MAX_RETRY_INTERVAL	3600	
#define ERROR -1
#define normal 1
#define BLKFLSBUF  _IO(0x12,97) /* flush buffer. refer linux/hs.h */

/* GMainLoop *mainloop = NULL; */
const char *crm_system_name = "diskd";

#define OPTARGS	"V?p:a:N:Di:r:I:t:"

IPC_Channel *attrd = NULL;
GMainLoop*  mainloop = NULL;
const char *diskd_attr = "diskd";

const char *device = NULL; /* device name for disk check */

int retry = 1;			/* disk check retry. default 1 times */
int retry_interval = 5; /* disk check retry intarval time. default 5sec. */
int interval = 30;  	/* disk check interval. default 30sec.*/
int timeout = 30; 		/* disk check read func timeout. default 30sec. */
int old_status = 0;
const char *diskcheck_value = NULL;
int pagesize = 0;
void *ptr = NULL;
void *buf;

void send_update(void);

static gboolean
diskd_shutdown(int nsig, gpointer unused)
{
	crm_info("Exiting");
	
	if (mainloop != NULL && g_main_is_running(mainloop)) {
		g_main_quit(mainloop);
	} else {
		exit(0);
	}
	return FALSE;
}

static void
usage(const char *cmd, int exit_status)
{
	FILE *stream;

	stream = exit_status ? stderr : stdout;

	fprintf(stream, "usage: %s [-%s]\n", cmd, OPTARGS);
	fprintf(stream, "    Basic options\n");
	fprintf(stream, "\t--%s (-%c) \t\t\tThis text\n", "help", '?');
	fprintf(stream, "\t--%s (-%c) \t\t\tRun in verbose mode\n", "verbose", 'V');
	fprintf(stream, "\t--%s (-%c) \t\tRun in daemon mode\n", "daemonize", 'D');
	fprintf(stream, "\t--%s (-%c) <filename>\tFile in which to store the process' PID\n"
		"\t\t\t\t\t* Default=/tmp/diskd.pid\n", "pid-file", 'p');
	fprintf(stream, "\t--%s (-%c) <string>\tName of the node attribute to set\n"
		"\t\t\t\t\t* Default=diskd\n", "attr-name", 'a');
	fprintf(stream, "\t--%s (-%c) <devicename>\tName of the device to check status\n"
		"\t\t\t\t\t* Mandatory option\n", "device-name", 'N');
	fprintf(stream, "\t--%s (-%c) <time[s]>\tDisk status check interval time\n"
		"\t\t\t\t\t* Default=30 sec.\n", "interval", 'i');
	fprintf(stream, "    Advanced options\n");
	fprintf(stream, "\t--%s (-%c) <time[s]>\tRead timeout for select function\n"
		"\t\t\t\t\t* Default=5 sec.\n", "read-timeout", 't');
	fprintf(stream, "\t--%s (-%c) <times>\t\tDisk status check retry\n"
		"\t\t\t\t\t* Default=1 times\n", "retry", 'r');
	fprintf(stream, "\t--%s (-%c) <time[s]>\tDisk status check retry interval time\n"
		"\t\t\t\t\t* Default=5 sec.\n", "retry-interval", 'I');

	fflush(stream);

	exit(exit_status);
}

static gboolean
check_old_status(int new_status) 
{

	if (new_status != ERROR && new_status != normal) {
		crm_warn("non-defined status, new_status = %d", new_status);
		return FALSE;
	}
	
	if (old_status != new_status) {
		if (new_status == ERROR) {
			diskcheck_value = "ERROR";
		} else {
			diskcheck_value = "normal";
		}
		crm_warn("disk status is changed, new_status = %s", diskcheck_value);
		send_update();
		old_status = new_status;
	}
	return TRUE;
}


static int diskcheck(gpointer data) 
{
	int i;
	int fd = -1;
	int err;
	int select_err;
	struct timeval timeout_tv;
	fd_set read_fd_set;

	crm_debug_2("diskcheck start");

	for (i = 0; i <= retry; i++) {
		if ( i !=0 ) {
			sleep(retry_interval);
		}


		fd = open((const char *)device, O_RDONLY | O_NONBLOCK, 0);
		if (fd == -1) {
			crm_err("Could not open device %s", device);
			continue;
		} 

		err = ioctl(fd, BLKFLSBUF, 0);
		if (err != 0) {
			crm_err("iotcl error, Could not flush baffer");
			close(fd);
			continue;
		}

		while( 1 ) {
			err = read(fd, buf, pagesize);
			if (err == pagesize) {
				crm_debug_2("reading form data is OK");
				close(fd);
				check_old_status(normal);
				return normal;
			} else if (err != pagesize && errno == EAGAIN) {
				crm_warn("read function return errno:EAGAIN");
				FD_ZERO(&read_fd_set);
				FD_SET(fd, &read_fd_set);
				timeout_tv.tv_sec = timeout;
				select_err = select(1, &read_fd_set, NULL, NULL, &timeout_tv);
				if (select_err == 1) {
					crm_warn("select ok, read again");
					continue;
				} else if (select_err == -1) {
					crm_err("select failed on device %s", device);
					close(fd);
					break;
				}
			} else {
				crm_err("Could not read from device %s", device);
				close(fd);
				break;
			}
		}
	} 

	crm_warn("Error(s) occurred in diskcheck function.");
	check_old_status(ERROR);
	return ERROR;
}


int
main(int argc, char **argv)
{
	int lpc;
	int argerr = 0;
	int flag;
	char *pid_file = NULL;
	gboolean daemonize = FALSE;
	
#ifdef HAVE_GETOPT_H
	int option_index = 0;
	static struct option long_options[] = {
		/* Top-level Options */
		{"verbose", 0, 0, 'V'},
		{"help", 0, 0, '?'},
		{"pid-file",  1, 0, 'p'},		
		{"attr-name", 1, 0, 'a'},		
		{"device-name",  1, 0, 'N'},		
		{"daemonize", 0, 0, 'D'},		
		{"interval",  1, 0, 'i'},		
		{"retry", 1, 0, 'r'},		
		{"retry-interval", 1, 0, 'I'},		
		{"read-timeout", 1, 0, 't'},		

		{0, 0, 0, 0}
	};
#endif
	pid_file = crm_strdup("/tmp/diskd.pid");
	crm_system_name = basename(argv[0]);

	G_main_add_SignalHandler(
		G_PRIORITY_HIGH, SIGTERM, diskd_shutdown, NULL, NULL);
	
	crm_log_init(basename(argv[0]), LOG_INFO, TRUE, FALSE, argc, argv);

	/* check user. user shuld be root.*/
	if (strcmp("root", (const gchar *)g_get_user_name()) != 0) {
		crm_err("permission denied. diskd should be executed by root.\n");
		printf ("permission denied. diskd should be executed by root.\n");
		exit(LSB_EXIT_GENERIC);
	}
	
	while (1) {
#ifdef HAVE_GETOPT_H
		flag = getopt_long(argc, argv, OPTARGS,
				   long_options, &option_index);
#else
		flag = getopt(argc, argv, OPTARGS);
#endif
		if (flag == -1)
			break;

		switch(flag) {
			case 'V':
				cl_log_enable_stderr(TRUE);
				alter_debug(DEBUG_INC);
				break;
			case 'p':
				pid_file = crm_strdup(optarg);
				break;
			case 'a':  
				diskd_attr = crm_strdup(optarg);
				break;
			case 'r': 
				retry = crm_parse_int(optarg, "1");
				if ((retry == 0) && (strcmp(optarg, "0") != 0)) {
					argerr++;
					break;
				}
				if ((retry < MIN_RETRY) || (retry > MAX_RETRY)) 
					++argerr;
				break;
			case 'I':
				retry_interval = crm_parse_int(optarg, "1");
				if ((retry_interval < MIN_RETRY_INTERVAL) || (retry_interval > MAX_RETRY_INTERVAL)) 
					++argerr;
				break;
			case 'i': 
				interval = crm_parse_int(optarg, "1");
				if ((interval < MIN_INTERVAL) || (interval > MAX_INTERVAL)) 
					++argerr;
				break;
			case 't': 
				timeout = crm_parse_int(optarg, "1");
				if ((timeout < MIN_TIMEOUT) || (timeout > MAX_TIMEOUT)) 
					++argerr;
			case 'N':   
				device = crm_strdup(optarg);
				break;
			case 'D':
				daemonize = TRUE;
				break;
			case '?':
				usage(crm_system_name, LSB_EXIT_GENERIC);
				break;
			default:
				printf ("Argument code 0%o (%c) is not (?yet?) supported\n", flag, flag);
				crm_err("Argument code 0%o (%c) is not (?yet?) supported\n", flag, flag);
				++argerr;
				break;
		}
	}

	if (optind < argc) {
		crm_err("non-option ARGV-elements: ");
		printf ("non-option ARGV-elements: ");
		while (optind < argc) {
			crm_err("%s ", argv[optind++]);
			printf("%s ", argv[optind++]);
		}
		printf("\n");
		argerr ++;
	}
	if ((argerr) || (device == NULL)){
		usage(crm_system_name, LSB_EXIT_GENERIC);
	}

	crm_make_daemon(crm_system_name, daemonize, pid_file);

	for(lpc = 0; attrd == NULL && lpc < 30; lpc++) {
		crm_debug("attrd registration attempt: %d", lpc);
		sleep(5);
		attrd = init_client_ipc_comms_nodispatch(T_ATTRD);
	}
	
	if(attrd == NULL) {
		printf ("attrd registration failed\n");
		crm_err("attrd registration failed");
		cl_flush_logs();
		exit(LSB_EXIT_GENERIC);
	}  
	
	pagesize = getpagesize();
	ptr = (void *)malloc(2 * pagesize);
	buf = (void *)(((u_long)ptr + pagesize) & ~(pagesize-1));
	if (ptr == NULL) {
		crm_err("Could not allocate memory");
		check_old_status(ERROR);
		exit(LSB_EXIT_GENERIC);
	}
	
	diskcheck(NULL);
	
	Gmain_timeout_add(interval*1000, diskcheck, NULL);

	crm_info("Starting %s", crm_system_name);
	mainloop = g_main_new(FALSE);
	g_main_run(mainloop);
	
	free(ptr);
	crm_info("Exiting %s", crm_system_name);	
	return 0;
}

void
send_update()
{
	HA_Message *update = ha_msg_new(4);
	ha_msg_add(update, F_TYPE, T_ATTRD);
	ha_msg_add(update, F_ORIG, crm_system_name);
	ha_msg_add(update, F_ATTRD_TASK, "update");
	ha_msg_add(update, F_ATTRD_ATTRIBUTE, diskd_attr);

	ha_msg_add(update, F_ATTRD_VALUE, diskcheck_value);

	if(send_ipc_message(attrd, update) == FALSE) {
		crm_err("Could not send update");
		exit(1);
	}
	crm_msg_del(update);
}
